/**
 * RemoteIR library - Test program.
 *
 * Copyright (C) 2010 Shinichiro Nakamura (CuBeatSystems)
 * http://shinta.main.jp/
 */

#include <mbed.h>

#include "ReceiverIR.h"
#include "TransmitterIR.h"
#include "TextLCD.h"

#define TEST_LOOP_BACK  0

ReceiverIR ir_rx(p15);
TransmitterIR ir_tx(p21);
BusOut led(LED4, LED3, LED2, LED1);
Ticker ledTicker;

/**
 * Receive.
 *
 * @param format Pointer to a format.
 * @param buf Pointer to a buffer.
 * @param bufsiz Size of the buffer.
 *
 * @return Bit length of the received data.
 */
int receive(RemoteIR::Format *format, uint8_t *buf, int bufsiz, int timeout = 100) {
    int cnt = 0;
    while (ir_rx.getState() != ReceiverIR::Received) {
        cnt++;
        if (timeout < cnt) {
            return -1;
        }
    }
    return ir_rx.getData(format, buf, bufsiz * 8);
}

/**
 * Display a current status.
 */
void display_status(char *status, int bitlength) {
    printf("%-5.5s:%02d\r\n", status, bitlength);
}

/**
 * Display a format of a data.
 */
void display_format(RemoteIR::Format format) {
    switch (format) {
        case RemoteIR::UNKNOWN:
            printf("????????\r\n");
            break;
        case RemoteIR::NEC:
            printf("NEC     \r\n");
            break;
        case RemoteIR::NEC_REPEAT:
            printf("NEC  (R)\r\n");
            break;
        case RemoteIR::AEHA:
            printf("AEHA    \r\n");
            break;
        case RemoteIR::AEHA_REPEAT:
            printf("AEHA (R)\r\n");
            break;
        case RemoteIR::SONY:
            printf("SONY    \r\n");
            break;
    }
}

/**
 * Display a data.
 *
 * @param buf Pointer to a buffer.
 * @param bitlength Bit length of a data.
 */
void display_data(uint8_t *buf, int bitlength) {
    const int n = bitlength / 8 + (((bitlength % 8) != 0) ? 1 : 0);
    for (int i = 0; i < n; i++) {
        printf("%02X", buf[i]);
    }
    for (int i = 0; i < 8 - n; i++) {
        printf("--");
    }
    printf("\r\n");
}

void ledfunc(void) {
    led = led + 1;
}

/**
 * Entry point.
 */
int main(void) {

    ledTicker.attach(&ledfunc, 0.5);

    /*
     * Splash.
     */
    printf("RemoteIR        \r\n");
    printf("Program example.\r\n");

    /*
     * Initialize.
     */
    /*
     * Execute.
     */
    while (1) {
        uint8_t buf1[32];
        uint8_t buf2[32];
        int bitlength1;
        int bitlength2;
        RemoteIR::Format format;

        memset(buf1, 0x00, sizeof(buf1));
        memset(buf2, 0x00, sizeof(buf2));

        {
            bitlength1 = receive(&format, buf1, sizeof(buf1));
            if (bitlength1 < 0) {
                continue;
            }
            display_status("RECV", bitlength1);
            display_data(buf1, bitlength1);
            display_format(format);
        }
    }
}