/*-----------------------------------------------------------
 * F746_SLIDER Library v1.0
 * Copyright (c) 2018 Wynand Steenberg
 * s216875730@mandela.ac.za
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *-----------------------------------------------------------
 */

#ifndef F746_SLIDER_HPP
#define F746_SLIDER_HPP

#include "mbed.h"
#include <string>
#include "TS_DISCO_F746NG.h"
#include "LCD_DISCO_F746NG.h"
#include "F746GUI.hpp"

/** A Class library for using Sliders on the DISCO-F746NG Development board.  The class
 * uses the existing BSP class created by Team ST.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "F746_SLIDER.hpp"
 * 
 * TS_DISCO_F746NG ts_;
 * LCD_DISCO_F746NG lcd_;
 * 
 * 
 * char char_SliderDisplayValue[4];    // String Value to display for the Slider Control value 
 * uint16_t SliderDisplayValue_;       // Variable used to access Slider Control Value in F746SLIDER.cpp
 * 
 * int main()
 * {
 *     lcd_.Clear(LCD_COLOR_WHITE);                                        // Set LCD Background colour
 *    
 *     Slider sld1(lcd_, ts_, 20, 205, 150, 20, 5, 1, 20,     
 *                  LCD_COLOR_BLUE, LCD_COLOR_YELLOW, 1, "Slider1", Font12);   // Define sld1 slider            
 *     sld1.Render();                                                          // Draw sld1 Slider  
 *    
 *     Slider Slider2(lcd_, ts_, 240, 205, 220, 20, 50, 1, 100,        
 *                  LCD_COLOR_BLUE, LCD_COLOR_YELLOW, 2, "Slider2", Font12);   // Define Slider2 slider            
 *     Slider2.Render();                                                       // Draw Slider2 Slider      
 *        
 *     lcd_.SetTextColor(LCD_COLOR_BLACK);                                                         // Set Text colour to Black
 *     lcd_.SetFont(&Font12);                                                                      // Font size 12
 *     lcd_.DisplayStringAt(5, 5, (uint8_t *)"Slider driver for DISCO_F746", LEFT_MODE);  // Display main header text
 *     lcd_.DisplayStringAt(5, 135, (uint8_t *)"Slider example", LEFT_MODE);  // Display secondary header text
 * 
 *     while (true)                                                    // Main program loop
 *     {
 *         if (sld1.Moved())                                           // Check if sld1 Slider was touched and run instructions if true
 *         {    
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);                     
 *             lcd_.FillRect(130, 140, 80, 40);                         // Draw border to display Slider Control Value
 *             lcd_.SetTextColor(LCD_COLOR_WHITE);
 *             lcd_.FillRect(140, 145, 60, 30);                         // Draw border to display Slider Control Value                                     
 *             
 *             sprintf(char_SliderDisplayValue, "%3d", (int) SliderDisplayValue_);                     // Convert integer to text
 *             lcd_.SetFont(&Font12);                                                                  // Set Font size
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);                                                     // Set Font colour
 *             lcd_.DisplayStringAt(150, 155, (uint8_t *)char_SliderDisplayValue, LEFT_MODE);          // Write Slider Control Value to the LCD
 *             
 *             if (SliderDisplayValue_ < 10)                                                           // Slider Control Value decision
 *                 {
 *                     lcd_.DisplayStringAt(60, 190, (uint8_t *)"                     ", LEFT_MODE);   // Overwrite previous text
 *                     lcd_.DisplayStringAt(60, 190, (uint8_t *)"SLIDER1 < 50%", LEFT_MODE);           // Write text to LCD
 *                 }
 *             else if (SliderDisplayValue_ == 10)                                                     // Slider Control Value decision
 *                 {  
 *                     lcd_.DisplayStringAt(60, 190, (uint8_t *)"                     ", LEFT_MODE);   // Overwrite previous text
 *                     lcd_.DisplayStringAt(60, 190, (uint8_t *)"SLIDER1 = 50%", LEFT_MODE);           // Write text to LCD
 *                 }
 *             else                                                                                    // Slider Control Value decision
 *                 {  
 *                     lcd_.DisplayStringAt(60, 190, (uint8_t *)"                     ", LEFT_MODE);   // Overwrite previous text
 *                     lcd_.DisplayStringAt(60, 190, (uint8_t *)"SLIDER1 > 50%", LEFT_MODE);           // Write text to LCD
 *                 }          
 *         }                                                                                           // End sld1 instructions
 * 
 *         if (Slider2.Moved())                                      // Check if Slider2 Slider was touched and run instructions if true
 *         {    
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);                     
 *             lcd_.FillRect(390, 140, 80, 40);                         // Draw border to display Slider Control Value
 *             lcd_.SetTextColor(LCD_COLOR_WHITE);
 *             lcd_.FillRect(400, 145, 60, 30);                         // Draw border to display Slider Control Value                                     
 *             
 *             sprintf(char_SliderDisplayValue, "%3d", (int) SliderDisplayValue_);                     // Convert integer to text
 *             lcd_.SetFont(&Font12);                                                                  // Set Font size
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);                                                     // Set Font colour
 *             lcd_.DisplayStringAt(415, 155, (uint8_t *)char_SliderDisplayValue, LEFT_MODE);          // Write Slider Control Value to the LCD
 *         }    
 *     
 *         wait(0.02f);   
 *     }                                                               // End Main program loop
 * }                                                                   // End Main program 
 * @endcode
 */

    class Slider
    {
    public:
        //! Constructor
        Slider(LCD_DISCO_F746NG &lcd, TS_DISCO_F746NG &ts,
               uint16_t x, uint16_t y, uint16_t width, uint16_t height,
               uint16_t value_C, uint16_t min, uint16_t max,
               uint32_t sliderColour, uint32_t changeColour,
               int style, const string label = "", sFONT &fonts = Font12,
               uint32_t textColour = LCD_COLOR_WHITE)
               : lcd_(lcd), ts_(ts), SliderX_(x), SliderY_(y), SliderW_(width), SliderH_(height),
                 valueControl_(value_C), Min_ (min), Max_ (max), 
                 SLIDER_COLOUR_(sliderColour), CHANGE_COLOUR_(changeColour),
                 STYLE_(style), LABEL_(label), FONTS_(&fonts), FONT_WIDTH_(fonts.Width),
                 FONT_HEIGHT_(fonts.Height), active_(true)
                 { Render(); }

        /**
          * @brief  Draw Slider.
          *         
          */
        void Render();
        
        /**
          * @brief  Service Slider.
          *         
          */
        void Service();
               
        /**
          * @brief  Check touch detected on Slider.
          *         
          */
        bool Moved();
   
        /**
          * @brief  Check if touch is on Slider.
          *         
          */
        bool SliderBoundaryCheck();
        
        /**
          * @brief  Check previous state of Slider.
          *         
          */
        static TS_StateTypeDef GottenState()
        {   return state_; }

        /**
          * @brief  Set or reset multi-touch.
          *         
          */
        static void SetMultiTouch(bool tf) { multiTouch = tf; }

                
    private:
        LCD_DISCO_F746NG &lcd_;
        TS_DISCO_F746NG &ts_;

        const uint16_t SliderX_, SliderY_, SliderW_, SliderH_, valueControl_, Min_, Max_;
        const uint32_t SLIDER_COLOUR_;  // original color
        const uint32_t CHANGE_COLOUR_;  // colour to change to color
        const int STYLE_;               // Slider style        
        const string LABEL_;            // Slider label
        sFONT *const FONTS_;
        const uint16_t FONT_WIDTH_;
        const uint16_t FONT_HEIGHT_;
        bool active_;                   // Slider active boolean
        
        static TS_StateTypeDef state_;
        static bool multiTouch;

        /**
          * @brief  Disallow copy constructor and assignment operator.
          *         
          */
        Slider(const Slider&);
        Slider& operator=(const Slider&);
    };
#endif  // F746_SLIDER_HPP
