/*
        ___       ___       ___   
       /\  \     /\  \     /\  \
      /::\  \   /::\  \   /::\  \
     /:/\:\__\ /::\:\__\ /::\:\__\
     \:\ \/__/ \/\::/  / \/\::/  /
      \:\__\     /:/  /     \/__/ 
       \/__/     \/__/
                         
    2016 Flow design labs, Project LEAF
    
    #define CAP12xx_MAIN    0x00   //Controls power states and indicates an interrupt
    #define CAP12xx_MINT    0x1    //Clears interrupts on touch
    #define CAP12xx_SEN     0x1F   //Controls the sensitivity of the threshold and delta counts and datascaling of the base counts 
    #define CAP12xx_MTC     0x2A   //Determines the number of simultaneous touches to flag a multiple touch condition
    #define CAP12xx_SIS     0x03   //Returns the state of the sampled capacitive touch sensor inputs
    #define CAP12xx_PID     0xFD   //Stores a fixed value that identifies the CAP12xx
    #define CAP12xx_MID     0xFE   //Stores a fixed value that identifies Microchip
    #define CAP12xx_REV     0xFF   //Stores a fixed value that represents the revision number
    #define CAP12xx_ACT     0x24   //Controls averaging and sampling window for Active
    #define CAP12xx_CAL     0x26   //Forces calibration for capacitive touch sensor inputs and indicates calibration failure
    #define CAP12xx_ENA     0x27   //Determines which capacitive sensor inputs can generate interrupts
    #define CAP12xx_IRP     0x28   //Turn off interrupt repeat on button hold
    #define CAP12xx_GRD     0x29   //Enables the signal guard for specific sensor inputs
    #define CAP12xx_STB     0x41   //Controls averaging and sensing cycle time for Standby
    #define CAP12xx_IRR     0x44   //Set interrupt on press but not release - default 0x41
    
    //////////////////////////////////////////////////////
    //  CAP12xx_SEN - Sensitivity values - 1x minimum   //
    //  0x0F    //128x sensitivity                      //
    //  0x1F    //64x  sensitivity                      //
    //  0x2F    //32x  sensitivity                      //
    //  0x3F    //16x  sensitivity                      //
    //  0x4F    //8x   sensitivity                      //
    //  0x5F    //4x   sensitivity                      //    
    //  0x6F    //2x   sensitivity                      //
    //  0x7F    //1x   sensitivity                      //
    //////////////////////////////////////////////////////
*/

#include "CAP12xx.h"

CAP12xx::CAP12xx(PinName SDA, PinName SCL, PinName interrupt_pin) : communication(SDA, SCL), interrupt(interrupt_pin) 
{
    core_address = 0x28<<1;
    return;
}
    
bool CAP12xx::init()
{
    communication.frequency(400000);
    interrupt.mode(PullUp);
    
    readRegister(CAP12xx_PID);
     
    //printf("Product ID: 0x%02X\n\r", readRegister(CAP12xx_PID));
    //printf("Manufacturer ID: 0x%02X\n\r", readRegister(CAP12xx_MID));
    //printf("Revision ID: 0x%02X\n\r", readRegister(CAP12xx_REV));
    
    if ((readRegister(CAP12xx_PID) != 0x71) || (readRegister(CAP12xx_MID) != 0x5D) || (readRegister(CAP12xx_REV) != 0x01)) {
        return false; 
    }

    writeRegister(CAP12xx_SEN, 0x4F);   //Sensitivity settings
    writeRegister(CAP12xx_MTC, 0x80);   //Determines the number of simultaneous touches to flag a multiple touch condition 0x80 for OFF and 0x00 for ON
    writeRegister(CAP12xx_ENA, 0xFF);   //Determines which capacitive sensor inputs can generate interrupts
    writeRegister(CAP12xx_STB, 0x39);   //Controls averaging and sensing cycle time for Standby 0x41
    writeRegister(CAP12xx_ACT, 0x39);   //Controls averaging and sampling window for Active 0x39
    writeRegister(CAP12xx_IRP, 0x00);   //Turn off interrupt repeat on button hold
    
    return true;
}

void CAP12xx::setSensitivity(char level)
{
    writeRegister(CAP12xx_SEN, level);   //Sensitivity settings
    return;
}

void CAP12xx::enable(void)
{
    probe = 0;
    probe_touched = 0;
    interrupt.fall(this, &CAP12xx::handler);
    
    uint8_t clear_interrupt = readRegister(CAP12xx_MAIN);   //read main register
    clear_interrupt &= ~CAP12xx_MINT;                       //clear interrupt bit
    writeRegister(CAP12xx_MAIN, clear_interrupt);           //write main register with no interrupt bit
    return;
}

void CAP12xx::disable(void)
{
    interrupt.fall(NULL);
    probe = 0;
    probe_touched = 0;
    return;
}

uint8_t CAP12xx::touched(void)
{
    return probe_touched;
}

uint8_t CAP12xx::identifyProbeTouched(void)
{   
    uint8_t clear_interrupt = readRegister(CAP12xx_MAIN);   //read main register
    clear_interrupt &= ~CAP12xx_MINT;                       //clear interrupt bit
    writeRegister(CAP12xx_MAIN, clear_interrupt);           //write main register with no interrupt bit
    
    uint8_t probe = readRegister(CAP12xx_SIS);
    probe_touched = 0;
    
    return probe;
}

uint8_t CAP12xx::inspect(void)
{
    char touch_data = readRegister(CAP12xx_SIS);
    
    uint8_t clear_interrupt = readRegister(CAP12xx_MAIN);   //read main register
    clear_interrupt &= ~CAP12xx_MINT;                       //clear interrupt bit
    writeRegister(CAP12xx_MAIN, clear_interrupt);           //write main register with no interrupt bit
    
    return touch_data;
}

void CAP12xx::handler(void)
{    
    probe_touched = 1;
    return;
}

void CAP12xx::writeRegister(char address, char data)
{
    int write_status;
    char command[2] = {address, data};
    write_status = communication.write(core_address, command, 2, false);

    if(write_status != 0){
        //printf("ERROR - writing to CAP12xx failed..");
    } 
    return;    
}

uint8_t CAP12xx::readRegister(char address) 
{    
    int write_status, read_status;
    char data;
    
    write_status = communication.write(core_address, &address, 1, false);
    read_status  = communication.read(core_address, &data, 1);

    if(write_status != 0){
        //printf("ERROR - writing to CAP12xx failed..");
    } 
    if(read_status != 0){  
        //printf("ERROR - reading from CAP12xx failed..");
    }
    return data;
}