//*************************************************************************************************
//
//  スケールスピードメータプログラム
//
//  Projyect name   : ScaleSpeed
//  Date            : 2015.04.08
//  Target          : FRDM-KL25Z
//
//  仕様
//  Start swich     : 計測開始ボタン
//  Break switch    : 計測中止ボタン
//  Distance switch : 計測距離設定ボタン
//  Scale switch    : 縮尺切替ボタン
//
//  Start           : LED on green
//  Senser1         : LED on Yelow
//  Senser2         : LED on Blue              
//
//*************************************************************************************************
#include "mbed.h"
#include "TextLCD.h"
#define ON  0
#define OFF 1
Serial pc(USBTX, USBRX);                // USB COMポート設定
TextLCD lcd(A0, A1, A2, A3, A4, A5);    // TextLCD設定
Timer timer;                            // タイマー計測初期化
InterruptIn int1(D12);                  // 割込みピン設定（センサー1）
InterruptIn int2(D11);                  // 割込みピン設定（センサー2）

/* 三色LEDポート設定 */
DigitalOut greenLed(LED_GREEN); // 緑
DigitalOut redLed(LED_RED);     // 赤
DigitalOut blueLed(LED_BLUE);   // 青


/* スイッチポート設定 */
DigitalIn startSW(D2);
DigitalIn breakSW(D3);
DigitalIn distanceSW(D4);
DigitalIn scaleSW(D5);
DigitalIn sence1(D12);
DigitalIn sence2(D11);

/* グローバル定義 */
bool countFlag;     // 計測中フラッグ
bool endFlag;       // 計測完了フラッグ
bool changeFlag;    // 変化フラッグ

uint8_t charCounter;    // 割込み文字カウンタ

//*******************************************************
//  Serial受信割り込みベクタ関数（キーボード割込み）
//*******************************************************
void pcRx(){
        char c;
        c = pc.getc();
        if (c == 0xa0) charCounter = 0;
        else{
            lcd.locate(charCounter,1);
            lcd.putc(c);
            charCounter++;
            if (charCounter > 10) charCounter = 0;
        }
}

//*******************************************************
// センサー1割込み関数
//*******************************************************
void senser1(){
    if (countFlag){
        timer.start();
        redLed = 0;
    }
}

//*******************************************************
// センサー2割込み関数
//*******************************************************
void senser2(){
    if (countFlag){
        timer.stop();
        redLed = 1;
        countFlag = 0;
        endFlag = 1;
        changeFlag = 1;
    }
}

//*******************************************************
// Distance設定関数
//*******************************************************
uint32_t distanceConfig(int counter){
    
    /* Print distance */
    lcd.locate(2,0);
    switch(counter){
    case 0:
        lcd.printf("10mm");
        return 10000;
    case 1:
        lcd.printf("50mm");
        return 50000;
    case 2:
        lcd.printf("10cm");
        return 100000;
    case 3:
        lcd.printf("50cm");
        return 500000;
    case 4:
        lcd.printf("1.0m");
        return 1000000;
    case 5:
        lcd.printf("5.0m");
        return 5000000;
    default:
        return 0;
    }
}

//*******************************************************
// LCD表示関数
//*******************************************************
uint16_t scaleConfig(int counter){
        
    /* Print scale */
    lcd.locate(11,0);
    switch(counter){
    case 0:
        lcd.printf("1/150");
        return 150;
    case 1:
        lcd.printf("1/120");
        return 120;
    case 2:
        lcd.printf("1/100");
        return 100;
    case 3:
        lcd.printf("1/87 ");
        return 87;
    case 4:
        lcd.printf("1/76 ");
        return 76;
    case 5:
        lcd.printf("1/48 ");
        return 48;
    case 6:
        lcd.printf("1/35 ");
        return 35;
    case 7:
        lcd.printf("1/30 ");
        return 30;
    case 8:
        lcd.printf("1/24 ");
        return 24;
    case 9:
        lcd.printf("1/20 ");
        return 20;
    case 10:
        lcd.printf("1/16 ");
        return 16;
    case 11:
        lcd.printf("1/12 ");
        return 12;
    case 12:
        lcd.printf("1/10 ");
        return 10;
    case 13:
        lcd.printf("1/8  ");
        return 8;
    case 14:
        lcd.printf("1/6  ");
        return 6;
    case 15:
        lcd.printf("1/4  ");
        return 4;
    case 16:
        lcd.printf("1/2  ");
        return 2;
    case 17:
        lcd.printf("1/1  ");
        return 1;
    default:
        return 0;    
    }
}

//*******************************************************
// メイン関数
//*******************************************************
int main() {
    bool startFlag = 0;
    bool breakFlag = 0;
    bool distanceFlag = 0;
    bool scaleFlag = 0;
        
    uint8_t scaleCounter = 0;       // 縮尺設定番号
    uint8_t distanceCounter = 0;    // 計測距離設定番号
    
    uint32_t distance = 0;  // 距離(mm)
    uint16_t scale = 0;     // 縮尺比
    uint32_t speedSec;      // 秒速値
    uint32_t speedScale;    // 縮尺変換秒速値
    uint16_t speed;         // 速度データ(km/h)
    
    redLed = 1;
    greenLed = 1;
    blueLed = 1;
    
    /* Print start */
    wait(0.5);   
    lcd.printf("   FRDM-KL25Z");
    wait(0.5);   
    pc.printf("Start!\r\n");
    charCounter = 0;
    wait(1);
    
    /* Print LCD fixed charactor */
    lcd.cls();
    lcd.locate(0,0);
    lcd.printf("D=");    
    lcd.locate(7,0);
    lcd.printf("SCL=");
    lcd.locate(12,1);
    lcd.printf("km/h");
    lcd.locate(0,1);
    lcd.printf("1:  2:");
    changeFlag = 1;
    
    /* Enable interrupt */
    int1.fall(&senser1);
    int2.fall(&senser2);
    
//*******************************************************
// メインループ
//*******************************************************
    while(1) {
        
        /* Recieve serial  */
        pc.attach(pcRx,Serial::RxIrq);
        
        /* Push start switch */
        if(!startSW && !startFlag){
            wait_ms(10);            // チャタリング防止
            if (!startSW){            
                changeFlag = 1;
                startFlag = 1;
                endFlag = 0;
                breakFlag = 0;
                countFlag = 1;
                timer.reset();
            }
        }
        else if (startFlag) startFlag = 0;
        
        /* Push break switch */
        if(!breakSW && !breakFlag){
            wait_ms(10);
            if (!breakSW){
                changeFlag = 1;
                breakFlag = 1;
                countFlag = 0;
                timer.stop();
                speed = 0;
                redLed = 1;
            }
        }
        else if (breakSW) breakFlag = 0;
        
        /* Push scale switch */
        if(!scaleSW && !scaleFlag){
            wait_ms(10);
            if (!scaleSW){
                changeFlag = 1;
                scaleFlag = 1;
                scaleCounter++;
                if (scaleCounter > 17) scaleCounter = 0;
            }
        }
        else if (scaleSW) scaleFlag = 0;
        
        /* Push distance switch */
        if(!distanceSW && !distanceFlag){
            wait_ms(10);
            if (!distanceSW){
                changeFlag = 1;
                distanceFlag = 1;
                distanceCounter++;
                if (distanceCounter > 5) distanceCounter = 0;
            }
        }
        else if (distanceSW) distanceFlag = 0;
        
        /* sence1 */
        lcd.locate(2,1);
        if (sence1) lcd.printf("H");
        else lcd.printf("L"); 
        
        /* sence2 */
        lcd.locate(6,1);
        if (sence2) lcd.printf("H");
        else lcd.printf("L");
        
        /* Set LED */
        if (countFlag) greenLed =0;
        else greenLed = 1;
        
        if (endFlag) blueLed = 0;
        else blueLed = 1;
        
        /* 速度計算と表示 */
        if (changeFlag){
            changeFlag = 0;
            
            /* 距離設定 */
            distance = distanceConfig(distanceCounter);
            
            /* 縮尺設定 */
            scale = scaleConfig(scaleCounter);
            
            /* 速度計算 */
            speedSec = distance / timer.read_ms();  // 速度(mm/sec) = 距離 / 時間
            speedScale = speedSec * scale;          // 縮尺速度 = 秒速 × 縮尺
            speed = (speedScale * 36) / 10000;  // 時速km/h = 縮尺速度 × 3600sec(1h) / 1000000
    
            /* 速度表示 */
            lcd.locate(8,1);
            if (speed < 10000) lcd.printf("%4d",speed);
            else lcd.printf("----");
            pc.printf("%d",speed); // For debug
        }
    }
}
/* End of main */