/****************************************************************************
*
*  httpserver.c - General HTTP Server implementation
*  Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*****************************************************************************/

#include "httpserver.h"
#include "doTCPIP.h"
#include "TSISensor.h"

extern axis6_t axis6;
extern TSISensor tsi;

/** \brief Socket used by server to listen and accept connections */
long httpServerSocket;

/** \brief Variable holding the server's port */
int serverPort = 0;

/** \brief Pointer to the index HTML page */
char * indexPage;
extern unsigned char *dataPacket;

/** \brief Pointer to CGI handler structure */
cgi_handler * chList;

/** \brief Pointer to Dynamic HTML handler structure */
dyn_html_handler * htmlList;

/** \brief Pointer to Dynamic HTML handler structure */
http_server_event_handler * eventHandlers = NULL;


/** \brief Client socket handle list */
int clientList[MAX_CLIENTS];

/** \brief Page view counter */
int viewCounter = 1;
#define REQ_BUFFER_SIZE     400
#define HTTP_TX_BLOCK_SIZE  256
//#define HTTP_TX_BLOCK_SIZE  1024
//#define HTTP_TX_BLOCK_SIZE  512


extern char requestBuffer[REQ_BUFFER_SIZE];


volatile int Delay;

/********************************************************************/
void getAccelXYZ_Str(char * str) // MMA8451Q accelerometer - report axis with highest value 
{
    sprintf(str,"                                    "); //clears field (needed if previous string had more characters)   
    sprintf(str, "X= %1.2f, Y= %1.2f, Z= %1.2f", axis6.fGax, axis6.fGay, axis6.fGaz);;  
}  

/********************************************************************/
void getTemperatureStr(char * str) // 
{
    sprintf(str, "%+d C", axis6.temp); 
}

/********************************************************************/
void getTSI_sliderStr(char * str) // TSI Slider position 
{
    uint8_t slider_position; 
    
    slider_position = tsi.readPercentage() * 100; // Slider position as percentage
    sprintf(str,"    "); //clears field (needed if previous string had more characters)
    sprintf(str, "%d %%", slider_position);
}   

void getCompassStr(char * str)   // Mag3110 generated Compass bearing 
{
    char *compass_points[9] = {"North", "N-East", "East", "S-East", "South", "S-West", "West", "N-West", "North"};
    signed short compass_bearing = (axis6.compass + 23) / 45;
    sprintf(str,"                                        "); //clears field (needed if previous string had more characters)
    sprintf(str, "Roll=%-d  Pitch=%-d  Yaw=%-d [%s]", axis6.roll, axis6.pitch, axis6.yaw, compass_points[compass_bearing]);   // 
}   

void getM3110Str(char * str)   // Mag3110 displayed in units of UT 
{
    sprintf(str,"                                    "); //clears field (needed if previous string had more characters)
    sprintf(str, "X= %3.1f, Y= %3.1f, Z= %3.1f", axis6.fUTmx, axis6.fUTmy, axis6.fUTmz);
}   

/********************************************************************/
extern void getAltitudeStr(char * str)    // Get Altitude 
{
    sprintf(str, "%+d meters", axis6.alt);   // str = integer portion of result 
}   

//*****************************************************************************
//
//! \brief  Initializes HTTP Server
//!
//! \param cnum is the client socket handle to be used
//!
//! \return 0 if successful
//!
//
//*****************************************************************************
char initHTTPServer(int port,
                    char * ipage,
                    cgi_handler * handleList,
                    dyn_html_handler * dhList)
{
    sockaddr serverSocketAddr;
    serverPort = port;
    indexPage = ipage;
    chList = handleList;
    htmlList = dhList;


    httpServerSocket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (httpServerSocket == -1)
    {
        printf("oops\n");
        wlan_stop();
        return((char)-1);
    }

    serverSocketAddr.sa_family = AF_INET;

    // Set the Port Number
    serverSocketAddr.sa_data[0] = (port & 0xFF00)>> 8;
    serverSocketAddr.sa_data[1] = (port & 0x00FF);

    memset (&serverSocketAddr.sa_data[2], 0, 4);

    if (bind(httpServerSocket, &serverSocketAddr, sizeof(sockaddr)) != 0);

    return 0;
}

//*****************************************************************************
//
//! \brief  Main HTTP Server
//!
//! \param none
//!
//! \return none
//!
//
//*****************************************************************************
void serverMain()
{
    sockaddr clientaddr;
    socklen_t addrlen;
    int i = 0;
    int currentClient = 0;
    LED_D2_OFF;
    printf("Main HTTP server\n");
    for(i = 0; i < MAX_CLIENTS; i++)
        clientList[i] = -1;

    // Start Listening
    if (listen (httpServerSocket, MAX_CLIENTS) != 0);

    // Handle Clients and Data
    while(1)
    {
        addrlen = sizeof(clientaddr);
        printf("Current Client= %d\n", currentClient);
        // accept blocks until we receive a connection
        while ( (clientList[currentClient] == -1) || (clientList[currentClient] == -2) )
        {
            clientList[currentClient] = accept(httpServerSocket, (sockaddr *) &clientaddr, &addrlen);
        }
     
        if(clientList[currentClient] >= 0)
        {
            LED_D2_ON;
            // Connection Accepted, Send Data
            // Wait for a data update
            newData = 0;
            while(!newData);
//            SysTick->CTRL &= ~SysTick_CTRL_TICKINT_Msk;  // *** Disable SysTick Timer
            handleHTTPRequest(currentClient);
            closesocket(clientList[currentClient]);
            clientList[currentClient]=-1;
//            SysTick->CTRL |=  SysTick_CTRL_TICKINT_Msk;  // *** Re-Enable SysTick Timer
            LED_D2_OFF;
        }
        else if(clientList[currentClient] == -57)
        {
            // BUG: Socket inactive so reopen socket
            // Inactive Socket, close and reopen it
            printf("Oops!!!\n");
            closesocket(httpServerSocket);
            httpServerSocket = 0xFFFFFFFF;
            initHTTPServer(serverPort, indexPage, chList, htmlList);
            // Start Listening
            if ( listen (httpServerSocket, 5) != 0 );
        }
    }
}

//*****************************************************************************
//
//! \brief  Handles HTTP Requests
//!
//! \param cnum is the client socket handle to be used
//!
//! \return none
//!
//
//*****************************************************************************
void handleHTTPRequest(int cnum)
{

    char * reqline[3];
    char * cgiTok;

    int i = 0;
    char paramBuf[20];
    int bytesRecvd;
    char tempStr[40]; //PF was 26

    memset(requestBuffer,0,sizeof (requestBuffer));
    bytesRecvd = recv(clientList[cnum], requestBuffer, sizeof(requestBuffer), 0);

    printf("\nhandleHTTPRequest\n");

    if(bytesRecvd > 0)
    {
        // Received some data, check it and send data back
        reqline[0] = strstr(requestBuffer, "GET");
        if ( reqline[0] != NULL )
        {
            if (strstr (requestBuffer, "HTTP/1.0") != NULL && strstr (requestBuffer, "HTTP/1.1") != NULL )
            {
                send(clientList[cnum], "HTTP/1.0 400 Bad Request\n", 25,0);
            }
            else
            {

#ifdef HTTP_CGI_ENABLED
                // Do we have CGI parameters we need to parse?
                if(strchr(requestBuffer, '?') != NULL)
                {
                    // Decode URL and handle each parameter sequentially
                    // according to table previously setup.
                    cgiTok = strstr(requestBuffer,"=");
                    if(cgiTok != NULL)
                    {
                        memset(paramBuf,0,sizeof(paramBuf));
                        memcpy(paramBuf,cgiTok+1,5);     // hard-coded for demo: 5 character parameter (-Red-/Green/Blue-)
                        chList->cgiHandlerFunc[0](paramBuf);

                    }
                }
#endif

#ifdef HTTP_DYN_HTML_ENABLED
                // The code below replaces data in the HTML page
                // with that generated by the specified functions.
                for(i = 0; i < 9; i++)  // change the range here for more dynamic fields on webpage
                {
                    memset(tempStr,0,sizeof(tempStr));
                    htmlList->dynHtmlFunc[i](tempStr);
                    tempStr[strlen(tempStr)]= ' ';
                    pageReplace((char *)indexPage,
                                (char *)htmlList->dynHtmlParamName[i],
                                (char *)tempStr);
                }
#endif
                viewCounter++;

                sendHTTPData(clientList[cnum], HTTP_RESP, strlen(HTTP_RESP));
                                
                for(i = 0; i < strlen(indexPage); i += HTTP_TX_BLOCK_SIZE)
                {
                                    
                    if(strlen(indexPage) - i < HTTP_TX_BLOCK_SIZE)
                    {
                        sendHTTPData(clientList[cnum], &indexPage[i], strlen(indexPage) - i);
                    }
                    else
                    {
                        sendHTTPData(clientList[cnum], &indexPage[i], HTTP_TX_BLOCK_SIZE);
                    }
                }
            }
        }
    }
}

//*****************************************************************************
//
//! \brief  Inserts characters in page that appear after an indicator ind
//! with the value from val
//!
//! \param  page is a pointer to the array holding the page's HTML code
//! \param ind is a pointer to a string that has the name of the parameter on the page to modify
//! \param val is the pointer to a string holding the string to insert in the XXX
//!
//! \return none
//!
//
//*****************************************************************************
void pageReplace(char * page, char * ind, char * val)
{
    char * indicLoc;
    indicLoc = strstr (page,ind);
    memcpy(indicLoc+strlen(ind), val, strlen(val));
}

//*****************************************************************************
//
//! \brief  Returns a string with the number of views of the page
//!
//! \param  str is a pointer to the array where the number of views will be put
//!
//! \return none
//!
//
//*****************************************************************************
extern uint32 cal_count;


void getViewsNum(char * str)
{
    sprintf(str, "%d", viewCounter);
}

//*****************************************************************************
//
//! \brief  Sends HTTP Data
//!
//! \param sdesc is the socket descriptor of the socket used for sending data
//! \param buf is a pointer to the buffer with the data to be sent
//! \param len is the number of bytes to send
//!
//! \return none
//!
//
//*****************************************************************************
void sendHTTPData(long sdesc, const void *buf, long len)
{
    int bytesSent = -2;
//    SysTick->CTRL &= ~SysTick_CTRL_TICKINT_Msk;  // *** Disable SysTick Timer
    while(bytesSent == -2) bytesSent = send(sdesc, buf, len,0);
//    SysTick->CTRL |=  SysTick_CTRL_TICKINT_Msk;  // *** Re-Enable SysTick Timer
    if(bytesSent == -1)
    {
        // General Send Error
//        printf("General Send Error.\n");
    }
    if (bytesSent != strlen(indexPage))
    {
        // ERROR: not all bytes sent
//        printf("Not all bytes sent.\n");
//        printf("Length : %d - Sent : %d\n", strlen(indexPage), bytesSent);
    }
}

//*****************************************************************************
//
//! \brief  Initializes HTTP Server Event Handler
//!
//! \param eh is a pointer to the array handling server events
//!
//! \return none
//!
//
//*****************************************************************************
void initEventHandlers(http_server_event_handler * eh)
{
    eventHandlers = eh;
}

