/**************************************************************************************************
 *****                                                                                        *****
 *****  Name: PCA9635_6.h                                                                     *****
 *****  Date: 09/03/2014                                                                      *****
 *****  Auth: Frank Vannieuwkerke                                                             *****
 *****  Func: library for I2C RGB LED Demo board (6 x PCA9635)                                *****
 *****        Ported from http://mbed.org/users/JimCarver/code/RGB2/                          *****
 **************************************************************************************************/

#ifndef PCA9635_6_H
#define PCA9635_6_H

#include "mbed.h"

typedef struct
{
    uint8_t   address;             /* slave address to sent/receive message     */
    uint8_t   *buf;                /* pointer to application message buffer     */
} I2C_MESSAGE;

/** Class for I2C RGB LED Demo board (6 x PCA9635).
*
* 'en' pin can be any digital pin.
* Uses I2C bus.
*/ 

class PCA9635_6 {
public:
    /** Create a PCA9635_6 object connected to enable pin and I2C bus.
    *
    * @param sda = SDA pin.
    * @param scl = SCL pin.
    * @param en  = Board enable pin.
    * @return none
    */
    PCA9635_6(PinName sda, PinName scl, PinName en);

    /** Send RGB color to a LED.
    *
    * @param LEDn = LED number.
    * @param R = Red.
    * @param G = Green.
    * @param B = Blue.
    * @return none.
    *
    * Lookup the buffer and index to set the desired values
    * in the appropriate locations within the I2C message buffers
    * update_LED() actually sends the buffers through I2C to the drivers
    */
    void set_LED(int LEDn, uint8_t R, uint8_t G, uint8_t B);

    /** Initalization of the buffers.
    *
    * @param none.
    * @return none.
    */
    void Init_Buffers(void);

    /** Fill all the PWM register of a device with a RGB setting.
    *
    * @param A,B,C --> Primary colors.
    * @return none.
    */
    void Set_Same_Color_Single_Device(uint8_t A, uint8_t B, uint8_t C);

    /** Set the I2C address to the ALL CALL Address and send the I2C_Write command.
    *
    * @param none.
    * @return none.
    */
    void Write_All_Devices(void);

    /** Set the I2C address to the required address and send the I2C_Write command.
    *
    * @param I2C Address of the targetted device (LSB = 0).
    * @return none.
    */
    void Write_To_Device(uint8_t I2C_Address);

    /** Set the I2C address to the required address and send the I2C_Write command.
    * Wait (delay) before moving to the next step.
    *
    * @param I2C Address of the targetted device (LSB = 0).
    * @param Delay = delay in ms.
    * @return none.
    */
    void Write_To_Device_And_Wait(uint8_t I2C_Address, uint8_t Delay_Value);

    /** Swtich of ALL the LEDs using the ALL CALL address.
    *
    * @param none.
    * @return none.
    */
    void ALL_LED_OFF(void);
    
    /** Write new values to PCA9635.
    *
    * @param none.
    * @return none.
    */
    void update_LED(void);

    /** Read current RGB LED values.
    *
    * @param LEDn = LED number.
    * @param *R = Pointer to Red return value.
    * @param *G = Pointer to Green return value.
    * @param *B = Pointer to Blue return value.
    */
    void read_LED( int LEDn, uint8_t *R, uint8_t *G, uint8_t *B);

    /** Send GLOBAL_INTENSITY to LED board.
    *
    * @param none.
    * @return none.
    */
    void LED_INTENSITY(void);

    /** Change GLOBAL_INTENSITY.
    *
    * @param intensity value.
    * @return none.
    */
    void set_global_intensity(uint8_t globi);

private:
    void i2cfail(void);
    void I2C_Write(I2C_MESSAGE *mp);
    I2C _i2c;
    DigitalOut _en;
};

#endif
