/**************************************************************************************************************
 *****                                                                                                    *****
 *****  Name: hsi2rgbw.h                                                                                  *****
 *****  Date: 22/12/2013                                                                                  *****
 *****  Auth: Frank Vannieuwkerke                                                                         *****
 *****  Func: library for converting HSI color space values to RGBW                                       *****
 *****                                                                                                    *****
 *****  Code ported from http://saikoled.com - Copyright 2013, Brian Neltner                              *****
 *****  http://blog.saikoled.com/post/44677718712/how-to-convert-from-hsi-to-rgb-white                    *****
 *****  http://blog.saikoled.com/post/43693602826/why-every-led-light-should-be-using-hsi-colorspace      *****
 *****  https://github.com/saikoLED/MyKi/blob/master/myki_16_bit_random_fade/myki_16_bit_random_fade.ino  *****
 *****  https://github.com/saikoLED/MyKi/blob/master/myki_16_bit_fade/myki_16_bit_fade.ino                *****
 *****                                                                                                    *****
 *****  This program is free software: you can redistribute it and/or modify                              *****
 *****  it under the terms of the GNU General Public License as published by                              *****
 *****  the Free Software Foundation, version 3 of the License.                                           *****
 *****                                                                                                    *****
 *****  This program is distributed in the hope that it will be useful,                                   *****
 *****  but WITHOUT ANY WARRANTY; without even the implied warranty of                                    *****
 *****  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                                     *****
 *****  GNU General Public License for more details.                                                      *****
 *****                                                                                                    *****
 *****  A copy of the GNU General Public License can be found at                                          *****
 *****  http://www.gnu.org/licenses/gpl.html                                                              *****
 **************************************************************************************************************/

#include "mbed.h"

#ifndef HSI2RGBW_PWM_H
#define HSI2RGBW_PWM_H

#define RGBW 1
#define RGB  0

/** Class to convert HSI colorspace value to its corresponding RGBW value.
* 'red', 'green', 'blue' and 'white' pin can be any PWM pin.
*/ 

class hsi2rgbw_pwm {
public:
    /** Create a HSI2RGBW object connected to 4 PWM pins.
    *
    * @param red   = RED PWM pin\n
    * @param blue  = BLUE PWM pin\n
    * @param green = GREEN PWM pin\n
    * @param white = WHITE PWM pin\n
    * @return none\n
    * When you want to use hsi2rgbw standalone (no PWMout), use following initializer:\n
    *   hsi2rgbw_pwm led(NC);\n
    *   Create a float array in your program to return the rgbw values, eg : float rgbw[4]\n
    *   Call hsi2rgbw with the rgbw pointer : hsi2rgbw(H, S, I, rgbw)\n
    * When PWM ouptuts are used, RGB or RGBW mode is automatically set.\n
    * When no PWM outputs are used, the default output is RGBW.\n
    */
    hsi2rgbw_pwm(PinName pred = NC, PinName pgreen = NC, PinName pblue = NC, PinName pwhite = NC);

    /** Convert HSI to RGBW or RGB.
    *
    * @param Hue         = float (0..360)\n
    * @param Saturation  = float (0..1)\n
    * @param Intensity   = float (0..1)\n
    * @param rgbw        Optional pointer to float array[4]\n
    *                    This parameter can be omitted when PWM output is enabled:\n
    *                    hsi2rgbw(H, S, I)\n
    *
    * This function can be used to convert HSI to RGBW or RGB:\n
    *   call colorMode(RGBW) to enable HSI->RGBW\n
    *   call colorMode(RGB)  to enable HSI->RGB\n
    *   IMPORTANT NOTE : When RGB is selected and you want the function to return\n 
    *                    the converted value, do declare a 4 element array!!\n
    *                    The value in the 4th element can be discarded.\n
    */
    void hsi2rgbw(float H, float S, float I, float* rgbw = NULL);
    
    /** Change PWM period.
    *
    * @param Period (in us)\n
    * @return none\n
    */
    void period(uint32_t per);
    
    /** Change PWM value.
    *
    * @param pointer to float array[4]\n
    * @param array[0] = RED PWM value (0...1)\n
    * @param array[1] = BLUE PWM value (0...1)\n
    * @param array[2] = GREEN PWM value (0...1)\n
    * @param array[3] = WHITE PWM value (0...1)\n
    *
    * Use this function to set the PWM values without conversion.\n
    * @return none\n
    */
    void pwm(float* rgbw);

    /** Set parabolic mode.
    *
    * @param Parabolic   1 = map rgbw output value to a parabolic curve (default).\n
    *                    0 = linear mapping.\n
    * @return none\n
    */
    void parabolic(bool para);
    
    /** Set color conversion mode.
    *
    * @param Colormode   RGBW (1) = convert HSI to RGBW.\n
    *                    RGB  (0) = convert HSI to RGB.\n
    * @return none\n
    */
    void colorMode(bool como);

    /** Invert PWM output.
    *
    * @param InvertPWM   0 : Normal PWM output.\n
    *                    1 : PWM output is inverted.\n
    * @return none\n
    *
    * Enable (1) when common anode LEDs are used.\n
    */
    void invertpwm(bool invpwm);

private:
    bool parabol;
    bool use_pwm;
    bool use_rgbw;
    bool use_invpwm;
}; 
#endif
