/* GetTypeName.h */
/* Copyright (C) 2015 mbed.org, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */
#ifndef TYPE_NAME_H_
#define TYPE_NAME_H_

#define DECLARE_TYPE_NAME(x) template<> const char *type_name<x>::name = #x;
#define GetTypeName(x) (type_name<typeof(x)>::name)

namespace mbed {
/** <H1>Get a variable type using type traits - a replacement for RTTI typeid().</H1>
* 
* Sources:<BR>
*
* https://developer.mbed.org/forum/bugs-suggestions/topic/4494/?page=1#comment-22386<BR>
* http://stackoverflow.com/questions/81870/print-variable-type-in-c<BR>
* <BR>
* In RTTI, we were able to use something like.
* @code
* // Depending on the compiler used, this returns 'int', 'i', ....
*     int main() {
*         int a= 4;
*         printf("%s", typeid(a).name());
*     }
* @endcode
* To reduce the code overhead, this can no longer be used in the mbed compiler, however, we can replace this with type traits.<BR>
* <BR>
* Example:<BR>
* @code
* #include "mbed.h"
* #include "GetTypeName.h"
*
* int main()
* {
*     char a = 65;
*     printf("Type name for <char> '%c' is %s\r\n", a, GetTypeName(a));
*     // Check whether GetTypeName is of 'char' type.
*     // Note that strcmp returns 0 when both strings are equal.
*     if(!strcmp(GetTypeName(a),"char")) printf("'%c' is of 'char' type\r\n", a);
* }
* @endcode
* <BR>
* The DECLARE_TYPE_NAME define exists to make your life easier in declaring this traits class for all the types you expect to need.<BR>
* This might be more useful than the solutions involving typeid because you get to control the output.<BR>
* <BR>
* Supported declarations
* @verbatim
* | Variable           | Returns    |
* |--------------------|------------|
* | char               | char       |
* | uint8_t            | uint8_t    |
* | signed char        | int8_t     |
* | int8_t             | int8_t     |
* | unsigned short     | uint16_t   |
* | uint16_t           | uint16_t   |
* | short              | int16_t    |
* | int16_t            | int16_t    |
* | unsigned int       | uint32_t   |
* | uint32_t           | uint32_t   |
* | int                | int32_t    |
* | int32_t            | int32_t    |
* | unsigned long long | uint64_t   |
* | uint64_t           | uint64_t   |
* | long long          | int64_t    |
* | int64_t            | int64_t    |
* | float              | float      |
* | double             | double     |
* | bool               | bool       |
* @endverbatim
*/

    template <typename T> class type_name
    {
    public:
        static const char *name;
    };

    DECLARE_TYPE_NAME(char);
    DECLARE_TYPE_NAME(uint8_t);
    DECLARE_TYPE_NAME(int8_t);      // also valid for 'signed char'
    DECLARE_TYPE_NAME(uint16_t);    // also valid for 'unsigned short'
    DECLARE_TYPE_NAME(int16_t);     // also valid for 'short'
    DECLARE_TYPE_NAME(uint32_t);    // also valid for 'unsigned int'
    DECLARE_TYPE_NAME(int32_t);     // also valid for 'int'
    DECLARE_TYPE_NAME(uint64_t);    // also valid for 'unsigned long long'
    DECLARE_TYPE_NAME(int64_t);     // also valid for 'long long
    DECLARE_TYPE_NAME(float);
    DECLARE_TYPE_NAME(double);
    DECLARE_TYPE_NAME(bool);

} // namespace mbed

#endif //TYPE_NAME_H_