/*****************************************************************************
*
*  wlan  - CC3000 Host Driver Implementation.
*  Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the   
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*****************************************************************************/

#include "wlan.h"

extern void WriteWlanPin( unsigned char val );
volatile sSimplLinkInformation tSLInformation;

#ifndef CC3000_UNENCRYPTED_SMART_CONFIG
unsigned char key[AES128_KEY_SIZE];    
unsigned char profileArray[SMART_CONFIG_PROFILE_SIZE];
#endif //CC3000_UNENCRYPTED_SMART_CONFIG


static void SimpleLink_Init_Start(unsigned short usPatchesAvailableAtHost)
{
    unsigned char *ptr;
    unsigned char *args;
    
    ptr = tSLInformation.pucTxCommandBuffer;
    args = (unsigned char *)(ptr + HEADERS_SIZE_CMD);
    
    UINT8_TO_STREAM(args, ((usPatchesAvailableAtHost) ? SL_PATCHES_REQUEST_FORCE_HOST : SL_PATCHES_REQUEST_DEFAULT));
    
    // IRQ Line asserted - send HCI_CMND_SIMPLE_LINK_START to CC3000
    hci_command_send(HCI_CMND_SIMPLE_LINK_START, ptr, WLAN_SL_INIT_START_PARAMS_LEN);
    SimpleLinkWaitEvent(HCI_CMND_SIMPLE_LINK_START, 0);
}


void wlan_init(tWlanCB               sWlanCB,
               tFWPatches            sFWPatches,
               tDriverPatches        sDriverPatches,
               tBootLoaderPatches    sBootLoaderPatches,
               tWlanReadInteruptPin  sReadWlanInterruptPin,
               tWlanInterruptEnable  sWlanInterruptEnable,
               tWlanInterruptDisable sWlanInterruptDisable,
               tWriteWlanPin         sWriteWlanPin)
{
    
    tSLInformation.sFWPatches = sFWPatches;
    tSLInformation.sDriverPatches = sDriverPatches;
    tSLInformation.sBootLoaderPatches = sBootLoaderPatches;
    
    // init io callback
    tSLInformation.ReadWlanInterruptPin = sReadWlanInterruptPin;
    tSLInformation.WlanInterruptEnable  = sWlanInterruptEnable;
    tSLInformation.WlanInterruptDisable = sWlanInterruptDisable;
    tSLInformation.WriteWlanPin = sWriteWlanPin;
    
    //init asynchronous events callback
    tSLInformation.sWlanCB= sWlanCB;
    
    // By default TX Complete events are routed to host too
    tSLInformation.InformHostOnTxComplete = 1;
}


void SpiReceiveHandler(void *pvBuffer)
{    
    tSLInformation.usEventOrDataReceived = 1;
    tSLInformation.pucReceivedData = (unsigned char *)pvBuffer;
    
    hci_unsolicited_event_handler();
}


void wlan_start(unsigned short usPatchesAvailableAtHost)
{
    
    unsigned long ulSpiIRQState;
    
    tSLInformation.NumberOfSentPackets = 0;
    tSLInformation.NumberOfReleasedPackets = 0;
    tSLInformation.usRxEventOpcode = 0;
    tSLInformation.usNumberOfFreeBuffers = 0;
    tSLInformation.usSlBufferLength = 0;
    tSLInformation.usBufferSize = 0;
    tSLInformation.usRxDataPending = 0;
    tSLInformation.slTransmitDataError = 0;
    tSLInformation.usEventOrDataReceived = 0;
    tSLInformation.pucReceivedData = 0;

    // Allocate the memory for the RX/TX data transactions
    tSLInformation.pucTxCommandBuffer = (unsigned char *)wlan_tx_buffer;
    
    // init spi
    SpiOpen(SpiReceiveHandler);
    // Check the IRQ line
    ulSpiIRQState = tSLInformation.ReadWlanInterruptPin();
    // ASIC 1273 chip enable: toggle WLAN EN line
    tSLInformation.WriteWlanPin( WLAN_ENABLE );

    if (ulSpiIRQState)
    {
        // wait till the IRQ line goes low
        while(tSLInformation.ReadWlanInterruptPin() != 0)
        {
        }
    }
    else
    {
        // wait till the IRQ line goes high and then low
        while(tSLInformation.ReadWlanInterruptPin() == 0)
        {
        }
        while(tSLInformation.ReadWlanInterruptPin() != 0)
        {
        }
    }
    SimpleLink_Init_Start(usPatchesAvailableAtHost);
    
    // Read Buffer's size and finish
    hci_command_send(HCI_CMND_READ_BUFFER_SIZE, tSLInformation.pucTxCommandBuffer, 0);
    SimpleLinkWaitEvent(HCI_CMND_READ_BUFFER_SIZE, 0);
}


void wlan_stop(void)
{
    // ASIC 1273 chip disable
    tSLInformation.WriteWlanPin( WLAN_DISABLE );
    
    // Wait till IRQ line goes high...
    while(tSLInformation.ReadWlanInterruptPin() == 0)
    {
    }
    
    // Free the used by WLAN Driver memory
    if (tSLInformation.pucTxCommandBuffer)
    {
        tSLInformation.pucTxCommandBuffer = 0;
    }
    
    SpiClose();
}


#ifndef CC3000_TINY_DRIVER
long wlan_connect(unsigned long ulSecType,
                  char *ssid,
                  long ssid_len,
                  unsigned char *bssid,
                  unsigned char *key,
                  long key_len)
{
    long ret;
    unsigned char *ptr;
    unsigned char *args;
    unsigned char bssid_zero[] = {0, 0, 0, 0, 0, 0};
    
    ret      = EFAIL;
    ptr      = tSLInformation.pucTxCommandBuffer;
    args     = (ptr + HEADERS_SIZE_CMD);
    
    // Fill in command buffer
    args = UINT32_TO_STREAM(args, 0x0000001c);
    args = UINT32_TO_STREAM(args, ssid_len);
    args = UINT32_TO_STREAM(args, ulSecType);
    args = UINT32_TO_STREAM(args, 0x00000010 + ssid_len);
    args = UINT32_TO_STREAM(args, key_len);
    args = UINT16_TO_STREAM(args, 0);
    
    // padding shall be zeroed
    if(bssid)
    {
        ARRAY_TO_STREAM(args, bssid, ETH_ALEN);
    }
    else
    {
        ARRAY_TO_STREAM(args, bssid_zero, ETH_ALEN);
    }
    
    ARRAY_TO_STREAM(args, ssid, ssid_len);
    
    if(key_len && key)
    {
        ARRAY_TO_STREAM(args, key, key_len);
    }
    
    // Initiate a HCI command
    hci_command_send(HCI_CMND_WLAN_CONNECT, ptr, WLAN_CONNECT_PARAM_LEN + ssid_len + key_len - 1);
    
    // Wait for command complete event
    SimpleLinkWaitEvent(HCI_CMND_WLAN_CONNECT, &ret);
    errno = ret;
    
    return(ret);
}
#else
long wlan_connect(char *ssid, long ssid_len)
{
    long ret;
    unsigned char *ptr;
    unsigned char *args;
    unsigned char bssid_zero[] = {0, 0, 0, 0, 0, 0};
    
    ret      = EFAIL;
    ptr      = tSLInformation.pucTxCommandBuffer;
    args     = (ptr + HEADERS_SIZE_CMD);
    
    // Fill in command buffer
    args = UINT32_TO_STREAM(args, 0x0000001c);
    args = UINT32_TO_STREAM(args, ssid_len);
    args = UINT32_TO_STREAM(args, 0);
    args = UINT32_TO_STREAM(args, 0x00000010 + ssid_len);
    args = UINT32_TO_STREAM(args, 0);
    args = UINT16_TO_STREAM(args, 0);
    
    // padding shall be zeroed
    ARRAY_TO_STREAM(args, bssid_zero, ETH_ALEN);
    ARRAY_TO_STREAM(args, ssid, ssid_len);
    
    // Initiate a HCI command
    hci_command_send(HCI_CMND_WLAN_CONNECT, ptr, WLAN_CONNECT_PARAM_LEN + ssid_len  - 1);
    
    // Wait for command complete event
    SimpleLinkWaitEvent(HCI_CMND_WLAN_CONNECT, &ret);
    errno = ret;
    
    return(ret);
}
#endif


long wlan_disconnect()
{
    long ret;
    unsigned char *ptr;
    
    ret = EFAIL;
    ptr = tSLInformation.pucTxCommandBuffer;
    
    hci_command_send(HCI_CMND_WLAN_DISCONNECT, ptr, 0);
    
    // Wait for command complete event
    SimpleLinkWaitEvent(HCI_CMND_WLAN_DISCONNECT, &ret);
    errno = ret;
    
    return(ret);
}


long wlan_ioctl_set_connection_policy(unsigned long should_connect_to_open_ap, 
                                      unsigned long ulShouldUseFastConnect,
                                      unsigned long ulUseProfiles)
{
    long ret;
    unsigned char *ptr;
    unsigned char *args;
    
    ret = EFAIL;
    ptr = tSLInformation.pucTxCommandBuffer;
    args = (unsigned char *)(ptr + HEADERS_SIZE_CMD);
    
    // Fill in HCI packet structure
    args = UINT32_TO_STREAM(args, should_connect_to_open_ap);
    args = UINT32_TO_STREAM(args, ulShouldUseFastConnect);
    args = UINT32_TO_STREAM(args, ulUseProfiles);
    
    // Initiate a HCI command
    hci_command_send(HCI_CMND_WLAN_IOCTL_SET_CONNECTION_POLICY, ptr, WLAN_SET_CONNECTION_POLICY_PARAMS_LEN);
    
    // Wait for command complete event
    SimpleLinkWaitEvent(HCI_CMND_WLAN_IOCTL_SET_CONNECTION_POLICY, &ret);
    
    return(ret);
}


#ifndef CC3000_TINY_DRIVER
long wlan_add_profile(unsigned long ulSecType, 
                      unsigned char* ucSsid,
                      unsigned long ulSsidLen, 
                      unsigned char *ucBssid,
                      unsigned long ulPriority,
                      unsigned long ulPairwiseCipher_Or_TxKeyLen,
                      unsigned long ulGroupCipher_TxKeyIndex,
                      unsigned long ulKeyMgmt,
                      unsigned char* ucPf_OrKey,
                      unsigned long ulPassPhraseLen)
{
    unsigned short arg_len = 0x00;
    long ret;
    unsigned char *ptr;
    long i = 0;
    unsigned char *args;
    unsigned char bssid_zero[] = {0, 0, 0, 0, 0, 0};
    
    ptr = tSLInformation.pucTxCommandBuffer;
    args = (ptr + HEADERS_SIZE_CMD);
    
    args = UINT32_TO_STREAM(args, ulSecType);
    
    // Setup arguments in accordance with the security type
    switch (ulSecType)
    {
        //OPEN
    case WLAN_SEC_UNSEC:
        {
            args = UINT32_TO_STREAM(args, 0x00000014);
            args = UINT32_TO_STREAM(args, ulSsidLen);
            args = UINT16_TO_STREAM(args, 0);
            if(ucBssid)
            {
                ARRAY_TO_STREAM(args, ucBssid, ETH_ALEN);
            }
            else
            {
                ARRAY_TO_STREAM(args, bssid_zero, ETH_ALEN);
            }
            args = UINT32_TO_STREAM(args, ulPriority);
            ARRAY_TO_STREAM(args, ucSsid, ulSsidLen);
            
            arg_len = WLAN_ADD_PROFILE_NOSEC_PARAM_LEN + ulSsidLen;
        }
        break;
        
        //WEP
    case WLAN_SEC_WEP:
        {
            args = UINT32_TO_STREAM(args, 0x00000020);
            args = UINT32_TO_STREAM(args, ulSsidLen);
            args = UINT16_TO_STREAM(args, 0);
            if(ucBssid)
            {
                ARRAY_TO_STREAM(args, ucBssid, ETH_ALEN);
            }
            else
            {
                ARRAY_TO_STREAM(args, bssid_zero, ETH_ALEN);
            }
            args = UINT32_TO_STREAM(args, ulPriority);
            args = UINT32_TO_STREAM(args, 0x0000000C + ulSsidLen);
            args = UINT32_TO_STREAM(args, ulPairwiseCipher_Or_TxKeyLen);
            args = UINT32_TO_STREAM(args, ulGroupCipher_TxKeyIndex);
            ARRAY_TO_STREAM(args, ucSsid, ulSsidLen);
            
            for(i = 0; i < 4; i++)
            {
                unsigned char *p = &ucPf_OrKey[i * ulPairwiseCipher_Or_TxKeyLen];
                
                ARRAY_TO_STREAM(args, p, ulPairwiseCipher_Or_TxKeyLen);
            }        
            
            arg_len = WLAN_ADD_PROFILE_WEP_PARAM_LEN + ulSsidLen + 
                ulPairwiseCipher_Or_TxKeyLen * 4;
            
        }
        break;
        
        //WPA
        //WPA2
    case WLAN_SEC_WPA:
    case WLAN_SEC_WPA2:
        {
            args = UINT32_TO_STREAM(args, 0x00000028);
            args = UINT32_TO_STREAM(args, ulSsidLen);
            args = UINT16_TO_STREAM(args, 0);
            if(ucBssid)
            {
                ARRAY_TO_STREAM(args, ucBssid, ETH_ALEN);
            }
            else
            {
                ARRAY_TO_STREAM(args, bssid_zero, ETH_ALEN);
            }
            args = UINT32_TO_STREAM(args, ulPriority);
            args = UINT32_TO_STREAM(args, ulPairwiseCipher_Or_TxKeyLen);
            args = UINT32_TO_STREAM(args, ulGroupCipher_TxKeyIndex);
            args = UINT32_TO_STREAM(args, ulKeyMgmt);
            args = UINT32_TO_STREAM(args, 0x00000008 + ulSsidLen);
            args = UINT32_TO_STREAM(args, ulPassPhraseLen);
            ARRAY_TO_STREAM(args, ucSsid, ulSsidLen);
            ARRAY_TO_STREAM(args, ucPf_OrKey, ulPassPhraseLen);
            
            arg_len = WLAN_ADD_PROFILE_WPA_PARAM_LEN + ulSsidLen + ulPassPhraseLen;
        }
        
        break;
    }    
    
    // Initiate a HCI command
    hci_command_send(HCI_CMND_WLAN_IOCTL_ADD_PROFILE, ptr, arg_len);
    
    // Wait for command complete event
    SimpleLinkWaitEvent(HCI_CMND_WLAN_IOCTL_ADD_PROFILE, &ret);
    
    return(ret);
}
#else
long wlan_add_profile(unsigned long ulSecType, 
                      unsigned char* ucSsid,
                      unsigned long ulSsidLen, 
                      unsigned char *ucBssid,
                      unsigned long ulPriority,
                      unsigned long ulPairwiseCipher_Or_TxKeyLen,
                      unsigned long ulGroupCipher_TxKeyIndex,
                      unsigned long ulKeyMgmt,
                      unsigned char* ucPf_OrKey,
                      unsigned long ulPassPhraseLen)
{
    return -1;
}
#endif


long wlan_ioctl_del_profile(unsigned long ulIndex)
{
    long ret;
    unsigned char *ptr;
    unsigned char *args;
    
    ptr = tSLInformation.pucTxCommandBuffer;
    args = (unsigned char *)(ptr + HEADERS_SIZE_CMD);
    
    // Fill in HCI packet structure
    args = UINT32_TO_STREAM(args, ulIndex);
    ret = EFAIL;
    
    // Initiate a HCI command
    hci_command_send(HCI_CMND_WLAN_IOCTL_DEL_PROFILE, ptr, WLAN_DEL_PROFILE_PARAMS_LEN);
    
    // Wait for command complete event
    SimpleLinkWaitEvent(HCI_CMND_WLAN_IOCTL_DEL_PROFILE, &ret);
    
    return(ret);
}


#ifndef CC3000_TINY_DRIVER
long wlan_ioctl_get_scan_results(unsigned long ulScanTimeout,
                                 unsigned char *ucResults)
{
    unsigned char *ptr;
    unsigned char *args;
    
    ptr = tSLInformation.pucTxCommandBuffer;
    args = (ptr + HEADERS_SIZE_CMD);
    
    // Fill in temporary command buffer
    args = UINT32_TO_STREAM(args, ulScanTimeout);
    
    // Initiate a HCI command
    hci_command_send(HCI_CMND_WLAN_IOCTL_GET_SCAN_RESULTS, ptr, WLAN_GET_SCAN_RESULTS_PARAMS_LEN);
    
    // Wait for command complete event
    SimpleLinkWaitEvent(HCI_CMND_WLAN_IOCTL_GET_SCAN_RESULTS, ucResults);
    
    return(0);
}
#endif


#ifndef CC3000_TINY_DRIVER
long wlan_ioctl_set_scan_params(unsigned long uiEnable,
                                unsigned long uiMinDwellTime,
                                unsigned long uiMaxDwellTime,
                                unsigned long uiNumOfProbeRequests,
                                unsigned long uiChannelMask,
                                long iRSSIThreshold,
                                unsigned long uiSNRThreshold,
                                unsigned long uiDefaultTxPower,
                                unsigned long *aiIntervalList)
{
    unsigned long  uiRes;
    unsigned char *ptr;
    unsigned char *args;
    
    ptr = tSLInformation.pucTxCommandBuffer;
    args = (ptr + HEADERS_SIZE_CMD);
    
    // Fill in temporary command buffer
    args = UINT32_TO_STREAM(args, 36);
    args = UINT32_TO_STREAM(args, uiEnable);
    args = UINT32_TO_STREAM(args, uiMinDwellTime);
    args = UINT32_TO_STREAM(args, uiMaxDwellTime);
    args = UINT32_TO_STREAM(args, uiNumOfProbeRequests);
    args = UINT32_TO_STREAM(args, uiChannelMask);
    args = UINT32_TO_STREAM(args, iRSSIThreshold);
    args = UINT32_TO_STREAM(args, uiSNRThreshold);
    args = UINT32_TO_STREAM(args, uiDefaultTxPower);
    ARRAY_TO_STREAM(args, aiIntervalList, sizeof(unsigned long) * SL_SET_SCAN_PARAMS_INTERVAL_LIST_SIZE);
    
    // Initiate a HCI command
    hci_command_send(HCI_CMND_WLAN_IOCTL_SET_SCANPARAM, ptr, WLAN_SET_SCAN_PARAMS_LEN);
    
    // Wait for command complete event
    SimpleLinkWaitEvent(HCI_CMND_WLAN_IOCTL_SET_SCANPARAM, &uiRes);
    
    return(uiRes);
}
#endif


long wlan_set_event_mask(unsigned long ulMask)
{
    long ret;
    unsigned char *ptr;
    unsigned char *args;
    
    
    if ((ulMask & HCI_EVNT_WLAN_TX_COMPLETE) == HCI_EVNT_WLAN_TX_COMPLETE)
    {
        tSLInformation.InformHostOnTxComplete = 0;
        
        // Since an event is a virtual event - i.e. it is not coming from CC3000
        // there is no need to send anything to the device if it was an only event
        if (ulMask == HCI_EVNT_WLAN_TX_COMPLETE)
        {
            return 0;
        }
        
        ulMask &= ~HCI_EVNT_WLAN_TX_COMPLETE;
        ulMask |= HCI_EVNT_WLAN_UNSOL_BASE;
    }
    else
    {
        tSLInformation.InformHostOnTxComplete = 1;
    }
    
    ret = EFAIL;
    ptr = tSLInformation.pucTxCommandBuffer;
    args = (unsigned char *)(ptr + HEADERS_SIZE_CMD);
    
    // Fill in HCI packet structure
    args = UINT32_TO_STREAM(args, ulMask);
    
    // Initiate a HCI command
    hci_command_send(HCI_CMND_EVENT_MASK, ptr, WLAN_SET_MASK_PARAMS_LEN);
    
    // Wait for command complete event
    SimpleLinkWaitEvent(HCI_CMND_EVENT_MASK, &ret);
    
    return(ret);
}


#ifndef CC3000_TINY_DRIVER
long wlan_ioctl_statusget(void)
{
    long ret;
    unsigned char *ptr;
    
    ret = EFAIL;
    ptr = tSLInformation.pucTxCommandBuffer;
    
    hci_command_send(HCI_CMND_WLAN_IOCTL_STATUSGET,ptr, 0);
    
    // Wait for command complete event
    SimpleLinkWaitEvent(HCI_CMND_WLAN_IOCTL_STATUSGET, &ret);
    
    return(ret);    
}
#endif


long wlan_smart_config_start(unsigned long algoEncryptedFlag)
{
    long ret;
    unsigned char *ptr;
    unsigned char *args;
    
    ret = EFAIL;
    ptr = tSLInformation.pucTxCommandBuffer;
    args = (unsigned char *)(ptr + HEADERS_SIZE_CMD);
    
    // Fill in HCI packet structure
    args = UINT32_TO_STREAM(args, algoEncryptedFlag);
    ret = EFAIL;
    
    hci_command_send(HCI_CMND_WLAN_IOCTL_SIMPLE_CONFIG_START, ptr, WLAN_SMART_CONFIG_START_PARAMS_LEN);
    
    // Wait for command complete event
    SimpleLinkWaitEvent(HCI_CMND_WLAN_IOCTL_SIMPLE_CONFIG_START, &ret);
    
    return(ret);    
}


long wlan_smart_config_stop(void)
{
    long ret;
    unsigned char *ptr;
    
    ret = EFAIL;
    ptr = tSLInformation.pucTxCommandBuffer;
    
    hci_command_send(HCI_CMND_WLAN_IOCTL_SIMPLE_CONFIG_STOP, ptr, 0);
    
    // Wait for command complete event
    SimpleLinkWaitEvent(HCI_CMND_WLAN_IOCTL_SIMPLE_CONFIG_STOP, &ret);
    
    return(ret);    
}


long wlan_smart_config_set_prefix(char* cNewPrefix)
{
    long ret;
    unsigned char *ptr;
    unsigned char *args;
    
    ret = EFAIL;
    ptr = tSLInformation.pucTxCommandBuffer;
    args = (ptr + HEADERS_SIZE_CMD);
    
    if (cNewPrefix == NULL)
        return ret;
    else    // with the new Smart Config, prefix must be TTT
    {
        *cNewPrefix = 'T';
        *(cNewPrefix + 1) = 'T';
        *(cNewPrefix + 2) = 'T';
    }
    
    ARRAY_TO_STREAM(args, cNewPrefix, SL_SIMPLE_CONFIG_PREFIX_LENGTH);
    
    hci_command_send(HCI_CMND_WLAN_IOCTL_SIMPLE_CONFIG_SET_PREFIX, ptr, SL_SIMPLE_CONFIG_PREFIX_LENGTH);
    
    // Wait for command complete event
    SimpleLinkWaitEvent(HCI_CMND_WLAN_IOCTL_SIMPLE_CONFIG_SET_PREFIX, &ret);
    
    return(ret);    
}


#ifndef CC3000_UNENCRYPTED_SMART_CONFIG
long wlan_smart_config_process()
{
    signed long    returnValue;
    unsigned long ssidLen, keyLen;
    unsigned char *decKeyPtr;
    unsigned char *ssidPtr;
    
    // read the key from EEPROM - fileID 12
    returnValue = aes_read_key(key);
    
    if (returnValue != 0)
        return returnValue;
    
    // read the received data from fileID #13 and parse it according to the followings:
    // 1) SSID LEN - not encrypted
    // 2) SSID - not encrypted
    // 3) KEY LEN - not encrypted. always 32 bytes long
    // 4) Security type - not encrypted
    // 5) KEY - encrypted together with true key length as the first byte in KEY
    //     to elaborate, there are two corner cases:
    //        1) the KEY is 32 bytes long. In this case, the first byte does not represent KEY length
    //        2) the KEY is 31 bytes long. In this case, the first byte represent KEY length and equals 31
    returnValue = nvmem_read(NVMEM_SHARED_MEM_FILEID, SMART_CONFIG_PROFILE_SIZE, 0, profileArray);
    
    if (returnValue != 0)
        return returnValue;
    
    ssidPtr = &profileArray[1];
    
    ssidLen = profileArray[0];
    
    decKeyPtr = &profileArray[profileArray[0] + 3];
    
    aes_decrypt(decKeyPtr, key);
    if (profileArray[profileArray[0] + 1] > 16)
        aes_decrypt((unsigned char *)(decKeyPtr + 16), key);
    
    if (*(unsigned char *)(decKeyPtr +31) != 0)
    {
        if (*decKeyPtr == 31)
        {
            keyLen = 31;
            decKeyPtr++;
        }
        else
        {
            keyLen = 32;
        }
    }
    else
    {
        keyLen = *decKeyPtr;
        decKeyPtr++;
    }
    
    // add a profile
    switch (profileArray[profileArray[0] + 2])
    {
    case WLAN_SEC_UNSEC://None
         {
            returnValue = wlan_add_profile(profileArray[profileArray[0] + 2],     // security type
                                           ssidPtr,                               // SSID
                                           ssidLen,                               // SSID length
                                           NULL,                                  // BSSID
                                           1,                                     // Priority
                                           0, 0, 0, 0, 0);
            
            break;
         }
        
    case WLAN_SEC_WEP://WEP
        {
            returnValue = wlan_add_profile(profileArray[profileArray[0] + 2],     // security type
                                           ssidPtr,                               // SSID
                                           ssidLen,                               // SSID length
                                           NULL,                                  // BSSID
                                           1,                                     // Priority
                                           keyLen,                                // KEY length
                                           0,                                     // KEY index
                                           0,
                                           decKeyPtr,                             // KEY
                                           0);
            
            break;
        }
        
    case WLAN_SEC_WPA:  //WPA
    case WLAN_SEC_WPA2: //WPA2
        {
            returnValue = wlan_add_profile(WLAN_SEC_WPA2,     // security type
                                           ssidPtr,
                                           ssidLen,
                                           NULL,              // BSSID
                                           1,                 // Priority
                                           0x18,              // PairwiseCipher
                                           0x1e,              // GroupCipher
                                           2,                 // KEY management
                                           decKeyPtr,         // KEY
                                           keyLen);           // KEY length
            
            break;
        }
    }
    
    return returnValue;
}
#endif //CC3000_UNENCRYPTED_SMART_CONFIG        


