/*****************************************************************************
*
*  netapp.h  - CC3000 Host Driver Implementation.
*  Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the   
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*****************************************************************************/
#ifndef __NETAPP_H__
#define __NETAPP_H__

#include "nvmem.h"

//*****************************************************************************
//
//! \addtogroup netapp
//! @{
//
//*****************************************************************************

/** CC3000 Host driver - netapp
*
*/

#ifdef __cplusplus
extern "C" {
#endif

#define MIN_TIMER_VAL_SECONDS      20
#define MIN_TIMER_SET(t)    if ((0 != t) && (t < MIN_TIMER_VAL_SECONDS)) \
                            { \
                                t = MIN_TIMER_VAL_SECONDS; \
                            }


#define NETAPP_DHCP_PARAMS_LEN                 (20)
#define NETAPP_SET_TIMER_PARAMS_LEN            (20)
#define NETAPP_SET_DEBUG_LEVEL_PARAMS_LEN      (4)
#define NETAPP_PING_SEND_PARAMS_LEN            (16)


typedef struct _netapp_dhcp_ret_args_t
{
    unsigned char aucIP[4];
    unsigned char aucSubnetMask[4];
    unsigned char aucDefaultGateway[4];
    unsigned char aucDHCPServer[4];
    unsigned char aucDNSServer[4];
}tNetappDhcpParams;

typedef struct _netapp_ipconfig_ret_args_t
{
    unsigned char aucIP[4];
    unsigned char aucSubnetMask[4];
    unsigned char aucDefaultGateway[4];
    unsigned char aucDHCPServer[4];
    unsigned char aucDNSServer[4];
    unsigned char uaMacAddr[6];
    unsigned char uaSSID[32];
}tNetappIpconfigRetArgs;


/*Ping send report parameters*/
typedef struct _netapp_pingreport_args
{
    unsigned long packets_sent;
    unsigned long packets_received;
    unsigned long min_round_time;
    unsigned long max_round_time;
    unsigned long avg_round_time;
} netapp_pingreport_args_t;


/**
* Configure device MAC address and store it in NVMEM. 
* The value of the MAC address configured through the API will be\n
* stored in CC3000 non volatile memory, thus preserved over resets.\n
* @param  mac   device mac address, 6 bytes. Saved: yes 
* @return       return on success 0, otherwise error.
*/
extern long netapp_config_mac_adrress( unsigned char *mac );

/**
* Configure the network interface, static or dynamic (DHCP).
* In order to activate DHCP mode, aucIP, aucSubnetMask, aucDefaultGateway must be 0.\n
* The default mode of CC3000 is DHCP mode. The configuration is saved in non volatile memory\n
* and thus preserved over resets.\n
* @param  aucIP               device mac address, 6 bytes. Saved: yes 
* @param  aucSubnetMask       device mac address, 6 bytes. Saved: yes 
* @param  aucDefaultGateway   device mac address, 6 bytes. Saved: yes 
* @param  aucDNSServer        device mac address, 6 bytes. Saved: yes 
* @return       0 on success, otherwise error.
* @note         If the mode is altered, a reset of CC3000 device is required to apply the changes.\n
*               Also note that an asynchronous event of type 'DHCP_EVENT' is generated only when\n
*               a connection to the AP was established. This event is generated when an IP address\n
*               is allocated either by the DHCP server or by static allocation.\n
*/
extern long netapp_dhcp(unsigned long *aucIP, unsigned long *aucSubnetMask,unsigned long *aucDefaultGateway, unsigned long *aucDNSServer);

/**
* Set new timeout values for DHCP lease timeout, ARP  refresh timeout, keepalive event timeout and socket inactivity timeout 
* @param  aucDHCP    DHCP lease time request, also impact\n 
*                    the DHCP renew timeout.\n
*                    Range:               [0-0xffffffff] seconds,\n
*                                         0 or 0xffffffff = infinite lease timeout.\n
*                    Resolution:          10 seconds.\n
*                    Influence:           only after reconnecting to the AP. \n
*                    Minimal bound value: MIN_TIMER_VAL_SECONDS - 20 seconds.\n
*                    The parameter is saved into the CC3000 NVMEM.\n 
*                    The default value on CC3000 is 14400 seconds.\n
*
* @param  aucARP     ARP refresh timeout, if ARP entry is not updated by\n
*                    incoming packet, the ARP entry will be  deleted by\n
*                    the end of the timeout. \n
*                    Range:               [0-0xffffffff] seconds, 0 = infinite ARP timeout\n
*                    Resolution:          10 seconds.\n
*                    Influence:           at runtime.\n
*                    Minimal bound value: MIN_TIMER_VAL_SECONDS - 20 seconds\n
*                    The parameter is saved into the CC3000 NVMEM.\n 
*                    The default value on CC3000 is 3600 seconds.\n
*
* @param  aucKeepalive   Keepalive event sent by the end of keepalive timeout\n
*                        Range:               [0-0xffffffff] seconds, 0 == infinite timeout\n
*                        Resolution:          10 seconds.\n
*                        Influence:           at runtime.\n
*                        Minimal bound value: MIN_TIMER_VAL_SECONDS - 20 sec\n
*                        The parameter is saved into the CC3000 NVMEM. \n
*                        The default value on CC3000 is 10 seconds.\n
*
* @param  aucInactivity   Socket inactivity timeout, socket timeout is\n
*                         refreshed by incoming or outgoing packet, by the\n
*                         end of the socket timeout the socket will be closed\n
*                         Range:               [0-0xffffffff] sec, 0 == infinite timeout.\n
*                         Resolution:          10 seconds.\n
*                         Influence:           at runtime.\n
*                         Minimal bound value: MIN_TIMER_VAL_SECONDS - 20 sec\n
*                         The parameter is saved into the CC3000 NVMEM.\n 
*                         The default value on CC3000 is 60 seconds.\n
*
* @return 0 on success,otherwise error.
*
* @note   A parameter set to a non zero value less than 20s automatically changes to 20s.
*/
#ifndef CC3000_TINY_DRIVER
extern long netapp_timeout_values(unsigned long *aucDHCP, unsigned long *aucARP,unsigned long *aucKeepalive,    unsigned long *aucInactivity);
#endif

/**
* send ICMP ECHO_REQUEST to network hosts
* @param  ip              destination IP address
* @param  pingAttempts    number of echo requests to send
* @param  pingSize        send buffer size which may be up to 1400 bytes 
* @param  pingTimeout     Time to wait for a response,in milliseconds.
* @return 0 on success, otherwise error.
*
* @note     A succesful operation will generate an asynchronous ping report event.\n
*           The report structure is defined by structure netapp_pingreport_args_t.\n
* @warning  Calling this function while a Ping Request is in progress will kill the ping request in progress.
*/
#ifndef CC3000_TINY_DRIVER
extern long netapp_ping_send(unsigned long *ip, unsigned long ulPingAttempts, unsigned long ulPingSize, unsigned long ulPingTimeout);
#endif

/**
* Stop any ping request.
* @param   none
* @return   0 on success
*          -1 on error      
*/
#ifndef CC3000_TINY_DRIVER
extern long netapp_ping_stop(void);
#endif

/**
* Ping status request.
* This API triggers the CC3000 to send asynchronous events: HCI_EVNT_WLAN_ASYNC_PING_REPORT.\n
* This event will create the report structure in netapp_pingreport_args_t.\n
* This structure is filled with ping results until the API is triggered.\n
* netapp_pingreport_args_t: packets_sent     - echo sent\n
*                           packets_received - echo reply\n
*                           min_round_time   - minimum round time\n
*                           max_round_time   - max round time\n
*                           avg_round_time   - average round time\n
*
* @param   none
* @return  none
* @note    When a ping operation is not active, the returned structure fields are 0.
*/
#ifndef CC3000_TINY_DRIVER
extern void netapp_ping_report(void);
#endif


/**
* Get the CC3000 Network interface information.
* This information is only available after establishing a WLAN connection.\n
* Undefined values are returned when this function is called before association.\n
* @param[out]  ipconfig  pointer to a tNetappIpconfigRetArgs structure for storing the network interface configuration.\n
*                        tNetappIpconfigRetArgs: aucIP             - ip address,\n
*                                                aucSubnetMask     - mask
*                                                aucDefaultGateway - default gateway address\n
*                                                aucDHCPServer     - dhcp server address\n
*                                                aucDNSServer      - dns server address\n
*                                                uaMacAddr         - mac address\n
*                                                uaSSID            - connected AP ssid\n
* @return  none
* @note    This function is useful for figuring out the IP Configuration of\n
*          the device when DHCP is used and for figuring out the SSID of\n
*          the Wireless network the device is associated with.\n
*/
extern void netapp_ipconfig( tNetappIpconfigRetArgs * ipconfig );


/**
* Flush ARP table
* @param   none
* @return  none
*/
#ifndef CC3000_TINY_DRIVER
extern long netapp_arp_flush(void);
#endif


#ifdef    __cplusplus
}
#endif // __cplusplus

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************

#endif    // __NETAPP_H__






