/****************************************************************************
* File       : cc3000_soi_hci
* Date       : 12/11/2012 (Menu "banner" reports actual build date)
* Purpose    : Wi-Go SPI interface driver to CC3000 Wi-Fi module 
* Author     : Peter Fenn, Avnet Global Technical Marketing
* Description: SPI interface driver between Host MCU (KL25Z) and CC3000
*****************************************************************************

*****************************************************************************
*
*  spi.c - CC3000 Host Driver Implementation.
*  Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*****************************************************************************/

#include "cc3000_spi_hci.h"

/* ===========================================================================================
                                              SPI 
   =========================================================================================== */

tSpiInformation sSpiInformation;

//TX and RX buffers
char spi_buffer[CC3000_RX_BUFFER_SIZE];
unsigned char wlan_tx_buffer[CC3000_TX_BUFFER_SIZE];

void SpiClose(void)
{
   if (sSpiInformation.pRxPacket)
   {
      sSpiInformation.pRxPacket = 0;
   }
    tSLInformation.WlanInterruptDisable();
}


void SpiOpen(gcSpiHandleRx pfRxHandler)
{
   sSpiInformation.ulSpiState = eSPI_STATE_POWERUP;
   sSpiInformation.SPIRxHandler = pfRxHandler;
   sSpiInformation.usTxPacketLength = 0;
   sSpiInformation.pTxPacket = NULL;
   sSpiInformation.pRxPacket = (unsigned char *)spi_buffer;
   sSpiInformation.usRxPacketLength = 0;
   spi_buffer[CC3000_RX_BUFFER_SIZE - 1] = CC3000_BUFFER_MAGIC_NUMBER;
   wlan_tx_buffer[CC3000_TX_BUFFER_SIZE - 1] = CC3000_BUFFER_MAGIC_NUMBER;
   tSLInformation.WlanInterruptEnable();
}


long SpiFirstWrite(unsigned char *ucBuf, unsigned short usLength)
{
    WLAN_ASSERT_CS;
    wait_us(50);

    // SPI writes first 4 bytes of data
    SpiWriteDataSynchronous(ucBuf, 4);
    wait_us(50);

    SpiWriteDataSynchronous(ucBuf + 4, usLength - 4);

    // From this point on - operate in a regular way
    sSpiInformation.ulSpiState = eSPI_STATE_IDLE;

    WLAN_DEASSERT_CS;

    return(0);
}


long SpiWrite(unsigned char *pUserBuffer, unsigned short usLength)
{
   unsigned char ucPad = 0;
   // check the total length of the packet in order to figure out if padding is necessary
   if(!(usLength & 0x0001))
   {
      ucPad++;
   }

   pUserBuffer[0] = WRITE;
   pUserBuffer[1] = HI(usLength + ucPad);
   pUserBuffer[2] = LO(usLength + ucPad);
   pUserBuffer[3] = 0;
   pUserBuffer[4] = 0;

   usLength += (SPI_HEADER_SIZE + ucPad);

   // The magic number resides at the end of the TX/RX buffer (1 byte after the allocated size)
   // If the magic number is overwitten - buffer overrun occurred - we will be stuck here forever!
   if (wlan_tx_buffer[CC3000_TX_BUFFER_SIZE - 1] != CC3000_BUFFER_MAGIC_NUMBER)
   {
      while (1);
   }

   if (sSpiInformation.ulSpiState == eSPI_STATE_POWERUP)
   {
      while (sSpiInformation.ulSpiState != eSPI_STATE_INITIALIZED);
   }

   if (sSpiInformation.ulSpiState == eSPI_STATE_INITIALIZED)
   {
      // TX/RX transaction over SPI after powerup: IRQ is low - send read buffer size command
      SpiFirstWrite(pUserBuffer, usLength);
   }
   else
   {
      // Prevent occurence of a race condition when 2 back to back packets are sent to the
      // device, so the state will move to IDLE and once again to not IDLE due to IRQ
      tSLInformation.WlanInterruptDisable();

      while (sSpiInformation.ulSpiState != eSPI_STATE_IDLE);

      sSpiInformation.ulSpiState = eSPI_STATE_WRITE_IRQ;
      sSpiInformation.pTxPacket = pUserBuffer;
      sSpiInformation.usTxPacketLength = usLength;

      // Assert the CS line and wait until the IRQ line is active, then initialize the write operation
      WLAN_ASSERT_CS;

      tSLInformation.WlanInterruptEnable();
   }

   // Wait until the transaction ends
   while (sSpiInformation.ulSpiState != eSPI_STATE_IDLE);
   return(0);
}


void SpiWriteDataSynchronous(unsigned char *data, unsigned short size)
{
   while(size)
   {
        WLAN_SPI_WRITE;
        size--;
   }
}


void SpiReadDataSynchronous(unsigned char *data, unsigned short size)
{
   long i = 0;
   for (i = 0; i < size; i++)
   {
      data[i] = WLAN_SPI_READ;
   }
}


long SpiReadDataCont(void)
{
   long data_to_recv;
   unsigned char *evnt_buff, type;

   //determine the packet type
   evnt_buff =  sSpiInformation.pRxPacket;
   data_to_recv = 0;
   STREAM_TO_UINT8((char *)(evnt_buff + SPI_HEADER_SIZE), HCI_PACKET_TYPE_OFFSET, type);

    switch(type)
    {
        case HCI_TYPE_DATA:
        {
         // Read the remaining data..
         STREAM_TO_UINT16((char *)(evnt_buff + SPI_HEADER_SIZE), HCI_DATA_LENGTH_OFFSET, data_to_recv);
         if (!((HEADERS_SIZE_EVNT + data_to_recv) & 1))
         {
              data_to_recv++;
         }

         if (data_to_recv)
         {
               SpiReadDataSynchronous(evnt_buff + 10, data_to_recv);
         }
            break;
        }
        case HCI_TYPE_EVNT:
        {
         // Calculate the rest length of the data
            STREAM_TO_UINT8((char *)(evnt_buff + SPI_HEADER_SIZE), HCI_EVENT_LENGTH_OFFSET, data_to_recv);
         data_to_recv -= 1;
         // Add padding byte if needed
         if ((HEADERS_SIZE_EVNT + data_to_recv) & 1)
         {
               data_to_recv++;
         }

         if (data_to_recv)
         {
               SpiReadDataSynchronous(evnt_buff + 10, data_to_recv);
         }

         sSpiInformation.ulSpiState = eSPI_STATE_READ_EOT;
            break;
        }
    }
    return (0);
}


void WLAN_IRQHandler(void)
{
   if (sSpiInformation.ulSpiState == eSPI_STATE_POWERUP)
   {
      // Inform HCI Layer that IRQ occured after powerup
      sSpiInformation.ulSpiState = eSPI_STATE_INITIALIZED;
   }
   else if (sSpiInformation.ulSpiState == eSPI_STATE_IDLE)
   {
      sSpiInformation.ulSpiState = eSPI_STATE_READ_IRQ;
      /* IRQ line goes low - acknowledge it */
      WLAN_ASSERT_CS;
      SpiReadDataSynchronous(sSpiInformation.pRxPacket, 10);
      sSpiInformation.ulSpiState = eSPI_STATE_READ_EOT;


      // The header was read - continue with the payload read
      if (!SpiReadDataCont())
      {
          // All the data was read - finalize handling by switching to the task
          // Trigger Rx processing
          tSLInformation.WlanInterruptDisable();
          WLAN_DEASSERT_CS;
          // The magic number resides at the end of the TX/RX buffer (1 byte after the allocated size)
          // If the magic number is overwitten - buffer overrun occurred - we will be stuck here forever!
          if (sSpiInformation.pRxPacket[CC3000_RX_BUFFER_SIZE - 1] != CC3000_BUFFER_MAGIC_NUMBER)
              {
                  while (1);
              }
              sSpiInformation.ulSpiState = eSPI_STATE_IDLE;
              sSpiInformation.SPIRxHandler(sSpiInformation.pRxPacket + SPI_HEADER_SIZE);
      }
   }
   else if (sSpiInformation.ulSpiState == eSPI_STATE_WRITE_IRQ)
   {
      SpiWriteDataSynchronous(sSpiInformation.pTxPacket, sSpiInformation.usTxPacketLength);
      sSpiInformation.ulSpiState = eSPI_STATE_IDLE;
      WLAN_DEASSERT_CS;
   }
}


/* ===========================================================================================
                                              HCI 
   =========================================================================================== */

unsigned short hci_command_send(unsigned short usOpcode, unsigned char *pucBuff, unsigned char ucArgsLength)
{ 
    unsigned char *stream;
    
    stream = (pucBuff + SPI_HEADER_SIZE);
    
    UINT8_TO_STREAM(stream, HCI_TYPE_CMND);
    stream = UINT16_TO_STREAM(stream, usOpcode);
    UINT8_TO_STREAM(stream, ucArgsLength);
    //Update the opcode of the event we will be waiting for
    SpiWrite(pucBuff, ucArgsLength + SIMPLE_LINK_HCI_CMND_HEADER_SIZE);
    return(0);
}


long hci_data_send(unsigned char ucOpcode, 
                   unsigned char *ucArgs,
                   unsigned short usArgsLength, 
                   unsigned short usDataLength,
                   const unsigned char *ucTail,
                   unsigned short usTailLength)
{
    unsigned char *stream;
    
    stream = ((ucArgs) + SPI_HEADER_SIZE);
    
    UINT8_TO_STREAM(stream, HCI_TYPE_DATA);
    UINT8_TO_STREAM(stream, ucOpcode);
    UINT8_TO_STREAM(stream, usArgsLength);
    stream = UINT16_TO_STREAM(stream, usArgsLength + usDataLength + usTailLength);
    
    // Send the packet
    SpiWrite(ucArgs, SIMPLE_LINK_HCI_DATA_HEADER_SIZE + usArgsLength + usDataLength + usTailLength);
    
    return(ESUCCESS);
}


void hci_data_command_send(unsigned short usOpcode,
                           unsigned char *pucBuff,
                           unsigned char ucArgsLength,
                           unsigned short ucDataLength)
{ 
     unsigned char *stream = (pucBuff + SPI_HEADER_SIZE);
    
    UINT8_TO_STREAM(stream, HCI_TYPE_DATA);
    UINT8_TO_STREAM(stream, usOpcode);
    UINT8_TO_STREAM(stream, ucArgsLength);
    stream = UINT16_TO_STREAM(stream, ucArgsLength + ucDataLength);
    
    // Send the command
    SpiWrite(pucBuff, ucArgsLength + ucDataLength + SIMPLE_LINK_HCI_DATA_CMND_HEADER_SIZE);
    
    return;
}


void hci_patch_send(unsigned char ucOpcode,
                    unsigned char *pucBuff,
                    char *patch,
                    unsigned short usDataLength)
{ 
    unsigned short usTransLength;
    unsigned char *stream = (pucBuff + SPI_HEADER_SIZE);
    UINT8_TO_STREAM(stream, HCI_TYPE_PATCH);
    UINT8_TO_STREAM(stream, ucOpcode);
    stream = UINT16_TO_STREAM(stream, usDataLength + SIMPLE_LINK_HCI_PATCH_HEADER_SIZE);
    if (usDataLength <= SL_PATCH_PORTION_SIZE)
    {
        UINT16_TO_STREAM(stream, usDataLength);
        stream = UINT16_TO_STREAM(stream, usDataLength);
        memcpy((pucBuff + SPI_HEADER_SIZE) + HCI_PATCH_HEADER_SIZE, patch, usDataLength);
        // Update the opcode of the event we will be waiting for
        SpiWrite(pucBuff, usDataLength + HCI_PATCH_HEADER_SIZE);
    }
    else
    {
        
        usTransLength = (usDataLength/SL_PATCH_PORTION_SIZE);
        UINT16_TO_STREAM(stream, usDataLength + SIMPLE_LINK_HCI_PATCH_HEADER_SIZE + usTransLength*SIMPLE_LINK_HCI_PATCH_HEADER_SIZE);
        stream = UINT16_TO_STREAM(stream, SL_PATCH_PORTION_SIZE);
        memcpy(pucBuff + SPI_HEADER_SIZE + HCI_PATCH_HEADER_SIZE, patch, SL_PATCH_PORTION_SIZE);
        usDataLength -= SL_PATCH_PORTION_SIZE;
        patch += SL_PATCH_PORTION_SIZE;
        
        // Update the opcode of the event we will be waiting for
        SpiWrite(pucBuff, SL_PATCH_PORTION_SIZE + HCI_PATCH_HEADER_SIZE);
        
        stream = (pucBuff + SPI_HEADER_SIZE);
        while (usDataLength)
        {
            if (usDataLength <= SL_PATCH_PORTION_SIZE)
            {
                usTransLength = usDataLength;
                usDataLength = 0;
                
            }
            else
            {
                usTransLength = SL_PATCH_PORTION_SIZE;
                usDataLength -= usTransLength;
            }
            
            *(unsigned short *)stream = usTransLength;
            memcpy(stream + SIMPLE_LINK_HCI_PATCH_HEADER_SIZE, patch, usTransLength);
            patch += usTransLength;
            
            // Update the opcode of the event we will be waiting for
            SpiWrite((unsigned char *)stream, usTransLength + sizeof(usTransLength));
        }
    }
}




