/*****************************************************************************
*
*  wlan.h  - CC3000 Host Driver Implementation.
*  Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the   
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*****************************************************************************/
#ifndef __WLAN_H__
#define __WLAN_H__

#include "security.h"

//*****************************************************************************
//
//! \addtogroup wlan
//! @{
//
//*****************************************************************************

/** CC3000 Host driver - WLAN
*
*/

#ifdef    __cplusplus
extern "C" {
#endif

#define SMART_CONFIG_PROFILE_SIZE        67        // 67 = 32 (max ssid) + 32 (max key) + 1 (SSID length) + 1 (security type) + 1 (key length)

/* patches type */
#define PATCHES_HOST_TYPE_WLAN_DRIVER   0x01
#define PATCHES_HOST_TYPE_WLAN_FW       0x02
#define PATCHES_HOST_TYPE_BOOTLOADER    0x03

#define SL_SET_SCAN_PARAMS_INTERVAL_LIST_SIZE    (16)
#define SL_SIMPLE_CONFIG_PREFIX_LENGTH           (3)
#define ETH_ALEN                                 (6)
#define MAXIMAL_SSID_LENGTH                      (32)

#define SL_PATCHES_REQUEST_DEFAULT               (0)
#define SL_PATCHES_REQUEST_FORCE_HOST            (1)
#define SL_PATCHES_REQUEST_FORCE_NONE            (2)


#define      WLAN_SEC_UNSEC  (0)
#define      WLAN_SEC_WEP    (1)
#define      WLAN_SEC_WPA    (2)
#define      WLAN_SEC_WPA2   (3)


#define WLAN_SL_INIT_START_PARAMS_LEN           (1)
#define WLAN_PATCH_PARAMS_LENGTH                (8)
#define WLAN_SET_CONNECTION_POLICY_PARAMS_LEN   (12)
#define WLAN_DEL_PROFILE_PARAMS_LEN             (4)
#define WLAN_SET_MASK_PARAMS_LEN                (4)
#define WLAN_SET_SCAN_PARAMS_LEN                (100)
#define WLAN_GET_SCAN_RESULTS_PARAMS_LEN        (4)
#define WLAN_ADD_PROFILE_NOSEC_PARAM_LEN        (24)            
#define WLAN_ADD_PROFILE_WEP_PARAM_LEN          (36)
#define WLAN_ADD_PROFILE_WPA_PARAM_LEN          (44)
#define WLAN_CONNECT_PARAM_LEN                  (29)
#define WLAN_SMART_CONFIG_START_PARAMS_LEN      (4)


/**
* Send HCI_CMND_SIMPLE_LINK_START to CC3000
* @param  usPatchesAvailableAtHost  flag to indicate if patches available\n
*                                   from host or from EEPROM. Due to the \n
*                                   fact the patches are burn to the EEPROM\n
*                                   using the patch programmer utility, the \n
*                                   patches will be available from the EEPROM\n
*                                   and not from the host.\n
* @return   none
*/
static void SimpleLink_Init_Start(unsigned short usPatchesAvailableAtHost);

/**
* Initialize wlan driver
* @param  sWlanCB   Asynchronous events callback.  \n
*                   0 no event call back.\n
*                 -call back parameters:\n
*                  1) event_type: HCI_EVNT_WLAN_UNSOL_CONNECT connect event,\n
*                    HCI_EVNT_WLAN_UNSOL_DISCONNECT disconnect event,\n
*                    HCI_EVNT_WLAN_ASYNC_SIMPLE_CONFIG_DONE config done,\n
*                    HCI_EVNT_WLAN_UNSOL_DHCP dhcp report, \n
*                    HCI_EVNT_WLAN_ASYNC_PING_REPORT ping report OR \n
*                    HCI_EVNT_WLAN_KEEPALIVE keepalive.\n
*                  2) data: pointer to extra data that received by the event\n
*                    (NULL no data).\n
*                  3) length: data length.\n
*                 -Events with extra data:\n
*                    HCI_EVNT_WLAN_UNSOL_DHCP: 4 bytes IP, 4 bytes Mask,\n 
*                    4 bytes default gateway, 4 bytes DHCP server and 4 bytes\n
*                    for DNS server.\n
*                    HCI_EVNT_WLAN_ASYNC_PING_REPORT: 4 bytes Packets sent,\n 
*                    4 bytes Packets received, 4 bytes Min round time, \n
*                    4 bytes Max round time and 4 bytes for Avg round time.\n
* @param    sFWPatches              0 no patch or pointer to FW patches 
* @param    sDriverPatches          0 no patch or pointer to driver patches
* @param    sBootLoaderPatches      0 no patch or pointer to bootloader patches
* @param    sReadWlanInterruptPin   init callback. the callback read wlan interrupt status.
* @param    sWlanInterruptEnable    init callback. the callback enable wlan interrupt.
* @param    sWlanInterruptDisable   init callback. the callback disable wlan interrupt.
* @param    sWriteWlanPin           init callback. the callback write value to device pin.  
* @return   none
* @sa       wlan_set_event_mask , wlan_start , wlan_stop 
* @warning  This function must be called before ANY other wlan driver function
*/
void wlan_init(tWlanCB               sWlanCB,
               tFWPatches            sFWPatches,
               tDriverPatches        sDriverPatches,
               tBootLoaderPatches    sBootLoaderPatches,
               tWlanReadInteruptPin  sReadWlanInterruptPin,
               tWlanInterruptEnable  sWlanInterruptEnable,
               tWlanInterruptDisable sWlanInterruptDisable,
               tWriteWlanPin         sWriteWlanPin);

/**
* Trigger Received event/data processing - called from the SPI library to receive the data
* @param         pvBuffer - pointer to the received data buffer\n
*                The function triggers Received event/data processing\n
* @param         Pointer to the received data
* @return        none
*/
void SpiReceiveHandler(void *pvBuffer);


/**
* Start WLAN device.
* This function asserts the enable pin of \n
* the device (WLAN_EN), starting the HW initialization process.\n
* The function blocked until device Initialization is completed.\n
* Function also configure patches (FW, driver or bootloader) \n
* and calls appropriate device callbacks.\n
* @param   usPatchesAvailableAtHost - flag to indicate if patches available\n
*                                     from host or from EEPROM. Due to the\n 
*                                     fact the patches are burn to the EEPROM\n
*                                     using the patch programmer utility, the \n
*                                     patches will be available from the EEPROM\n
*                                     and not from the host.\n
* @return        none
* @Note          Prior calling the function wlan_init shall be called.
* @Warning       This function must be called after wlan_init and before any other wlan API
* @sa            wlan_init , wlan_stop
*/
extern void wlan_start(unsigned short usPatchesAvailableAtHost);

/**
* Stop WLAN device by putting it into reset state.
* @param         none
* @return        none
* @sa            wlan_start
*/
extern void wlan_stop(void);

/**
* Connect to AP
* @param    sec_type   security options:\n
*              WLAN_SEC_UNSEC, \n
*              WLAN_SEC_WEP (ASCII support only),\n
*              WLAN_SEC_WPA or WLAN_SEC_WPA2\n
* @param    ssid       up to 32 bytes and is ASCII SSID of the AP
* @param    ssid_len   length of the SSID
* @param    bssid      6 bytes specified the AP bssid
* @param    key        up to 16 bytes specified the AP security key
* @param    key_len    key length 
* @return   On success, zero is returned. On error, negative is returned.\n 
*           Note that even though a zero is returned on success to trigger\n
*           connection operation, it does not mean that CCC3000 is already\n
*           connected. An asynchronous "Connected" event is generated when \n
*           actual association process finishes and CC3000 is connected to\n
*           the AP. If DHCP is set, An asynchronous "DHCP" event is \n
*           generated when DHCP process is finish.\n
* @warning  Please Note that when connection to AP configured with security\n
*           type WEP, please confirm that the key is set as ASCII and not as HEX.\n
* @sa       wlan_disconnect
*/
#ifndef CC3000_TINY_DRIVER
extern long wlan_connect(unsigned long ulSecType,
                         char *ssid,
                         long ssid_len,
                         unsigned char *bssid,
                         unsigned char *key,
                         long key_len);
#else
extern long wlan_connect(char *ssid, long ssid_len);

#endif

/**
* Disconnect connection from AP. 
* @return    0 disconnected done, other CC3000 already disconnected            
* @sa        wlan_connect
*/
extern long wlan_disconnect(void);

/**
* Add profile.
* When auto start is enabled, the device connects to\n
* station from the profiles table. Up to 7 profiles are supported. \n
* If several profiles configured the device choose the highest \n
* priority profile, within each priority group, device will choose \n
* profile based on security policy, signal strength, etc \n
* parameters. All the profiles are stored in CC3000 NVMEM.\n
*
* @param    ulSecType  WLAN_SEC_UNSEC,WLAN_SEC_WEP,WLAN_SEC_WPA,WLAN_SEC_WPA2
* @param    ucSsid    ssid  SSID up to 32 bytes
* @param    ulSsidLen ssid length
* @param    ucBssid   bssid  6 bytes
* @param    ulPriority ulPriority profile priority. Lowest priority:0.
* @param    ulPairwiseCipher_Or_TxKeyLen  key length for WEP security
* @param    ulGroupCipher_TxKeyIndex  key index
* @param    ulKeyMgmt        KEY management 
* @param    ucPf_OrKey       security key
* @param    ulPassPhraseLen  security key length for WPA\WPA2
* @return    On success, zero is returned. On error, -1 is returned        
* @sa        wlan_ioctl_del_profile 
*/
extern long wlan_add_profile(unsigned long ulSecType,
                             unsigned char* ucSsid,
                             unsigned long ulSsidLen, 
                             unsigned char *ucBssid,
                             unsigned long ulPriority,
                             unsigned long ulPairwiseCipher_Or_Key,
                             unsigned long ulGroupCipher_TxKeyLen,
                             unsigned long ulKeyMgmt,
                             unsigned char* ucPf_OrKey,
                             unsigned long ulPassPhraseLen);

/**
* Delete WLAN profile.
* @param    index   number of profile to delete
* @return   On success, zero is returned. On error, -1 is returned        
* @Note     In order to delete all stored profile, set index to 255.
* @sa       wlan_add_profile 
*/
extern long wlan_ioctl_del_profile(unsigned long ulIndex);

/**
* Mask event according to bit mask.
* In case that event is masked (1), the device will not send the masked event to host. 
* @param    mask   mask option:\n
*                  HCI_EVNT_WLAN_UNSOL_CONNECT connect event\n
*                  HCI_EVNT_WLAN_UNSOL_DISCONNECT disconnect event\n
*                  HCI_EVNT_WLAN_ASYNC_SIMPLE_CONFIG_DONE  smart config done\n
*                  HCI_EVNT_WLAN_UNSOL_INIT init done\n
*                  HCI_EVNT_WLAN_UNSOL_DHCP dhcp event report\n
*                  HCI_EVNT_WLAN_ASYNC_PING_REPORT ping report\n
*                  HCI_EVNT_WLAN_KEEPALIVE keepalive\n
*                  HCI_EVNT_WLAN_TX_COMPLETE - disable information on end of transmission\n
*           Saved: no.
* @return   On success, zero is returned. On error, -1 is returned        
*/
extern long wlan_set_event_mask(unsigned long ulMask);

/**
* Get wlan status: disconnected, scanning, connecting or connected
* @param none 
* @return    WLAN_STATUS_DISCONNECTED, WLAN_STATUS_SCANING,\n 
*            STATUS_CONNECTING or WLAN_STATUS_CONNECTED   \n   
*/
extern long wlan_ioctl_statusget(void);


/**
* Set connection policy.
* When auto is enabled, the device tries to connect according the following policy:\n
* 1) If fast connect is enabled and last connection is valid, \n
*    the device will try to connect to it without the scanning \n
*    procedure (fast). The last connection will be marked as\n
*    invalid, due to adding/removing profile. \n
* 2) If profile exists, the device will try to connect it \n
*    (Up to seven profiles).\n
* 3) If fast and profiles are not found, and open mode is\n
*    enabled, the device will try to connect to any AP.\n
* Note that the policy settings are stored in the CC3000 NVMEM.\n
* @param    should_connect_to_open_ap  enable(1), disable(0) connect to any \n
*           available AP. This parameter corresponds to the configuration of \n
*           item # 3 in the brief description.\n
* @param    should_use_fast_connect enable(1), disable(0). if enabled, tries \n
*           to connect to the last connected AP. This parameter corresponds \n
*           to the configuration of item # 1 in the brief description.\n
* @param    auto_start enable(1), disable(0) auto connect \n
*           after reset and periodically reconnect if needed. This \n
*           configuration configures option 2 in the above description.\n
* @return   On success, zero is returned.\n
*           On error, -1 is returned  \n      
* @sa       wlan_add_profile , wlan_ioctl_del_profile 
*/
extern long wlan_ioctl_set_connection_policy(unsigned long should_connect_to_open_ap,
                                             unsigned long should_use_fast_connect,
                                             unsigned long ulUseProfiles);

/**
* Get entry from scan result table.
* The scan results are returned one by one, and each entry represents a single AP found in the area.\n
* The following is a format of the scan result: \n
* - 4 Bytes: number of networks found\n
* - 4 Bytes: The status of the scan: 0 - aged results, 1 - results valid, 2 - no results\n
* - 42 bytes: Result entry, where the bytes are arranged as  follows:\n
*             - 1 bit      isValid      - is result valid or not\n
*             - 7 bits     rssi         - RSSI value     \n
*             - 2 bits     securityMode - security mode of the AP: 0 - Open, 1 - WEP, 2 WPA, 3 WPA2\n
*             - 6 bits     SSID name length\n
*             - 2 bytes    the time at which the entry has entered into scans result table\n
*             - 32 bytes   SSID name\n
*             - 6 bytes    BSSID \n
* @param[in]    scan_timeout   parameter not supported
* @param[out]   ucResults  scan result (_wlan_full_scan_results_args_t)
* @return       On success, zero is returned. On error, -1 is returned        
* @Note      scan_timeout, is not supported on this version.
* @sa        wlan_ioctl_set_scan_params 
*/
extern long wlan_ioctl_get_scan_results(unsigned long ulScanTimeout,
                                        unsigned char *ucResults);

/**
* start and stop scan procedure. Set scan parameters. 
* @param    uiEnable             start/stop application scan: \n
*                                1 = start scan with default interval value of 10 min. \n
*                                To set a different scan interval value, apply the value in milliseconds.\n
*                                minimum = 1 second. 0=stop.\n
*                                Wlan reset (wlan_stop() wlan_start()) is needed when changing scan interval value.\n
*                                Saved: No\n
* @param   uiMinDwellTime        minimum dwell time value to be used for each channel, in milliseconds.\n
*                                Saved: yes\n
*                                Recommended Value: 100\n
*                                Default: 20\n
* @param   uiMaxDwellTime        maximum dwell time value to be used for each channel, in milliseconds.\n
*                                Saved: yes\n
*                                Recommended Value: 100\n
*                                Default: 30\n
* @param   uiNumOfProbeRequests  max probe request between dwell time. \n
*                                Saved: yes.\n
*                                Recommended Value: 5\n
*                                Default:2\n
* @param   uiChannelMask         bitwise, up to 13 channels (0x1fff). \n
*                                Saved: yes.\n
*                                Default: 0x7ff\n
* @param   uiRSSIThreshold       RSSI threshold.\n
*                                Saved: yes\n
*                                Default: -80\n
* @param   uiSNRThreshold        NSR threshold.\n
*                                Saved: yes\n
*                                Default: 0\n
* @param   uiDefaultTxPower      probe Tx power.\n
*                                Saved: yes\n
*                                Default: 205\n
* @param   aiIntervalList        pointer to array with 16 entries (16 channels) \n
*                                Each entry (unsigned long) holds timeout between periodic scan \n
*                                and connection scan - in milliseconds.\n
*                                Saved: yes.\n
*                                Default 2000ms.\n
* @return  On success, zero is returned. On error, -1 is returned        
* @Note    uiDefaultTxPower, is not supported on this version.
* @sa      wlan_ioctl_get_scan_results 
*/
extern long wlan_ioctl_set_scan_params(unsigned long uiEnable,
                                       unsigned long uiMinDwellTime,
                                       unsigned long uiMaxDwellTime,
                                       unsigned long uiNumOfProbeRequests,
                                       unsigned long uiChannelMask,
                                       long iRSSIThreshold,
                                       unsigned long uiSNRThreshold,
                                       unsigned long uiDefaultTxPower, 
                                       unsigned long *aiIntervalList);

                                           
/**
* Start to acquire device profile.
* The device will acquires its own profile if a profile message is found.\n
* The acquired AP information is stored in CC3000 EEPROM only when AES128 encryption is used.\n
* When AES128 encryption is not used, a profile is internally created by the device.\n
* @param    algoEncryptedFlag indicates whether the information is encrypted
* @return   On success, zero is returned. On error, -1 is returned        
* @Note     An asynchronous event - Smart Config Done will be generated as soon\n
*           as the process finishes successfully.\n
* @sa       wlan_smart_config_set_prefix , wlan_smart_config_stop
*/
extern long wlan_smart_config_start(unsigned long algoEncryptedFlag);

/**
* Stop the acquire profile procedure 
* @param    algoEncryptedFlag indicates whether the information is encrypted
* @return   On success, zero is returned. On error, -1 is returned        
* @sa      wlan_smart_config_start , wlan_smart_config_set_prefix
*/
extern long wlan_smart_config_stop(void);

/**
* Configure station ssid prefix.
* The prefix is used internally in CC3000. It should always be TTT.\n
* @param   newPrefix  3 bytes identify the SSID prefix for the Smart Config. 
* @return  On success, zero is returned. On error, -1 is returned        
* @Note    The prefix is stored in CC3000 NVMEM
* @sa      wlan_smart_config_start , wlan_smart_config_stop
*/
extern long wlan_smart_config_set_prefix(char* cNewPrefix);

/**
* Process the acquired data and store it as a profile.
* The acquired AP information is stored in CC3000 EEPROM encrypted.
* The encrypted data is decrypted and stored as a profile.\n
* behavior is as defined by connection policy.\n
* @param    none 
* @return   On success, zero is returned. On error, -1 is returned        
*/
extern long wlan_smart_config_process(void);

#ifdef    __cplusplus
}
#endif // __cplusplus

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************

#endif    // __WLAN_H__
