/*****************************************************************************
*
*  socket.h  - CC3000 Host Driver Implementation.
*  Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the   
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*****************************************************************************/
#ifndef __SOCKET_H__
#define __SOCKET_H__

#include "evnt_handler.h"

//*****************************************************************************
//
//! \addtogroup socket
//! @{
//
//*****************************************************************************

/** CC3000 Host driver - Socket API
*
*/

#ifdef  __cplusplus
extern "C" {
#endif

// BEWARE! When using send-non-blocking, always check the return value.
//         If -2, nothing was actually sent and you have to retry if you still want the data. 
//#define SEND_NON_BLOCKING

//Enable this flag if and only if you must comply with BSD socket 
//close() function
#ifdef _API_USE_BSD_CLOSE
    #define close(sd) closesocket(sd)
#endif

//Enable this flag if and only if you must comply with BSD socket read() and 
//write() functions
#ifdef _API_USE_BSD_READ_WRITE
    #define read(sd, buf, len, flags) recv(sd, buf, len, flags)
    #define write(sd, buf, len, flags) send(sd, buf, len, flags)
#endif

#define SOCKET_OPEN_PARAMS_LEN                 (12)
#define SOCKET_CLOSE_PARAMS_LEN                (4)
#define SOCKET_ACCEPT_PARAMS_LEN               (4)
#define SOCKET_BIND_PARAMS_LEN                 (20)
#define SOCKET_LISTEN_PARAMS_LEN               (8)
#define SOCKET_GET_HOST_BY_NAME_PARAMS_LEN     (9)
#define SOCKET_CONNECT_PARAMS_LEN              (20)
#define SOCKET_SELECT_PARAMS_LEN               (44)
#define SOCKET_SET_SOCK_OPT_PARAMS_LEN         (20)
#define SOCKET_GET_SOCK_OPT_PARAMS_LEN         (12)
#define SOCKET_RECV_FROM_PARAMS_LEN            (12)
#define SOCKET_SENDTO_PARAMS_LEN               (24)
#define SOCKET_MDNS_ADVERTISE_PARAMS_LEN       (12)
#define SOCKET_MAX_FREE_BUFFERS                (6)

//#define NULL 0

// The legnth of arguments for the SEND command: sd + buff_offset + len + flags, 
// while size of each parameter is 32 bit - so the total length is 16 bytes;

#define HCI_CMND_SEND_ARG_LENGTH                    (16)
#define SELECT_TIMEOUT_MIN_MICRO_SECONDS            5000
#define HEADERS_SIZE_DATA                           (SPI_HEADER_SIZE + 5)
#define SIMPLE_LINK_HCI_CMND_TRANSPORT_HEADER_SIZE  (SPI_HEADER_SIZE + SIMPLE_LINK_HCI_CMND_HEADER_SIZE)
#define MDNS_DEVICE_SERVICE_MAX_LENGTH              (32)


#define HOSTNAME_MAX_LENGTH (230)  // 230 bytes + header shouldn't exceed 8 bit value

//--------- Address Families --------

#define  AF_INET                2
#define  AF_INET6               23

//------------ Socket Types ------------

#define  SOCK_STREAM            1
#define  SOCK_DGRAM             2
#define  SOCK_RAW               3           // Raw sockets allow new IPv4 protocols to be implemented in user space. A raw socket receives or sends the raw datagram not including link level headers
#define  SOCK_RDM               4
#define  SOCK_SEQPACKET         5

//----------- Socket Protocol ----------

#define IPPROTO_IP              0           // dummy for IP
#define IPPROTO_ICMP            1           // control message protocol
#define IPPROTO_IPV4            IPPROTO_IP  // IP inside IP
#define IPPROTO_TCP             6           // tcp
#define IPPROTO_UDP             17          // user datagram protocol
#define IPPROTO_IPV6            41          // IPv6 in IPv6
#define IPPROTO_NONE            59          // No next header
#define IPPROTO_RAW             255         // raw IP packet
#define IPPROTO_MAX             256

//----------- Socket retunr codes  -----------

#define SOC_ERROR                (-1)        // error 
#define SOC_IN_PROGRESS          (-2)        // socket in progress

//----------- Socket Options -----------
#define  SOL_SOCKET             0xffff       //  socket level
#define  SOCKOPT_RECV_TIMEOUT   1            //  optname to configure recv and recvfromtimeout
#define  SOCKOPT_NONBLOCK       2            // accept non block mode set SOCK_ON or SOCK_OFF (default block mode )
#define  SOCK_ON                0            // socket non-blocking mode    is enabled        
#define  SOCK_OFF               1            // socket blocking mode is enabled

#define  TCP_NODELAY            0x0001
#define  TCP_BSDURGENT          0x7000

#define  MAX_PACKET_SIZE        1500
#define  MAX_LISTEN_QUEUE       4

#define  IOCTL_SOCKET_EVENTMASK

#define ENOBUFS                 55          // No buffer space available

#define __FD_SETSIZE            32

#define  ASIC_ADDR_LEN          8
    
#define NO_QUERY_RECIVED        -3
    
    
typedef struct _in_addr_t
{
    unsigned long s_addr;                   // load with inet_aton()
} in_addr;

/*typedef struct _sockaddr_t
{
    unsigned short int  sa_family;
    unsigned char       sa_data[14];
} sockaddr;*/

typedef struct _sockaddr_in_t
{
    short            sin_family;            // e.g. AF_INET
    unsigned short   sin_port;              // e.g. htons(3490)
    in_addr          sin_addr;              // see struct in_addr, below
    char             sin_zero[8];           // zero this if you want to
} sockaddr_in;

typedef unsigned long socklen_t;

// The fd_set member is required to be an array of longs.
typedef long int __fd_mask;

// It's easier to assume 8-bit bytes than to get CHAR_BIT.
#define __NFDBITS               (8 * sizeof (__fd_mask))
#define __FDELT(d)              ((d) / __NFDBITS)
#define __FDMASK(d)             ((__fd_mask) 1 << ((d) % __NFDBITS))

// fd_set for select and pselect.
typedef struct
{
    __fd_mask fds_bits[__FD_SETSIZE / __NFDBITS];
#define __FDS_BITS(set)        ((set)->fds_bits)
} fd_set;

// We don't use `memset' because this would require a prototype and
//   the array isn't too big.
#define __FD_ZERO(set)                               \
  do {                                                \
    unsigned int __i;                                 \
    fd_set *__arr = (set);                            \
    for (__i = 0; __i < sizeof (fd_set) / sizeof (__fd_mask); ++__i) \
      __FDS_BITS (__arr)[__i] = 0;                    \
  } while (0)
#define __FD_SET(d, set)       (__FDS_BITS (set)[__FDELT (d)] |= __FDMASK (d))
#define __FD_CLR(d, set)       (__FDS_BITS (set)[__FDELT (d)] &= ~__FDMASK (d))
#define __FD_ISSET(d, set)     (__FDS_BITS (set)[__FDELT (d)] & __FDMASK (d))

// Access macros for 'fd_set'.
#define FD_SET(fd, fdsetp)      __FD_SET (fd, fdsetp)
#define FD_CLR(fd, fdsetp)      __FD_CLR (fd, fdsetp)
#define FD_ISSET(fd, fdsetp)    __FD_ISSET (fd, fdsetp)
#define FD_ZERO(fdsetp)         __FD_ZERO (fdsetp)

//Use in case of Big Endian only
  
#define htonl(A)    ((((unsigned long)(A) & 0xff000000) >> 24) | \
                     (((unsigned long)(A) & 0x00ff0000) >> 8) | \
                     (((unsigned long)(A) & 0x0000ff00) << 8) | \
                     (((unsigned long)(A) & 0x000000ff) << 24))

#define ntohl                   htonl

//Use in case of Big Endian only
#define htons(A)     ((((unsigned long)(A) & 0xff00) >> 8) | \
                      (((unsigned long)(A) & 0x00ff) << 8))


#define ntohs                   htons

// mDNS port - 5353    mDNS multicast address - 224.0.0.251 
#define SET_mDNS_ADD(sockaddr) sockaddr.sa_data[0] = 0x14; \
                               sockaddr.sa_data[1] = 0xe9; \
                               sockaddr.sa_data[2] = 0xe0; \
                               sockaddr.sa_data[3] = 0x0;  \
                               sockaddr.sa_data[4] = 0x0;  \
                               sockaddr.sa_data[5] = 0xfb; 


//*****************************************************************************
//
// Prototypes for the APIs.
//
//*****************************************************************************

/**
* HostFlowControlConsumeBuff.
* if SEND_NON_BLOCKING is not defined - block until a free buffer is available,\n
* otherwise return the status of the available buffers.\n
* @param  sd  socket descriptor
* @return  0 in case there are buffers available, \n
*         -1 in case of bad socket\n
*         -2 if there are no free buffers present (only when SEND_NON_BLOCKING is enabled)\n
*/
int HostFlowControlConsumeBuff(int sd);

/**
* create an endpoint for communication.
* The socket function creates a socket that is bound to a specific transport service provider.\n
* This function is called by the application layer to obtain a socket handle.\n
* @param   domain    selects the protocol family which will be used for\n 
*                    communication. On this version only AF_INET is supported\n
* @param   type      specifies the communication semantics. On this version\n 
*                    only SOCK_STREAM, SOCK_DGRAM, SOCK_RAW are supported\n
* @param   protocol  specifies a particular protocol to be used with the\n 
*                    socket IPPROTO_TCP, IPPROTO_UDP or IPPROTO_RAW are supported.\n
* @return  On success, socket handle that is used for consequent socket operations\n 
*          On error, -1 is returned.\n
*/
extern int socket(long domain, long type, long protocol);

/**
* The socket function closes a created socket.
* @param   sd    socket handle.
* @return  On success, zero is returned. On error, -1 is returned.
*/
extern long closesocket(long sd);

/**
* accept a connection on a socket.
* This function is used with connection-based socket types\n 
* (SOCK_STREAM). It extracts the first connection request on the\n 
* queue of pending connections, creates a new connected socket, and\n
* returns a new file descriptor referring to that socket.\n
* The newly created socket is not in the listening state.\n 
* The original socket sd is unaffected by this call.\n 
* The argument sd is a socket that has been created with socket(),\n
* bound to a local address with bind(), and is  listening for \n
* connections after a listen(). The argument addr is a pointer \n
* to a sockaddr structure. This structure is filled in with the \n
* address of the peer socket, as known to the communications layer.\n
* The exact format of the address returned addr is determined by the \n
* socket's address family. The addrlen argument is a value-result\n
* argument: it should initially contain the size of the structure\n
* pointed to by addr, on return it will contain the actual\n 
* length (in bytes) of the address returned.\n
* @param[in]   sd      socket descriptor (handle)\n              
* @param[out]  addr    the argument addr is a pointer to a sockaddr structure\n
*                      This structure is filled in with the address of the \n 
*                      peer socket, as known to the communications layer.  \n      
*                      determined. The exact format of the address returned \n            
*                      addr is by the socket's address sockaddr. \n
*                      On this version only AF_INET is supported.\n
*                      This argument returns in network order.\n
* @param[out] addrlen  the addrlen argument is a value-result argument: \n
*                      it should initially contain the size of the structure\n
*                      pointed to by addr.\n
* @return  For socket in blocking mode:\n
*           - On success, socket handle. on failure negative\n
*          For socket in non-blocking mode:\n
*           - On connection establishment, socket handle\n
*           - On connection pending, SOC_IN_PROGRESS (-2)\n
*           - On failure, SOC_ERROR    (-1)\n
* @sa     socket ; bind ; listen
*/
extern long accept(long sd, sockaddr *addr, socklen_t *addrlen);

/**
* assign a name to a socket.
* This function gives the socket the local address addr.\n
* addr is addrlen bytes long. Traditionally, this is called when a \n
* socket is created with socket, it exists in a name space (address \n
* family) but has no name assigned.\n
* It is necessary to assign a local address before a SOCK_STREAM\n
* socket may receive connections.\n
* @param[in]   sd      socket descriptor (handle)              
* @param[out]  addr    specifies the destination address. On this version\n 
*                      only AF_INET is supported.\n
* @param[out] addrlen  contains the size of the structure pointed to by addr.\n
* @return      On success, zero is returned.\n
*              On error, -1 is returned.\n
* @sa          socket ; accept ; listen
*/
extern long bind(long sd, const sockaddr *addr, long addrlen);

/**
* listen for connections on a socket.
* The willingness to accept incoming connections and a queue\n
* limit for incoming connections are specified with listen(),\n
* and then the connections are accepted with accept.\n
* The listen() call applies only to sockets of type SOCK_STREAM\n
* The backlog parameter defines the maximum length the queue of\n
* pending connections may grow to. \n
* @param[in]  sd       socket descriptor (handle)              
* @param[in]  backlog  specifies the listen queue depth. On this version\n
*                      backlog is not supported.\n
* @return     On success, zero is returned.\n
*             On error, -1 is returned.\n
* @sa         socket ; accept ; bind
* @note       On this version, backlog is not supported
*/
extern long listen(long sd, long backlog);

/**
* Get host IP by name.\n
* Obtain the IP Address of machine on network\n
* @param[in]   hostname     host name              
* @param[in]   usNameLen    name length 
* @param[out]  out_ip_addr  This parameter is filled in with host IP address.\n 
*                           In case that host name is not resolved, \n
*                           out_ip_addr is zero.\n     
* @return      On success, positive is returned.\n
*              On error, negative is returned by its name.\n
* @note  On this version, only blocking mode is supported. Also note that\n
*        The function requires DNS server to be configured prior to its usage.\n
*/
#ifndef CC3000_TINY_DRIVER 
extern int gethostbyname(char * hostname, unsigned short usNameLen, unsigned long* out_ip_addr);
#endif


/**
* initiate a connection on a socket.
* Function connects the socket referred to by the socket descriptor\n 
* sd, to the address specified by addr. The addrlen argument \n
* specifies the size of addr. The format of the address in addr is \n
* determined by the address space of the socket. If it is of type \n
* SOCK_DGRAM, this call specifies the peer with which the socket is \n
* to be associated; this address is that to which datagrams are to be\n
* sent, and the only address from which datagrams are to be received. \n 
* If the socket is of type SOCK_STREAM, this call attempts to make a \n
* connection to another socket. The other socket is specified  by \n
* address, which is an address in the communications space of the\n
* socket. Note that the function implements only blocking behavior \n
* thus the caller will be waiting either for the connection \n
* establishment or for the connection establishment failure.\n
* @param[in]   sd       socket descriptor (handle)         
* @param[in]   addr     specifies the destination addr. On this version\n
*                       only AF_INET is supported.\n
* @param[out]  addrlen  contains the size of the structure pointed to by addr    
* @return      On success, zero is returned.\n
               On error, -1 is returned\n
* @sa socket
*/
extern long connect(long sd, const sockaddr *addr, long addrlen);

/**
* Monitor socket activity. 
* Select allow a program to monitor multiple file descriptors,\n
* waiting until one or more of the file descriptors become \n
*"ready" for some class of I/O operation \n
* @param[in]    nfds       the highest-numbered file descriptor in any of the\n
*                          three sets, plus 1.  \n   
* @param[out]   writesds   socket descriptors list for write monitoring\n
* @param[out]   readsds    socket descriptors list for read monitoring\n  
* @param[out]   exceptsds  socket descriptors list for exception monitoring\n
* @param[in]    timeout    is an upper bound on the amount of time elapsed\n
*                          before select() returns. Null means infinity \n
*                          timeout. The minimum timeout is 5 milliseconds,\n
*                         less than 5 milliseconds will be set\n
*                          automatically to 5 milliseconds.\n
* @return    On success, select() returns the number of file descriptors\n
*            contained in the three returned descriptor sets (that is, the\n
*            total number of bits that are set in readfds, writefds,\n
*            exceptfds) which may be zero if the timeout expires before\n
*            anything interesting  happens.\n
*            On error, -1 is returned.\n
*                  *readsds - return the sockets on which Read request will\n
*                             return without delay with valid data.\n
*                  *writesds - return the sockets on which Write request \n
*                                will return without delay.\n
*                  *exceptsds - return the sockets which closed recently.\n
* @Note   If the timeout value set to less than 5ms it will automatically\n
*         change to 5ms to prevent overload of the system\n
* @sa socket
*/
extern int select(long nfds, fd_set *readsds, fd_set *writesds, fd_set *exceptsds, struct timeval *timeout);

/**
* set socket options.
* This function manipulate the options associated with a socket.\n
* Options may exist at multiple protocol levels; they are always\n
* present at the uppermost socket level.\n
* When manipulating socket options the level at which the option \n
* resides and the name of the option must be specified.\n
* To manipulate options at the socket level, level is specified as\n 
* SOL_SOCKET. To manipulate options at any other level the protocol \n
* number of the appropriate protocol controlling the option is \n
* supplied. For example, to indicate that an option is to be \n
* interpreted by the TCP protocol, level should be set to the \n
* protocol number of TCP; \n
* The parameters optval and optlen are used to access optval - \n
* use for setsockopt(). For getsockopt() they identify a buffer\n
* in which the value for the requested option(s) are to \n
* be returned. For getsockopt(), optlen is a value-result \n
* parameter, initially containing the size of the buffer \n
* pointed to by option_value, and modified on return to \n
* indicate the actual size of the value returned. If no option \n
* value is to be supplied or returned, option_value may be NULL.\n
* @param[in]   sd          socket handle
* @param[in]   level       defines the protocol level for this option
* @param[in]   optname     defines the option name to Interrogate
* @param[in]   optval      specifies a value for the option
* @param[in]   optlen      specifies the length of the option value
* @return      On success, zero is returned.\n
               On error, -1 is returned\n
*
* @Note   On this version the following two socket options are enabled:\n
*         The only protocol level supported in this version is SOL_SOCKET (level).\n
*              1. SOCKOPT_RECV_TIMEOUT (optname)\n
*                 SOCKOPT_RECV_TIMEOUT configures recv and recvfrom timeout in milliseconds.\n
*                 In that case optval should be pointer to unsigned long.\n
*              2. SOCKOPT_NONBLOCK (optname). sets the socket non-blocking mode on or off.\n
*                 In that case optval should be SOCK_ON or SOCK_OFF (optval).\n
* @sa getsockopt
*/
#ifndef CC3000_TINY_DRIVER 
extern int setsockopt(long sd, long level, long optname, const void *optval, socklen_t optlen);
#endif

/**
* get socket options.
* This function manipulate the options associated with a socket.\n
* Options may exist at multiple protocol levels; they are always\n
* present at the uppermost socket level.\n
* When manipulating socket options the level at which the option \n
* resides and the name of the option must be specified.  \n
* To manipulate options at the socket level, level is specified as \n
* SOL_SOCKET. To manipulate options at any other level the protocol \n
* number of the appropriate protocol controlling the option is \n
* supplied. For example, to indicate that an option is to be \n
* interpreted by the TCP protocol, level should be set to the \n
* protocol number of TCP; \n
* The parameters optval and optlen are used to access optval -\n 
* use for setsockopt(). For getsockopt() they identify a buffer\n
* in which the value for the requested option(s) are to \n
* be returned. For getsockopt(), optlen is a value-result \n
* parameter, initially containing the size of the buffer \n
* pointed to by option_value, and modified on return to \n
* indicate the actual size of the value returned. If no option \n
* value is to be supplied or returned, option_value may be NULL.\n
* @param[in]   sd          socket handle
* @param[in]   level       defines the protocol level for this option
* @param[in]   optname     defines the option name to Interrogate
* @param[out]  optval      specifies a value for the option
* @param[out]  optlen      specifies the length of the option value
* @return      On success, zero is returned. On error, -1 is returned
*
* @Note   On this version the following two socket options are enabled:\n
*         The only protocol level supported in this version is SOL_SOCKET (level).\n
*              1. SOCKOPT_RECV_TIMEOUT (optname)\n
*                 SOCKOPT_RECV_TIMEOUT configures recv and recvfrom timeout in milliseconds.\n
*                 In that case optval should be pointer to unsigned long.\n
*              2. SOCKOPT_NONBLOCK (optname). sets the socket non-blocking mode on or off.\n
*                 In that case optval should be SOCK_ON or SOCK_OFF (optval).\n
* @sa setsockopt
*/
extern int getsockopt(long sd, long level, long optname, void *optval, socklen_t *optlen);

/**
* Read data from socket (simple_link_recv).
* Return the length of the message on successful completion.\n
* If a message is too long to fit in the supplied buffer, excess bytes may\n
* be discarded depending on the type of socket the message is received from.\n
* @param sd       socket handle
* @param buf      read buffer
* @param len      buffer length
* @param flags    indicates blocking or non-blocking operation
* @param from     pointer to an address structure indicating source address
* @param fromlen  source address structure size
* @return         Return the number of bytes received, or -1 if an error occurred
*/
int simple_link_recv(long sd, void *buf, long len, long flags, sockaddr *from, socklen_t *fromlen, long opcode);

/**
* Receive a message from a connection-mode socket.
* @param[in]  sd     socket handle
* @param[out] buf    Points to the buffer where the message should be stored
* @param[in]  len    Specifies the length in bytes of the buffer pointed to \n
*                    by the buffer argument.\n
* @param[in] flags   Specifies the type of message reception. \n
*                    On this version, this parameter is not supported.\n
* @return         Return the number of bytes received, or -1 if an error occurred
* @sa recvfrom
* @Note On this version, only blocking mode is supported.
*/
extern int recv(long sd, void *buf, long len, long flags);

/**
* read data from socket (recvfrom).
* Receives a message from a connection-mode or connectionless-mode socket.\n
* Note that raw sockets are not supported.\n
* @param[in]  sd       socket handle
* @param[out] buf      Points to the buffer where the message should be stored
* @param[in]  len      Specifies the length in bytes of the buffer pointed to \n
*                      by the buffer argument.\n
* @param[in] flags     Specifies the type of message reception.\n 
*                      On this version, this parameter is not supported.\n
* @param[in] from      pointer to an address structure indicating the source\n
*                      address: sockaddr. On this version only AF_INET is\n
*                      supported.\n
* @param[in] fromlen   source address structure size
* @return              Return the number of bytes received, or -1 if an error occurred
* @sa recv
* @Note On this version, only blocking mode is supported.
*/
extern int recvfrom(long sd, void *buf, long len, long flags, sockaddr *from, socklen_t *fromlen);

/**
* Transmit a message to another socket (simple_link_send).
* @param sd       socket handle
* @param buf      write buffer
* @param len      buffer length
* @param flags    On this version, this parameter is not supported
* @param to       pointer to an address structure indicating destination address
* @param tolen    destination address structure size
* @return         Return the number of bytes transmitted, or -1 if an error\n
*                 occurred, or -2 in case there are no free buffers available\n
*                 (only when SEND_NON_BLOCKING is enabled)\n
*/
int simple_link_send(long sd, const void *buf, long len, long flags, const sockaddr *to, long tolen, long opcode);

/**
* Transmit a message to another socket (send).
* @param sd       socket handle
* @param buf      Points to a buffer containing the message to be sent
* @param len      message size in bytes
* @param flags    On this version, this parameter is not supported
* @return         Return the number of bytes transmitted, or -1 if an\n
*                 error occurred\n
* @Note           On this version, only blocking mode is supported.
* @sa             sendto
*/
extern int send(long sd, const void *buf, long len, long flags);

/**
* Transmit a message to another socket (sendto).
* @param sd       socket handle
* @param buf      Points to a buffer containing the message to be sent
* @param len      message size in bytes
* @param flags    On this version, this parameter is not supported
* @param to       pointer to an address structure indicating the destination\n
*                 address: sockaddr. On this version only AF_INET is\n
*                 supported.\n
* @param tolen    destination address structure size
* @return         Return the number of bytes transmitted, or -1 if an error occurred
* @Note           On this version, only blocking mode is supported.
* @sa             send
*/
extern int sendto(long sd, const void *buf, long len, long flags, const sockaddr *to, socklen_t tolen);

/**
* Set CC3000 in mDNS advertiser mode in order to advertise itself.
* @param[in] mdnsEnabled         flag to enable/disable the mDNS feature
* @param[in] deviceServiceName   Service name as part of the published\n
*                                canonical domain name\n
* @param[in] deviceServiceNameLength   Length of the service name
* @return   On success, zero is returned,\n
*           return SOC_ERROR if socket was not opened successfully, or if an error occurred.\n
*/
extern int mdnsAdvertiser(unsigned short mdnsEnabled, char * deviceServiceName, unsigned short deviceServiceNameLength);


#ifdef  __cplusplus
}
#endif // __cplusplus

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************

#endif // __SOCKET_H__





