/*****************************************************************************
*
*  cc3000_common.h  - CC3000 Host Driver Implementation.
*  Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the   
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*****************************************************************************/
#ifndef __COMMON_H__
#define __COMMON_H__

//#include "GlobalAssigns.h"
#include "mbed.h"
#include <errno.h>

//*****************************************************************************
//
//! \addtogroup cc3000_common
//! @{
//
//*****************************************************************************
/** CC3000 Host driver - common
*
*/

#ifdef  __cplusplus
extern "C" {
#endif

//*****************************************************************************
//                  ERROR CODES
//*****************************************************************************
#define ESUCCESS        0
#define EFAIL          -1
#define EERROR          EFAIL


//*****************************************************************************
//              PIN CONTROLS & COMPILE CONTROLS
//*****************************************************************************
// Compiler control
#define CC3000_UNENCRYPTED_SMART_CONFIG   // No encryption
//#define CC3000_TINY_DRIVER                // Driver for small memory model CPUs

//Interrupt controls
#define NVIC_ALL_IRQ        NVIC_set_all_irq_priorities(3);         // Set ALL interrupt priorities to level 3// Set ALL interrupt priorities to level 3
#define NVIC_SPI_IRQ        NVIC_SetPriority(SPI0_IRQn, 0x0);       // Wi-Fi SPI interrupt must be higher priority than SysTick
#define NVIC_PORT_IRQ       NVIC_SetPriority(PORTA_IRQn, 0x1);
#define NVIC_SYSTICK_IRQ    NVIC_SetPriority(SysTick_IRQn, 0x2);    // SysTick set to lower priority than Wi-Fi SPI bus interrupt
//#define NVIC_ADC_IRQ        NVIC_SetPriority(ADC0_IRQn, 0x3);       // ADC is the lowest of all

// Wlan controls
#define WLAN_ISF_PCR        PORTA->PCR[16]
#define WLAN_ISF_ISFR       PORTA->ISFR
#define WLAN_ISF_MASK       (1<<16)

#define WLAN_ASSERT_CS      wlan_cs = 0;   //CS : active low
#define WLAN_DEASSERT_CS    wlan_cs = 1;

#define WLAN_ASSERT_EN      wlan_en = 1;   //EN : active high
#define WLAN_DEASSERT_EN    wlan_en = 0;

#define WLAN_READ_IRQ       wlan_int

//#define WLAN_ENABLE_IRQ     NVIC_EnableIRQ(PORTA_IRQn);
//#define WLAN_DISABLE_IRQ    NVIC_DisableIRQ(PORTA_IRQn);
#define WLAN_ENABLE_IRQ     wlan_int.fall(&WLAN_IRQHandler);
#define WLAN_DISABLE_IRQ    wlan_int.fall(NULL);

#define WLAN_IRQ_PIN_CREATE         InterruptIn wlan_int (PTA16);
#define WLAN_EN_PIN_CREATE          DigitalOut  wlan_en  (PTA13);
#define WLAN_CS_PIN_CREATE          DigitalOut  wlan_cs  (PTD0);
#define WLAN_SPI_PORT_CREATE        SPI wlan(PTD2, PTD3, PTC5); // mosi, miso, sclk

#define WLAN_SPI_PORT_INIT          wlan.format(8,1);
#define WLAN_SPI_SET_FREQ           wlan.frequency(12000000);
#define WLAN_SPI_SET_IRQ_HANDLER    wlan_int.fall(&WLAN_IRQHandler);

#define WLAN_SPI_WRITE              wlan.write(*data++);
#define WLAN_SPI_READ               wlan.write(0x03);          // !! DO NOT MODIFY the 0x03 parameter (CC3000 will not respond).

extern DigitalOut  wlan_en;
extern DigitalOut  wlan_cs;
extern InterruptIn wlan_int;
extern SPI wlan;
extern void CC3000_UsynchCallback(long lEventType, char * data, unsigned char length);


/** Set basic controls and parameters for the HostDriver.
* @param  none
* @return none
*/
void Init_HostDriver(void);

/** Return a pointer to the driver patch.
* Since there is no patch, 0 is returned\n
* (the patches are taken from the EEPROM and not from the host)\n
* @param  pointer to the length
* @return NULL
*/ 
char *sendDriverPatch(unsigned long *Length);

/** Return a pointer to the bootloader patch.
* Since there is no patch, 0 is returned \n
* (the patches are taken from the EEPROM and not from the host)\n
* @param  pointer to the length
* @return NULL
*/
char *sendBootLoaderPatch(unsigned long *Length);

/** Return a pointer to the firmware patch.
* Since there is no patch, 0 is returned\n
* (the patches are taken from the EEPROM and not from the host)\n
* @param  pointer to the length
* @return NULL
*/
char *sendWLFWPatch(unsigned long *Length);

/** Read Wlan Interrupt pin.
* @param  none
* @return wlan interrup pin level
*/
long ReadWlanInterruptPin(void);

/** Enable waln IrQ pin.
* @param  none
* @return none
*/
void WlanInterruptEnable(void);

/** Disable waln IrQ pin.
* @param  none
* @return none
*/
void WlanInterruptDisable(void);

/** WriteWlanPin.
* @param  val (1: enable - 0: disable)
* @return none
*/
void WriteWlanPin( unsigned char val );


//*****************************************************************************
//                  COMMON DEFINES
//*****************************************************************************
#define ERROR_SOCKET_INACTIVE   -57 

#define HCI_CC_PAYLOAD_LEN      5

#define WLAN_ENABLE            (1)   
#define WLAN_DISABLE           (0)

#define MAC_ADDR_LEN           (6)



/*Defines for minimal and maximal RX buffer size. This size includes the spi 
  header and hci header.
  maximal buffer size: MTU + HCI header + SPI header + sendto() args size
  minimum buffer size: HCI header + SPI header + max args size

  This buffer is used for receiving events and data.
  The packet can not be longer than MTU size and CC3000 does not support 
  fragmentation. Note that the same buffer is used for reception of the data 
  and events from CC3000. That is why the minimum is defined. 
  The calculation for the actual size of buffer for reception is:
  Given the maximal data size MAX_DATA that is expected to be received by
  application, the required buffer Using recv() or recvfrom():
 
    max(CC3000_MINIMAL_RX_SIZE, MAX_DATA + HEADERS_SIZE_DATA + fromlen + ucArgsize + 1)
 
  Using gethostbyname() with minimal buffer size will limit the host name returned to 99 bytes.
  The 1 is used for the overrun detection 
*/

#define CC3000_MINIMAL_RX_SIZE      (118 + 1)
#define CC3000_MAXIMAL_RX_SIZE      (1519 + 1)

/*Defines for minimal and maximal TX buffer size.
  This buffer is used for sending events and data.
  The packet can not be longer than MTU size and CC3000 does not support 
  fragmentation. Note that the same buffer is used for transmission of the data
  and commands. That is why the minimum is defined.
  The calculation for the actual size of buffer for transmission is:
  Given the maximal data size MAX_DATA, the required buffer is:
  Using Sendto():
 
   max(CC3000_MINIMAL_TX_SIZE, MAX_DATA + SPI_HEADER_SIZE
   + SOCKET_SENDTO_PARAMS_LEN + SIMPLE_LINK_HCI_DATA_HEADER_SIZE + 1)
 
  Using Send():
 
   max(CC3000_MINIMAL_TX_SIZE, MAX_DATA + SPI_HEADER_SIZE
   + HCI_CMND_SEND_ARG_LENGTH + SIMPLE_LINK_HCI_DATA_HEADER_SIZE + 1)
 
  The 1 is used for the overrun detection */ 

#define CC3000_MINIMAL_TX_SIZE      (118 + 1) 
#define CC3000_MAXIMAL_TX_SIZE      (1519 + 1)

//TX and RX buffer size - allow to receive and transmit maximum data at lengh 8.
#ifdef CC3000_TINY_DRIVER
#define TINY_CC3000_MAXIMAL_RX_SIZE 44
#define TINY_CC3000_MAXIMAL_TX_SIZE 59
#endif

/*In order to determine your preferred buffer size, 
  change CC3000_MAXIMAL_RX_SIZE and CC3000_MAXIMAL_TX_SIZE to a value between
  the minimal and maximal specified above. 
  Note that the buffers are allocated by SPI.
*/
  
#ifndef CC3000_TINY_DRIVER
  
    #define CC3000_RX_BUFFER_SIZE   (CC3000_MAXIMAL_RX_SIZE)
    #define CC3000_TX_BUFFER_SIZE   (CC3000_MAXIMAL_TX_SIZE)
    #define SP_PORTION_SIZE         512
  
//TINY DRIVER: We use smaller rx and tx buffers in order to minimize RAM consumption
#else
    #define CC3000_RX_BUFFER_SIZE   (TINY_CC3000_MAXIMAL_RX_SIZE)
    #define CC3000_TX_BUFFER_SIZE   (TINY_CC3000_MAXIMAL_TX_SIZE)
    #define SP_PORTION_SIZE         32

#endif  
//*****************************************************************************
//                  Compound Types
//*****************************************************************************
typedef struct timeval timeval;

typedef struct _sockaddr_t
{
    unsigned short int  sa_family;
    unsigned char       sa_data[14];
} sockaddr;

struct timeval 
{
    long tv_sec;       /* seconds */
    long tv_usec;      /* microseconds */
};

typedef char *(*tFWPatches)(unsigned long *usLength);
typedef char *(*tDriverPatches)(unsigned long *usLength);
typedef char *(*tBootLoaderPatches)(unsigned long *usLength);
typedef void (*tWlanCB)(long event_type, char * data, unsigned char length );
typedef long (*tWlanReadInteruptPin)(void);
typedef void (*tWlanInterruptEnable)(void);
typedef void (*tWlanInterruptDisable)(void);
typedef void (*tWriteWlanPin)(unsigned char val);

typedef struct
{
    unsigned short        usRxEventOpcode;
    unsigned short        usEventOrDataReceived;
    unsigned char         *pucReceivedData;
    unsigned char         *pucTxCommandBuffer;
    tFWPatches            sFWPatches;
    tDriverPatches        sDriverPatches;
    tBootLoaderPatches    sBootLoaderPatches;
    tWlanCB               sWlanCB;
    tWlanReadInteruptPin  ReadWlanInterruptPin;
    tWlanInterruptEnable  WlanInterruptEnable;
    tWlanInterruptDisable WlanInterruptDisable;
    tWriteWlanPin         WriteWlanPin;
    signed long           slTransmitDataError;
    unsigned short        usNumberOfFreeBuffers;
    unsigned short        usSlBufferLength;
    unsigned short        usBufferSize;
    unsigned short        usRxDataPending;
    unsigned long         NumberOfSentPackets;
    unsigned long         NumberOfReleasedPackets;
    unsigned char         InformHostOnTxComplete;
}sSimplLinkInformation;

extern volatile sSimplLinkInformation tSLInformation;


//*****************************************************************************
// Prototypes for the APIs.
//*****************************************************************************

/**
* Wait for event, pass it to the hci_event_handler and update the event opcode in a global variable.
* @param  usOpcode      command operation code
* @param  pRetParams    command return parameters
* @return               none
*/
extern void SimpleLinkWaitEvent(unsigned short usOpcode, void *pRetParams);

/**
* Wait for data, pass it to the hci_event_handler and update in a global variable that there is data to read.
* @param  pBuf       data buffer
* @param  from       from information
* @param  fromlen    from information length
* @return            none
*/
extern void SimpleLinkWaitData(unsigned char *pBuf, unsigned char *from, unsigned char *fromlen);

/**
* Copy 32 bit to stream while converting to little endian format.
* @param  p       pointer to the new stream
* @param  u32     pointer to the 32 bit
* @return         pointer to the new stream
*/
extern unsigned char* UINT32_TO_STREAM_f (unsigned char *p, unsigned long u32);

/**
* Copy 16 bit to stream while converting to little endian format.
* @param  p       pointer to the new stream
* @param  u32     pointer to the 16 bit
* @return         pointer to the new stream
*/
extern unsigned char* UINT16_TO_STREAM_f (unsigned char *p, unsigned short u16);

/**
* Copy received stream to 16 bit in little endian format.
* @param  p          pointer to the stream
* @param  offset     offset in the stream
* @return            pointer to the new 16 bit
*/
extern unsigned short STREAM_TO_UINT16_f(char* p, unsigned short offset);

/**
* Copy received stream to 32 bit in little endian format.
* @param  p          pointer to the stream
* @param  offset     offset in the stream
* @return            pointer to the new 32 bit
*/
extern unsigned long STREAM_TO_UINT32_f(char* p, unsigned short offset);


//*****************************************************************************
//                    COMMON MACROs
//*****************************************************************************


//Copy 8 bit to stream while converting to little endian format.
#define UINT8_TO_STREAM(_p, _val)    {*(_p)++ = (_val);}
//Copy 16 bit to stream while converting to little endian format.
#define UINT16_TO_STREAM(_p, _u16)    (UINT16_TO_STREAM_f(_p, _u16))
//Copy 32 bit to stream while converting to little endian format.
#define UINT32_TO_STREAM(_p, _u32)    (UINT32_TO_STREAM_f(_p, _u32))
//Copy a specified value length bits (l) to stream while converting to little endian format.
#define ARRAY_TO_STREAM(p, a, l)     {register short _i; for (_i = 0; _i < l; _i++) *(p)++ = ((unsigned char *) a)[_i];}
//Copy received stream to 8 bit in little endian format.
#define STREAM_TO_UINT8(_p, _offset, _u8)    {_u8 = (unsigned char)(*(_p + _offset));}
//Copy received stream to 16 bit in little endian format.
#define STREAM_TO_UINT16(_p, _offset, _u16)    {_u16 = STREAM_TO_UINT16_f(_p, _offset);}
//Copy received stream to 32 bit in little endian format.
#define STREAM_TO_UINT32(_p, _offset, _u32)    {_u32 = STREAM_TO_UINT32_f(_p, _offset);}
#define STREAM_TO_STREAM(p, a, l)     {register short _i; for (_i = 0; _i < l; _i++) *(a)++= ((unsigned char *) p)[_i];}

#ifdef  __cplusplus
}
#endif // __cplusplus
//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************

#endif // __COMMON_H__










