/*****************************************************************************
*
*  nvmem  - CC3000 Host Driver Implementation.
*  Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the   
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*****************************************************************************/


#include "nvmem.h"

signed long nvmem_read(unsigned long ulFileId, unsigned long ulLength, unsigned long ulOffset, unsigned char *buff)
{
    unsigned char ucStatus = 0xFF;
    unsigned char *ptr;
    unsigned char *args;
    
    ptr = tSLInformation.pucTxCommandBuffer;
    args = (ptr + HEADERS_SIZE_CMD);
    // Fill in HCI packet structure
    args = UINT32_TO_STREAM(args, ulFileId);
    args = UINT32_TO_STREAM(args, ulLength);
    args = UINT32_TO_STREAM(args, ulOffset);
    
    // Initiate HCI command
    hci_command_send(HCI_CMND_NVMEM_READ, ptr, NVMEM_READ_PARAMS_LEN);
    SimpleLinkWaitEvent(HCI_CMND_NVMEM_READ, &ucStatus);
    
    // If data is present, read it even when an error is returned.
    // Note: It is the users responsibility to ignore the data when an error is returned.
    // Wait for the data in a synchronous way.
    //  We assume the buffer is large enough to also store nvmem parameters.
    SimpleLinkWaitData(buff, 0, 0);

    return(ucStatus);
}


signed long nvmem_write(unsigned long ulFileId, unsigned long ulLength, unsigned long ulEntryOffset, unsigned char *buff)
{
    long iRes;
    unsigned char *ptr;
    unsigned char *args;
    
    iRes = EFAIL;
    
    ptr = tSLInformation.pucTxCommandBuffer;
    args = (ptr + SPI_HEADER_SIZE + HCI_DATA_CMD_HEADER_SIZE);
    
    // Fill in HCI packet structure
    args = UINT32_TO_STREAM(args, ulFileId);
    args = UINT32_TO_STREAM(args, 12);
    args = UINT32_TO_STREAM(args, ulLength);
    args = UINT32_TO_STREAM(args, ulEntryOffset);
    
    memcpy((ptr + SPI_HEADER_SIZE + HCI_DATA_CMD_HEADER_SIZE + 
                    NVMEM_WRITE_PARAMS_LEN),buff,ulLength);
    
    // Initiate a HCI command on the data channel
    hci_data_command_send(HCI_CMND_NVMEM_WRITE, ptr, NVMEM_WRITE_PARAMS_LEN, ulLength);
    
    SimpleLinkWaitEvent(HCI_EVNT_NVMEM_WRITE, &iRes);
    
    return(iRes);
}


unsigned char nvmem_set_mac_address(unsigned char *mac)
{
    return  nvmem_write(NVMEM_MAC_FILEID, MAC_ADDR_LEN, 0, mac);
}


unsigned char nvmem_get_mac_address(unsigned char *mac)
{
    return  nvmem_read(NVMEM_MAC_FILEID, MAC_ADDR_LEN, 0, mac);
}


unsigned char nvmem_write_patch(unsigned long ulFileId, unsigned long spLength, const unsigned char *spData)
{
    unsigned char     status = 0;
    unsigned short    offset = 0;
    unsigned char*      spDataPtr = (unsigned char*)spData;
    
    while ((status == 0) && (spLength >= SP_PORTION_SIZE))
    {
        status = nvmem_write(ulFileId, SP_PORTION_SIZE, offset, spDataPtr);
        offset += SP_PORTION_SIZE;
        spLength -= SP_PORTION_SIZE;
        spDataPtr += SP_PORTION_SIZE;
    }
    
    if (status !=0)
    {
        // NVMEM error occurred
        return status;
    }
    
    if (spLength != 0)
    {
        // If spLength MOD 512 is nonzero, write the remaining bytes.
        status = nvmem_write(ulFileId, spLength, offset, spDataPtr);
    }
    
    return status;
}


#ifndef CC3000_TINY_DRIVER
unsigned char nvmem_read_sp_version(unsigned char* patchVer)
{
    unsigned char *ptr;
    // 1st byte is the status and the rest is the SP version
    unsigned char retBuf[5];    
    
    ptr = tSLInformation.pucTxCommandBuffer;
  
   // Initiate a HCI command, no args are required
    hci_command_send(HCI_CMND_READ_SP_VERSION, ptr, 0);    
    SimpleLinkWaitEvent(HCI_CMND_READ_SP_VERSION, retBuf);
    
    // package ID
    *patchVer = retBuf[3];            
    // package build number
    *(patchVer+1) = retBuf[4];        
    
    return(retBuf[0]);
}
#endif


signed long 
nvmem_create_entry(unsigned long ulFileId, unsigned long ulNewLen)
{
    unsigned char *ptr; 
    unsigned char *args;
    unsigned short retval;
    
    ptr = tSLInformation.pucTxCommandBuffer;
    args = (ptr + HEADERS_SIZE_CMD);
    
    // Fill in HCI packet structure
    args = UINT32_TO_STREAM(args, ulFileId);
    args = UINT32_TO_STREAM(args, ulNewLen);
    
    // Initiate a HCI command
    hci_command_send(HCI_CMND_NVMEM_CREATE_ENTRY,ptr, NVMEM_CREATE_PARAMS_LEN);
    
    SimpleLinkWaitEvent(HCI_CMND_NVMEM_CREATE_ENTRY, &retval);
    
    return(retval);
}


