/**
 * Copyright (c) 2017, Arm Limited and affiliates.
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include <stdio.h>
#include "mbed.h"
#include "lorawan/LoRaWANInterface.h"
#include "lorawan/system/lorawan_data_structures.h"
#include "events/EventQueue.h"
#include "TinyGPSPlus.h"
// Application helpers
#include "trace_helper.h"
#include "lora_radio_helper.h"

using namespace events;

// Max payload size can be LORAMAC_PHY_MAXPAYLOAD.
// This example only communicates with much shorter messages (<30 bytes).
// If longer messages are used, these buffers must be changed accordingly.
uint8_t tx_buffer[30];
uint8_t rx_buffer[30];
char buf[128];

char c;
/*
 * Sets up an application dependent transmission timer in ms. Used only when Duty Cycling is off for testing
 */
#define TX_TIMER                        10000



/**
 * Maximum number of events for the event queue.
 * 10 is the safe number for the stack events, however, if application
 * also uses the queue for whatever purposes, this number should be increased.
 */
#define MAX_NUMBER_OF_EVENTS            10

/**
 * Maximum number of retries for CONFIRMED messages before giving up
 */
#define CONFIRMED_MSG_RETRY_COUNTER     3

#define DATABUFFER_SIZE 128




Serial gpsPort(PC_10,PC_11,115200);
Serial pc(PA_9,PA_10);
TinyGPSPlus gps;
DigitalOut GPS_LEVEL_SHIFTER_EN(PC_6);
DigitalOut GPS_RESET(PB_2);
/**
* This event queue is the global event queue for both the
* application and stack. To conserve memory, the stack is designed to run
* in the same thread as the application and the application is responsible for
* providing an event queue to the stack that will be used for ISR deferment as
* well as application information event queuing.
*/
static EventQueue ev_queue(MAX_NUMBER_OF_EVENTS *EVENTS_EVENT_SIZE);

/**
 * Event handler.
 *
 * This will be passed to the LoRaWAN stack to queue events for the
 * application which in turn drive the application.
 */
static void lora_event_handler(lorawan_event_t event);

/**
 * Constructing Mbed LoRaWANInterface and passing it the radio object from lora_radio_helper.
 */
static LoRaWANInterface lorawan(radio);

/**
 * Application specific callbacks
 */
static lorawan_app_callbacks_t callbacks;

void gpsSetup()
{

    GPS_LEVEL_SHIFTER_EN=1;
    wait(0.2);
    GPS_RESET=0;

    wait(1);

    GPS_RESET=1;

   // wait(1);
    //gpsPort.printf("@GSR\r\n");//"@GNS 0x3\r\n"
    //wait(.25);
    //gpsPort.printf("@GNS 0x3\r\n");
    //wait(.25);
    //gpsPort.printf("@BSSL 0x25\r\n");
    //wait(.25);
    //gpsPort.printf("@GCD\r\n");
    //wait(.25);
}

void getGpsData()
{   
    
    c= gpsPort.getc();

    //pc.putc(c);
    switch (c) {
        case '\n':
            if (gps.satellites.isValid() && gps.satellites.value() > 3 && gps.hdop.hdop() > 0) {
                snprintf(buf, 128, "{\"lat\":%lf,\"lng\":%lf}", gps.location.lat(), gps.location.lng());
            } else {
                snprintf(buf, 128, "Satellites: %lu, time: %04d-%02d-%02dT%02d:%02d:%02d.%02d",
                         gps.satellites.value(), gps.date.year(), gps.date.month(), gps.date.day(),
                         gps.time.hour(), gps.time.minute(), gps.time.second(), gps.time.centisecond());
            }
            pc.printf("%s\r\n", buf);
        break;
        default:
            gps.encode(c);
            break;
    }
    
    ev_queue.call(getGpsData); 
}


/**
 * Entry point for application
 */
int main(void)

{
    //getGpsData();
    gpsSetup();
    

    pc.printf("hello");
    // setup tracing
    setup_trace();

    // stores the status of a call to LoRaWAN protocol
    lorawan_status_t retcode;

    // Initialize LoRaWAN stack
    if (lorawan.initialize(&ev_queue) != LORAWAN_STATUS_OK) {
        pc.printf("\r\n LoRa initialization failed! \r\n");
        return -1;
    }

    pc.printf("\r\n Mbed LoRaWANStack initialized \r\n");

    // prepare application callbacks
    callbacks.events = mbed::callback(lora_event_handler);
    lorawan.add_app_callbacks(&callbacks);

    // Set number of retries in case of CONFIRMED messages
    if (lorawan.set_confirmed_msg_retries(CONFIRMED_MSG_RETRY_COUNTER)
            != LORAWAN_STATUS_OK) {
        pc.printf("\r\n set_confirmed_msg_retries failed! \r\n\r\n");
        return -1;
    }

    pc.printf("\r\n CONFIRMED message retries : %d \r\n",
              CONFIRMED_MSG_RETRY_COUNTER);

    // Enable adaptive data rate
    if (lorawan.enable_adaptive_datarate() != LORAWAN_STATUS_OK) {
        pc.printf("\r\n enable_adaptive_datarate failed! \r\n");
        return -1;
    }

    pc.printf("\r\n Adaptive data  rate (ADR) - Enabled \r\n");

    retcode = lorawan.connect();

    if (retcode == LORAWAN_STATUS_OK ||
            retcode == LORAWAN_STATUS_CONNECT_IN_PROGRESS) {
    } else {
        pc.printf("\r\n Connection error, code = %d \r\n", retcode);
        return -1;
    }

    pc.printf("\r\n Connection - In Progress ...\r\n");
     
     
    ev_queue.call(getGpsData); 
    // make your event queue dispatching events forever
    ev_queue.dispatch_forever();

    return 0;
}

/**
 * Sends a message to the Network Server
 */
static void send_message()
{
    uint16_t packet_len;
    int16_t retcode;
    pc.printf("\n\rHello world");

    packet_len = sprintf((char *) tx_buffer, "Hello world");

    retcode = lorawan.send(MBED_CONF_LORA_APP_PORT, tx_buffer, packet_len,
                           MSG_UNCONFIRMED_FLAG);

    if (retcode < 0) {
        retcode == LORAWAN_STATUS_WOULD_BLOCK ? pc.printf("send - WOULD BLOCK\r\n")
        : pc.printf("\r\n send() - Error code %d \r\n", retcode);

        if (retcode == LORAWAN_STATUS_WOULD_BLOCK) {
            //retry in 3 seconds
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                ev_queue.call_in(3000, send_message);
            }
        }
        return;
    }

    pc.printf("\r\n %d bytes scheduled for transmission \r\n", retcode);
    memset(tx_buffer, 0, sizeof(tx_buffer));

}

/**
 * Event handler
 */
static void lora_event_handler(lorawan_event_t event)
{
    switch (event) {
        case CONNECTED:
            pc.printf("\r\n Connection - Successful \r\n");
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            } else {
                ev_queue.call_every(TX_TIMER, send_message);
            }

            break;
        case DISCONNECTED:
            ev_queue.break_dispatch();
            pc.printf("\r\n Disconnected Successfully \r\n");
            break;
        case TX_DONE:
            pc.printf("\r\n Message Sent to Network Server \r\n");
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();

            }
            break;
        case TX_TIMEOUT:
        case TX_ERROR:
        case TX_CRYPTO_ERROR:
        case TX_SCHEDULING_ERROR:
            pc.printf("\r\n Transmission Error - EventCode = %d \r\n", event);
            // try again
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            }
            break;
        case RX_DONE:
            pc.printf("\r\n Received message from Network Server \r\n");
            break;
        case RX_TIMEOUT:
        case RX_ERROR:
            pc.printf("\r\n Error in reception - Code = %d \r\n", event);
            break;
        case JOIN_FAILURE:
            pc.printf("\r\n OTAA Failed - Check Keys \r\n");
            break;
        case UPLINK_REQUIRED:
            pc.printf("\r\n Uplink required by NS \r\n");
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            }
            break;
        default:
            MBED_ASSERT("Unknown Event");
    }
}

// EOF
