#include "mbed.h"
#include "EthernetNetIf.h"
#include "HTTPClient.h"
#include "spdomparser.hpp"
#include "spxmlnode.hpp"
#include "spxmlhandle.hpp"
#include "uLCD_4DGL.h"
#include "emic2.h"
#define speakf printf

uLCD_4DGL lcd(p9,p10,p11); 
emic2 myTTS(p13, p14);
EthernetNetIf eth;
HTTPClient http;
HTTPResult result;
bool completed = false;
void request_callback(HTTPResult r) {
    result = r;
    completed = true;
}

// RGB Setup
class LEDColor
{
public:
    LEDColor(float r, float g, float b);
    float red;
    float green;
    float blue;
};
LEDColor:: LEDColor(float r, float g, float b)
    : red(r), green(g), blue(b)
{
}
//Operator overload to adjust brightness with no color change
LEDColor operator * (const LEDColor& x, const float& b)
{
    return LEDColor(x.red*b,x.green*b,x.blue*b);
}
//Operator overload to add colors
LEDColor operator + (const LEDColor& x, const LEDColor& y)
{
    return LEDColor(x.red+y.red,x.green+y.green,x.blue+y.blue);
}

//Class to control an RGB LED using three PWM pins
class RGBLed
{
public:
    RGBLed(PinName redpin, PinName greenpin, PinName bluepin);
    void write(float red,float green, float blue);
    void write(LEDColor c);
    RGBLed operator = (LEDColor c) {
        write(c);
        return *this;
    };
private:
    PwmOut _redpin;
    PwmOut _greenpin;
    PwmOut _bluepin;
};

RGBLed::RGBLed (PinName redpin, PinName greenpin, PinName bluepin)
    : _redpin(redpin), _greenpin(greenpin), _bluepin(bluepin)
{
    //50Hz PWM clock default a bit too low, go to 2000Hz (less flicker)
    _redpin.period(0.0005);
}

void RGBLed::write(float red,float green, float blue)
{
    _redpin = red;
    _greenpin = green;
    _bluepin = blue;
}
void RGBLed::write(LEDColor c)
{
    _redpin = c.red;
    _greenpin = c.green;
    _bluepin = c.blue;
}


RGBLed myRGBled(p23,p22,p21);
const LEDColor blue(0.0,0.0,1.0);
const LEDColor yellow(0.2,1.0,0.0);
//Setup a new class for TMP36 sensor
class TMP36
{
public:
    TMP36(PinName pin);
    TMP36();
    operator float ();
    float read();
private:
//class sets up the AnalogIn pin
    AnalogIn _pin;
};

TMP36::TMP36(PinName pin) : _pin(pin)
{
// _pin(pin) means pass pin to the AnalogIn constructor
}

float TMP36::read()
{
//convert sensor reading to temperature in degrees C
    return ((_pin.read()*3.3)-0.500)*100.0;
}
//overload of float conversion (avoids needing to type .read() in equations)
TMP36::operator float ()
{
//convert sensor reading to temperature in degrees C
    return ((_pin.read()*3.3)-0.500)*100.0;
}

//use the new class to set p15 to analog input to read and convert TMP36 sensor's voltage output
TMP36 myTMP36(p15);

int main() {
    lcd.background_color(WHITE);
    lcd.textbackground_color(WHITE);
    lcd.cls();
    lcd.locate(0,2);
    lcd.color(BLACK);
    lcd.printf("net setup");
    lcd.locate(0,3);
    EthernetErr ethErr = eth.setup();
    if (ethErr) {
        lcd.printf("Error in setup");
        return -1;
    }
    lcd.printf("net ok");

    SP_XmlDomParser parser;
    HTTPStream stream;

    char BigBuf[512 + 1] = {0};
    stream.readNext((byte*)BigBuf, 512);
    HTTPResult r = http.get("http://api.openweathermap.org/data/2.5/weather?q=Atlanta,us&appid=8df1acdaba0957258c46ced41ce2eedc&mode=xml", &stream, request_callback);
    while (!completed) {
        Net::poll(); 
        if (stream.readable()) {
            BigBuf[stream.readLen()] = 0; 
            parser.append( BigBuf, strlen(BigBuf)); 
            stream.readNext((byte*)BigBuf, 512); 
        }
    }
    lcd.cls();
    lcd.locate(0,2);
    if (result == HTTP_OK) {
        lcd.printf(" Read complete");
    } else {
        lcd.printf(" Error %d", http.getHTTPResponseCode());
        return -1;
    }

    SP_XmlHandle rootHandle( parser.getDocument()->getRootElement() );
    //temp
    SP_XmlElementNode * temp = rootHandle.getChild(1).toElement();
    lcd.cls();
    lcd.locate(0,1);
    lcd.text_bold(ON);
    lcd.text_char('A', 0, 1, BLACK);
    lcd.text_bold(ON);
    lcd.text_char('T', 1, 1, BLACK);
    lcd.text_bold(ON);
    lcd.text_char('L', 2, 1, BLACK);
    lcd.locate(0,2);
    float mytmp = myTMP36/10;
    lcd.printf("Indoor Temp: %5.1FC", mytmp);
    lcd.locate(0,3);
    //temperature
    float temperature = atof(temp->getAttrValue("value"))-273.15;
    lcd.printf("Outdoor Temp: %.1f C ", temperature );
    //humidity
    SP_XmlElementNode * humidity = rootHandle.getChild(2).toElement();
    lcd.locate(0,5);
    string humid = humidity->getAttrValue("value");
    lcd.printf("Humidity: %s%% ", humid);
    //condition
    SP_XmlElementNode * condition = rootHandle.getChild(8).toElement();
    string cond = condition->getAttrValue("value");
    //wind
    lcd.locate(0,7);
    SP_XmlElementNode * wind = rootHandle.getChild(4).getChild(0).toElement();
    string windy = wind->getAttrValue("name");
    lcd.printf("Weather: %s & %s ", cond, windy);
    //rain
    SP_XmlElementNode * rain = rootHandle.getChild(7).toElement();
    string rainy = rain->getAttrValue("mode");
    
    // RGB: Blue if raining, yellow if not raining
    if (rainy.compare("yes")) {
        myRGBled = blue; //blue
    } else if (rainy.compare("no")) {
        myRGBled = yellow; //yellow
    }
    
    myTTS.volume(10);
    myTTS.voice(6);
    myTTS.speakf("SThe current indoor temperature is %5.2f degrees. The current outdoor temperature is %.2f degrees with %s percent humidity. Currently the weather condition is %s with %s\r", mytmp, temperature, humid, cond, windy);
    myTTS.ready();
    if (temperature < 15.0) {
        myTTS.speakf("SBring a jacket with you\r");
        myTTS.ready();
    } else if(temperature > 25.0) {
        myTTS.speakf("SIt's hot outside, bring water with you\r");
        myTTS.ready();
    } else {
        myTTS.speakf("SThe temperature is pleasant outside\r");
        myTTS.ready();
    }
    if (rainy == "yes") {
        myTTS.speakf("SBring your rain gear with you. It is currently raining outside.\r");
        myTTS.ready();
    }
    
    myRGBled.write(0.0, 0.0, 0.0);
}
