#include "mbed.h"
#include <stdio.h>
#include <errno.h>

// Block devices
#if COMPONENT_SPIF
#include "SPIFBlockDevice.h"
#endif

#if COMPONENT_DATAFLASH
#include "DataFlashBlockDevice.h"
#endif 

#if COMPONENT_SD
#include "SDBlockDevice.h"
#endif 

#include "HeapBlockDevice.h"

// File systems
#include "LittleFileSystem.h"
#include "FATFileSystem.h"
#include "stm32l475e_iot01_accelero.h"
#include "nvstore.h"

// Physical block device, can be any device that supports the BlockDevice API
/*SPIFBlockDevice bd(
        MBED_CONF_SPIF_DRIVER_SPI_MOSI,
        MBED_CONF_SPIF_DRIVER_SPI_MISO,
        MBED_CONF_SPIF_DRIVER_SPI_CLK,
        MBED_CONF_SPIF_DRIVER_SPI_CS);*/

DigitalOut led_1(LED1);
DigitalOut led_2(LED2);
DigitalOut led_3(LED3);
Ticker check_led_status;

EventQueue queue(32 * EVENTS_EVENT_SIZE);
Thread t;

#define BLOCK_SIZE 512
HeapBlockDevice bd(32768, BLOCK_SIZE);

#define LED_1_VALUE 1
#define LED_2_VALUE 2
#define LED_3_VALUE 3

// File system declaration
LittleFileSystem fs("fs");

// Set up the button to trigger an erase
InterruptIn irq(BUTTON1);
void erase() {
    printf("Initializing the block device... ");
    fflush(stdout);
    int err = bd.init();
    printf("%s\n", (err ? "Fail :(" : "OK"));
    if (err) {
        error("error: %s (%d)\n", strerror(-err), err);
    }

    printf("Erasing the block device... ");
    fflush(stdout);
    err = bd.erase(0, bd.size());
    printf("%s\n", (err ? "Fail :(" : "OK"));
    if (err) {
        error("error: %s (%d)\n", strerror(-err), err);
    }

    printf("Deinitializing the block device... ");
    fflush(stdout);
    err = bd.deinit();
    printf("%s\n", (err ? "Fail :(" : "OK"));
    if (err) {
        error("error: %s (%d)\n", strerror(-err), err);
    }
}

static FILE *f;

static uint16_t key_led1 = 1; 
static uint16_t key_led2 = 2; 
static uint16_t key_led3 = 3;

static uint32_t value_led1;
static uint32_t value_led2;
static uint32_t value_led3;

volatile int counter = 0;

void read_acc() {
    int16_t pDataXYZ[3] = {0};
    BSP_ACCELERO_AccGetXYZ(pDataXYZ);
    if (abs(pDataXYZ[2]) > 900) { // board laying down
            fprintf(f, "%d\n", LED_1_VALUE);
        } else if (abs(pDataXYZ[1]) > 900) { // long side
            fprintf(f, "%d\n", LED_2_VALUE);
        } else if (abs(pDataXYZ[0]) > 900) { // short side
             fprintf(f, "%d\n", LED_3_VALUE);
        } else { // all other positions
             fprintf(f, "%d\n", -1);
        }
        
    fflush(f);
    fflush(stdout);
}

void read_data_from_file() {
      int led1_counter = 0;
      int led2_counter = 0;
      int led3_counter = 0;
      fflush(stdout);
      fflush(f);
      
      // read file and count occurrences of position numbers
      fseek(f, 0, SEEK_SET);
      int number;
      while (!feof(f)) {
        fscanf(f, "%d", &number);
        
        switch (number){
        case LED_1_VALUE:
            led1_counter += 1;
            break;
        case LED_2_VALUE:
            led2_counter += 1;
            break;
        case LED_3_VALUE:
            led3_counter += 1;
            break;
        default:
            break; 
        }
      }
      
      NVStore &nvstore = NVStore::get_instance();
      // check which position is the one present the most
      if (led1_counter > led2_counter && led1_counter > led3_counter) {
            led_1 = 1;
            led_2 = 0;
            led_3 = 0;
            value_led1 += 1;
            nvstore.set(key_led1, sizeof(value_led1), &value_led1);
        } else if (led2_counter > led1_counter && led2_counter > led3_counter ) {
            led_1 = 0;
            led_2 = 1;
            led_3 = 0;
            value_led2 += 1;
            nvstore.set(key_led2, sizeof(value_led2), &value_led2);
        } else if (led3_counter > led1_counter && led3_counter > led2_counter) {
            led_1 = 0;
            led_2 = 0;
            led_3 = 1;
            value_led3 += 1;
            nvstore.set(key_led3, sizeof(value_led3), &value_led3);
        }
      
      fflush(stdout);
      int err = fclose(f);
      printf("%s\n", (err < 0 ? "Fail :(" : "OK"));
      if (err < 0) {
        error("error: %s (%d)\n", strerror(err), -err);
      }
      err = fs.unmount();
      printf("%s\n", (err < 0 ? "Fail :(" : "OK"));
      if (err < 0) {
          error("error: %s (%d)\n", strerror(-err), err);
      }
        
      if (led_1){  
        printf("Lighting up corresponding LED 1!\n");
      } else if (led_2){
        printf("Lighting up corresponding LED 2!\n");
      } else if (led_3){
        printf("Lighting up corresponding LED 3!\n");
      } else {
        printf("No valid position detected during the sampling time, no LED lighting up!\n");
      }
    
}

void toggle_led() {
    queue.call(read_acc);
    counter++;
    if (counter == 100*10) {
      check_led_status.detach();
      queue.call(read_data_from_file);
    }
      
}

// Entry point for the example
int main() {
    t.start(callback(&queue, &EventQueue::dispatch_forever));
    BSP_ACCELERO_Init();
    
    NVStore &nvstore = NVStore::get_instance();
    
    uint16_t actual_len_bytes = 0;
    int rc;
    
    rc = nvstore.init();
    
    rc = nvstore.get(key_led1, sizeof(value_led1), &value_led1, actual_len_bytes);
    if (rc == NVSTORE_NOT_FOUND) {
        value_led1 = 0;
        value_led2 = 0;
        value_led3 = 0;
        nvstore.set(key_led1, sizeof(value_led1), &value_led1);
        nvstore.set(key_led2, sizeof(value_led2), &value_led2);
        nvstore.set(key_led3, sizeof(value_led3), &value_led3);
    } else {
        nvstore.get(key_led2, sizeof(value_led2), &value_led2, actual_len_bytes);
        nvstore.get(key_led3, sizeof(value_led3), &value_led3, actual_len_bytes);    
    }
    
    printf("\nLED1 has been on: %d times\nLED2 has been on: %d times\nLED3 has been on: %d times\n", value_led1, value_led2, value_led3);
    
    printf("--- Mbed OS filesystem example ---\n");

    // Setup the erase event on button press, use the event queue
    // to avoid running in interrupt context
    irq.fall(mbed_event_queue()->event(erase));

    // Try to mount the filesystem
    printf("Mounting the filesystem... ");
    fflush(stdout);
    int err = fs.mount(&bd);
    printf("%s\n", (err ? "Fail :(" : "OK"));
    if (err) {
        // Reformat if we can't mount the filesystem
        // this should only happen on the first boot
        printf("No filesystem found, formatting... ");
        fflush(stdout);
        err = fs.reformat(&bd);
        printf("%s\n", (err ? "Fail :(" : "OK"));
        if (err) {
            error("error: %s (%d)\n", strerror(-err), err);
        }
    }

    // Open the numbers file
    printf("Opening \"/fs/numbers.txt\"... ");
    fflush(stdout);
    f = fopen("/fs/numbers.txt", "r +");
    printf("%s\n", (!f ? "Fail :(" : "OK"));
    if (!f) {
        // Create the numbers file if it doesn't exist
        printf("No file found, creating a new file... ");
        fflush(stdout);
        f = fopen("/fs/numbers.txt", "w+");
        printf("%s\n", (!f ? "Fail :(" : "OK"));
        if (!f) {
            error("error: %s (%d)\n", strerror(errno), -errno);
        }

        printf("Seeking file... ");
        fflush(stdout);
        err = fseek(f, 0, SEEK_SET);
        printf("%s\n", (err < 0 ? "Fail :(" : "OK"));
        if (err < 0) {
            error("error: %s (%d)\n", strerror(errno), -errno);
        }
    }
    
    check_led_status.attach(&toggle_led, 0.01);
}