/*
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.

  * Demo Program
  * Barometer Sensor (Altimeter) MS5803-01BA of MEAS Switzerland     (www.meas-spec.com).
  * The driver uses I2C mode (sensor PS pin low).
  * Other types of MEAS are compatible but not tested.
  * Written by Raig Kaufer distribute freely!
  *
  * Modified for MS5803-14BA by Francesco Adamo, Italy
 */

#include <stdlib.h>
#include "MS5803_14BA.h"

MS5803_14BA::MS5803_14BA(PinName sda, PinName scl, uint8_t address) : _i2c(sda, scl) {
    _address = address << 1;
    _d1_osr = D1_OSR_4096;
    _d2_osr = D2_OSR_4096;
    reset();    // reset the sensor
    readPROM(); // read the calibration values
}

MS5803_14BA::MS5803_14BA(PinName sda, PinName scl, uint8_t address, uint8_t d1_osr, uint8_t d2_osr) : _i2c( sda, scl ) {
    _address = address << 1;
    _d1_osr = d1_osr;
    _d2_osr = d2_osr;
    reset();    // reset the sensor
    readPROM(); // read the calibration values and store them in array C 
}

/* Send soft reset to the sensor */
void MS5803_14BA::reset(void) {
    MS5803_tx_data[0] = COMMAND_RESET;
    _i2c.write(_address,  MS5803_tx_data, 1);
    wait_ms(20);
}

/* read the sensor calibration data from ROM */
void MS5803_14BA::readPROM(void) {
  
    for (uint8_t i = 0; i < 8; i++) {
        MS5803_tx_data[0] = COMMAND_READ_PROM + (i << 1);
        _i2c.write(_address,  MS5803_tx_data, 1);

        _i2c.read(_address,  MS5803_rx_data, 2);
        C[i] = (MS5803_rx_data[0] << 8) + MS5803_rx_data[1];
    }
}


/* Sensor reading and calculation procedure */
void MS5803_14BA::convert(void) {
    int32_t dT, TEMP;
    int64_t OFF, SENS, P;

    MS5803_tx_data[0] = _d1_osr;
    
    _i2c.write(_address,  MS5803_tx_data, 1);
      wait_ms(10);
    D1 = readADC(); // read the temperature value
        
    MS5803_tx_data[0] = _d2_osr;
    _i2c.write(_address,  MS5803_tx_data, 1);
      wait_ms(10);
    D2 = readADC(); // read the temperature value
        
    /* calculation according MS5803-14BA data sheet */
    dT = D2 - (C[5]* 256);
    TEMP = 2000 + (dT * C[6])/8388608;

        //Now we have our first order Temperature, let's calculate the second order.
    int64_t T2, OFF2, SENS2; //working variables

    if (TEMP < 2000) {
    // If temp_calc is below 20.0C
        T2 = 3 * (((int64_t)dT * dT) >> 33);
        OFF2 = 3 * (TEMP - 2000)^2 / 2;
        SENS2 = 5 * (TEMP - 2000)^2 / 8;
        
        if(TEMP < -1500) {
            // If temp_calc is below -15.0C 
            OFF2 = OFF2 + 7 * (TEMP + 1500)^2;
            SENS2 = SENS2 + 4 * (TEMP + 1500)^2;
        }
  } 
    else {
    // If temp_calc is above 20.0C
        T2 = 7 * ((uint64_t)dT * dT) >> 37;
        OFF2 = (TEMP - 2000)^2 / 16;
        SENS2 = 0;
    }
    
    // Now bring it all together to apply offsets 
    OFF = ((int64_t)C[2] << 16) + (((C[4] * (int64_t)dT)) >> 7);
    SENS = ((int64_t)C[1] << 15) + (((C[3] * (int64_t)dT)) >> 8);
    
    TEMP = TEMP - T2;
    OFF = OFF - OFF2;
    SENS = SENS - SENS2;

  temperature = (float) TEMP/100.0f; // result of temperature in degC in this var

  P = ((((int64_t)D1 * SENS) >> 21) - OFF) >> 15;
  pressure = (float) P/10.0f; // result of pressure in mBar in this var
}

/* Read the previous started conversion results */
int32_t MS5803_14BA::readADC(void) {
    //wait_ms(150);
    MS5803_tx_data[0] = COMMAND_READ_ADC;
    _i2c.write(_address,  MS5803_tx_data, 1);
        _i2c.read(_address,  MS5803_rx_data, 3);
        
    return (MS5803_rx_data[0] << 16) + (MS5803_rx_data[1] << 8) + MS5803_rx_data[2];
}
