/***************************************************************************
 * @author Francesco Adamo
 *
 * @section LICENSE
 *
 * Copyright (c) 2014 Francesco Adamo
 *
 * @section DESCRIPTION
 *
 *  ADS7828.CPP
 *  Implementation file for library ADS7828     
 *  The ADS7828 is a 12-bits, 8-channels, I2C-interfaced from Texas Instruments
 *   
 *  (c) 2014 - By Francesco Adamo, Italy
 ****************************************************************************/

#include "mbed.h"
#include "ADS7828.h"

// Default constructor
ADS7828::ADS7828(PinName sda, PinName scl) : _i2c(sda, scl) {
    _i2c.frequency(100000);  /* Default I2C clock frequency = 100 kHz, Normal Mode */
    address = ADS7828_BASE_ADDRESS; // ADS7828 address with pins A0 and A1 connected to ground
    VFSR = 2.5F;    // Default VFSR (valid if VREF pin is unconnected)
}

// Constructor overload
ADS7828::ADS7828(PinName sda, PinName scl, int freq) : _i2c(sda, scl) {
    _i2c.frequency(freq);  /* I2C clock frequency = set to input value */
    address = ADS7828_BASE_ADDRESS;
    VFSR = 2.5F;
}

// Constructor overload
ADS7828::ADS7828(PinName sda, PinName scl, int freq, char subAddress) : _i2c(sda, scl) {
    _i2c.frequency(freq);  /* I2C clock frequency = set to input value */
    address = ADS7828_BASE_ADDRESS | subAddress;
    VFSR = 2.5F;
}

// Constructor overload
ADS7828::ADS7828(PinName sda, PinName scl, int freq, char subAddress, double VREF) : _i2c(sda, scl) {
    _i2c.frequency(freq);  /* I2C clock frequency = set to input value */
    address = ADS7828_BASE_ADDRESS | subAddress;
    VFSR = VREF;
}

// Read a raw value from a given channel
int ADS7828::readRawValue(char mode, char channel) {
    char command, D[2], ack;
    
    command = mode | channel;
    ack = _i2c.write(address, &command, 1);
    if (ack)
        return -1;
        
    ack = _i2c.read(address + 1, D, 2); // Read conversion result
    if (ack)
        return -1;
    else
        return (int) (D[0] << 8 | D[1]);
}

// Read a value from a given channel and scales it to VFSR
double ADS7828::readAnalogValue(char mode, char channel) {
    char command, D[2], ack;
    
    command = mode | channel;
    ack = _i2c.write(address, &command, 1);
    if (ack)
        return -1;
        
    ack = _i2c.read(address + 1, D, 2); // Read conversion result
    if (ack)
        return -1;
    else
        return VFSR/4096*((int) (D[0] << 8 | D[1]));
}