/***************************************************************************
 * @author Francesco Adamo
 *
 * @section LICENSE
 *
 * Copyright (c) 2015 Francesco Adamo, Italy
 *
 * @section DESCRIPTION
 *
 *  AD7606.CPP
 *  Source file for AD7606 class library
 *  The AD7606BSTZ is a 16-bits, 8-channels, SPI/Parallel-interfaced ADC from Analog Devices
 *  
*****************************************************************************/

#include "ad7606.h"


// Constructor
AD7606::AD7606(PinName MISO, PinName SCLK, PinName CS, PinName CONVST, PinName BUSY, PinName RESET, int frequency) : _spi(NC, MISO, SCLK), _cs(CS), _convst(CONVST), _busy(BUSY), _reset(RESET) {    
    _spi.frequency(frequency);
    _spi.format(8, 0);
    _cs = 1;
    _convst = 1;
    _reset = 0;
    _q = 0.0;
    reset();
};

// Resets the AD7606
void AD7606::reset() {
    _reset = 1;
    _reset = 0;
}

// Computes and stores the quantization step of the AD7606 as a function of its dynamic range (10 V or 20 V)
void AD7606::setDR(double DR) {
    _q = (double) DR/65535.0; // Quantization step
}
    
// Read raw values from all 8 channels
// rawDataBuffer is a pointer to an array of 8 16-bit integers
void AD7606::readRAW(int16_t *rawDataBuffer) {
    uint8_t lowByte, highByte;

    _convst = 0; // Pulse CONVSTA/CONVSTB to start conversion
    _convst = 1;
    wait_us(5);   
    //while (_busy) {
        // wait for conversions to be completed (low level on BUSY)
    //}
    _cs = 0; // Enable DOUTA/DOUTB lines and shift-out the conversion results
    for (char k = 0;  k < 8; k++) {
        highByte = _spi.write(0x00);    
        lowByte = _spi.write(0x00);
        *(rawDataBuffer + k) = (int16_t) ((highByte << 8) + lowByte);    
    }   
    _cs = 1;
}

// Read analog values from all 8 channels
// analogDataBuffer is a pointer to an array of 8 doubles
void AD7606::readAnalog(double *analogDataBuffer) {
    uint8_t lowByte, highByte;
    
    _convst = 0; // Pulse CONVSTA/CONVSTB to start conversion
    wait_us(1);
    _convst = 1;
    wait_us(5);   
    //while (_busy) {
        // wait for conversions to be completed (low level on BUSY)
    //}
    _cs = 0; // Enable DOUTA/DOUTB lines and shift-out the conversion results
    for (char k = 0;  k < 8; k++) {
        highByte = _spi.write(0x00);    
        lowByte = _spi.write(0x00);
        *(analogDataBuffer + k) = (double) ((int16_t) ((highByte << 8) + lowByte))*_q;    
    }   
    _cs = 1;
}
