/**
 * My custom Xively code. This program will use most of the peripherals
 * on the board. If you need anything else let me know and I can probably
 * make some task for it to do.
 *
 * Eric Fossum
 */
#include "mbed.h"
#include "MMA7660.h"
#include "PinDetect.h"
#include "rtos.h"
#include "ST7565R.h"
#include "tricolor.h"

/// Updates the potentiometers and prints their values to the LCD
void pot_update(void const *args);
/// Updates the devices on the I2C bus and prints their values to the LCD
void i2c_update(void const *args);
/// Function call for when down on the joystick is pressed
void down();
/// Function call for when left on the joystick is pressed
void left();
/// Function call for when the joystick is clicked
void click();
/// Function call for when up on the joystick is pressed
void up();
/// Function call for when right on the joystick is pressed
void right();

Serial pc(USBTX, USBRX);
ST7565R lcd(p11, p8, p7, p5, p6, false);
Tricolor led(p23, p24, p25);
I2C i2c(p28, p27);
MMA7660 accl(p28, p27);

const int temp_addr = 0x90;

Mutex lcd_mutex;

AnalogIn pot1(p19);
AnalogIn pot2(p20);

DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);

PinDetect joy_down(p12);
PinDetect joy_left(p13);
PinDetect joy_click(p14);
PinDetect joy_up(p15);
PinDetect joy_right(p16);

/** Main
 * Sets up the hardware and threads, then toggles the LED every half a second.
 */
int main() {
    joy_left.mode(PullDown);
    joy_right.mode(PullDown);
    joy_click.mode(PullDown);
    joy_up.mode(PullDown);
    joy_down.mode(PullDown);
    
    joy_left.attach_asserted(&left);
    joy_right.attach_asserted(&right);
    joy_click.attach_asserted(&click);
    joy_up.attach_asserted(&up);
    joy_down.attach_asserted(&down);
    
    wait(0.2);
    
    joy_left.setSampleFrequency();
    joy_right.setSampleFrequency();
    joy_click.setSampleFrequency();
    joy_up.setSampleFrequency();
    joy_down.setSampleFrequency();
    
    led.SetLEDColor(0, 60, 0);
    
    pc.printf("Debug out\n");
    
    lcd.moveto(0,0);
    lcd.printf("Custom Xively!");
    // Create Tasks - task, argument, priority, stack, stk_ptr
    Thread pot_thread(pot_update);
    Thread i2c_thread(i2c_update);
    
    while(1) {
        wait(0.5);
        led.Toggle();
    }
}

void pot_update(void const *args) {
    char str[lcd.columns()+1];
    
    while(true) {
        lcd_mutex.lock();
        lcd.moveto(0,1);
        sprintf(str, "Pot1: %00.2f Pot2: %00.2f", pot1.read(), pot2.read());
        lcd.printf(str);
        lcd_mutex.unlock();
    }
}

void i2c_update(void const *args) {
    char str[lcd.columns()+1];
    char cmd[2];
    float temp;
    
    while(true) {
        // Temp Sensor
        cmd[0] = 0x01;
        cmd[1] = 0x00;
        i2c.write(temp_addr, cmd, 2);
        wait(0.5);
        cmd[0] = 0x00;
        i2c.write(temp_addr, cmd, 1);
        i2c.read(temp_addr, cmd, 2);
        temp = ((float((cmd[0]<<8)|cmd[1]) / 256.0)*1.8) + 32;

        lcd_mutex.lock();
        lcd.moveto(0,2);
        sprintf(str, "Temp: %2.1f, Z: % 2.1f", temp, lcd.rows());
        lcd.printf(str);
        
        // Accelerometer
        lcd.moveto(0,3);
        sprintf(str, "X: % 2.1f Y: % 2.1f", accl.x(), accl.y());
        lcd.printf(str);
        lcd_mutex.unlock();
    }
}

void down() {
    led3 = !led3;
}

void left() {
    led1 = !led1;
}

void click() {
    led1 = !led1;
    led2 = !led2;
    led3 = !led3;
    led4 = !led4;
}

void up() {
    led2 = !led2;
}

void right() {
    led4 = !led4;
}
