#ifndef TRICOLOR_H
#define TRICOLOR_H

#include "mbed.h"

/** Tricolor LED driver
 * 
 * This is a reentrant LED controller
 *
 * @code
#include "mbed.h"
#include "tricolor.h"

Tricolor led(p23, p24, p25); // red, green, blue

int main() {
    led.SetLEDColor(0, 100, 0); // Green @ about 40%
    
    while(1) {
        wait(0.5);
        led.Toggle();
    }
}
 * @endcode
 */
class Tricolor
{
    public:
        /** Basic Constructor
         */
        Tricolor(PinName r, PinName g, PinName b) : _Red(r), _Green(g), _Blue(b) {
            SetLEDColor(0, 0, 0);
        }
        /** Color Constructor
         *  Adds color values to constructor.
         */
        Tricolor(PinName r, uint8_t r_color,
                 PinName g, uint8_t g_color,
                 PinName b, uint8_t b_color) :
                 _Red(r), _Green(g), _Blue(b) {
            SetLEDColor(r_color, g_color, b_color);
        }
        /// Sets LED to blue (LED will be on afterwards)
        void Blue(void);
        /// Sets LED to green (LED will be on afterwards)
        void Green(void);
        /// Turns the LED off (saves color)
        void LEDOff(void);
        /// Restores LED to set color (color MUST be set beforehand)
        void LEDOn(void);
        /// Sets LED to red (LED will be on afterwards)
        void Red(void);
        /** Sets custom color for LED (LED will be on afterwards)
         * @param red Amount of red color (0-255)
         * @param green Amount of green color (0-255)
         * @param blue Amount of blue color (0-255)
         */
        void SetLEDColor(uint8_t, uint8_t, uint8_t);
        /// Toggles the LED on and off
        void Toggle(void);
        
    private:
        // LED Output
        bool _on;       // Is the LED on
        PwmOut _Red;    // LED red pin
        PwmOut _Green;  // LED green pin
        PwmOut _Blue;   // LED blue pin
        
        // LED Color
        float _RedPwm;  // Amount of red
        float _GreenPwm;// Amount of green
        float _BluePwm; // Amount of blue
};

#endif