#ifndef SerialLCD_h
#define SerialLCD_h

#include "mbed.h"

#define LCD_BAUD 9600

// Commands
#define LCD_BACKLIGHT       0x80
#define LCD_CLEARDISPLAY    0x01
#define LCD_CURSORSHIFT     0x10
#define LCD_DISPLAYCONTROL  0x08
#define LCD_ENTRYMODESET    0x04
#define LCD_FUNCTIONSET     0x20
#define LCD_SETCGRAMADDR    0x40
#define LCD_SETDDRAMADDR    0x80
#define LCD_SETSPLASHSCREEN 0x0A
#define LCD_SPLASHTOGGLE    0x09
#define LCD_RETURNHOME      0x02

// Flags for display entry mode
#define LCD_ENTRYRIGHT      0x00
#define LCD_ENTRYLEFT       0x02

// Flags for display on/off control
#define LCD_BLINKON         0x01
#define LCD_CURSORON        0x02
#define LCD_DISPLAYON       0x04

// Flags for display size
#define LCD_2LINE           0x02
#define LCD_4LINE           0x04
#define LCD_16CHAR          0x10
#define LCD_20CHAR          0x14

//  Flags for setting display size
#define LCD_SET2LINE        0x06
#define LCD_SET4LINE        0x05
#define LCD_SET16CHAR       0x04
#define LCD_SET20CHAR       0x03

// LCD Types
#define LCD_2X16            3
#define LCD_2X20            4
#define LCD_4X16            5
#define LCD_4X20            6

/**
 * SparkFun SerLCD v2.5 Controller
 * 
 * Copied from the Arduino.cc SerLCD library.
 * @see http://playground.arduino.cc/Code/SerLCD
 */
class SerialLCD : public Serial
{
  public:
    /** Constructor.
     * @param tx Connected to rx pin of LCD
     * @param type Type of LCD (Optional default 16x2) 
     */
    SerialLCD(PinName tx, uint8_t type = 3);
    
    /** Blinks the cursor. This blinks the whole
     *  box or just the cursor bar.
     */
    void blink();
    
    /** Clears the whole LCD and sets the
     *  cursor to the first row, first column.
     */
    void clear();
    /** Clears the current line and resets
     *  the cursor to the beginning of the
     *  line.
     */
    void clearLine(uint8_t);
    /** Creates a custom character for the
     *  LCD.
     * @param location Location to store the char (1-8)
     * @param data Byte array containing the enabled pixel bits
     * @code
     *  #include <NewSoftSerial.h>
     *  #include <serLCD.h>
     *  
     *  // Set pin to the LCD's rxPin
     *  int pin = 2;
     *   
     *  serLCD lcd(pin);
     *   
     *  byte bars[8][8] = {
     *    {B00000,B00000,B00000,B00000,B00000,B00000,B00000,B11111},
     *    {B00000,B00000,B00000,B00000,B00000,B00000,B11111,B11111},
     *    {B00000,B00000,B00000,B00000,B00000,B11111,B11111,B11111},
     *    {B00000,B00000,B00000,B00000,B11111,B11111,B11111,B11111},
     *    {B00000,B00000,B00000,B11111,B11111,B11111,B11111,B11111},
     *    {B00000,B00000,B11111,B11111,B11111,B11111,B11111,B11111},
     *    {B00000,B11111,B11111,B11111,B11111,B11111,B11111,B11111},
     *    {B11111,B11111,B11111,B11111,B11111,B11111,B11111,B11111}
     *  };
     *   
     *  void main() {
     *    for (int i=1; i < 9; i++){
     *      lcd.createChar(i, bars[i-1]);
     *    }
     *    for (int i=1; i < 9; i++){
     *      lcd.printCustomChar(i);
     *    }
     *    while(1);
     *  }
     *  
     * @endcode
     */
    void createChar(uint8_t, uint8_t[]);
    /** Display the LCD cursor: an underscore (line) at the
     *  position to which the next character will be written.
     *  This does not define if it blinks or not.
     */
    void cursor();
    /** Turns on the LCD display, after it's been turned off
     *  with noDisplay(). This will restore the text (and
     *  cursor) that was on the display.
     */
    void display();
    /** Positions the cursor in the upper-left of the LCD.
     */
    void home();
    /** Set the direction for text written to the LCD to
     *  left-to-right, the default. This means that subsequent
     *  characters written to the display will go from left to
     *  right, but does not affect previously-output text.
     */
    void leftToRight();
    /** Turns off the blinking LCD cursor.
     */
    void noBlink();
    /** Hides the LCD cursor.
     */
    void noCursor();
    /** Turns off the LCD display, without losing the text
     *  currently shown on it.
     */
    void noDisplay();
    /** Print a custom character (gylph) on to the LCD. Up
     *  to eight characters of 5x8 pixels are supported
     *  (numbered 1 to 8). The custom characters are designed
     *  by the createChar() command.
     * @param location Which character to print (1 to 8), created by createChar()
     */
    void printCustomChar(uint8_t);
    /** Set the direction for text written to the LCD to 
     *  right-to-left (the default is left-to-right). This
     *  means that subsequent characters written to the 
     *  display will go from right to left, but does not
     *  affect previously-output text.
     */
    void rightToLeft();
    /** Scrolls the text on the LCD one position to the left.
     */
    void scrollLeft();
    /** Scrolls the text on the LCD one position to the right.
     */
    void scrollRight();
    /** Moves the cursor to the beginning of selected line.
     *  Line numbers start with 1. Valid line numbers are 1-4.
     * @param line Line number to go to
     */
    void selectLine(uint8_t line);
    /** Sets the backlight brightness based on input value.
     *  Values range from 1-30.
     * @params brightness 1 = Off -> 30 = Full Brightness
     */
    void setBrightness(uint8_t num);
    /** Position the LCD cursor.
     * @param row Row position of the cursor (1 being the first row)
     * @param col Column position the cursor (1 being the first column)
     */
    void setCursor(uint8_t row, uint8_t col);
    /** Saves the first 2 lines of text that are currently 
     *  printed on the LCD screen to memory as the new splash screen.
     * @code
     *  void setup()
     *  {
     *     lcd.print("      New       ");
     *     lcd.print(" Splash Screen! ");
     *     lcd.setSplash();
     *  }
     * @endcode
     */
    void setSplash();
    /** The SerLCD firmware v2.5 supports setting different types
     *  of LCD's. This function allows to easily set the type of LCD.
     *
     *  The function expects a parameter value of 3 - 6:
     *  Defines:
     *    LCD_2X16(3) - 2x16 LCD
     *    LCD_2X20(4) - 2x20 LCD
     *    LCD_4X16(5) - 4x16 LCD
     *    LCD_4X20(6) - 4x20 LCD
     *
     *  After changing the settings, the value is written to the EEPROM
     *  so the SerialLCD will remember it even after a power-off.
     */
    void setType(uint8_t type);
    /** Toggles the splash screen on/off.
     */
    void toggleSplash();
    
private:
    void command(uint8_t);
    void specialCommand(uint8_t);

    uint8_t _displayfunction;
    uint8_t _displaycontrol;
    uint8_t _displaymode;
    uint8_t _numlines;
    uint8_t _numchars;
    uint8_t _rowoffset;
};

#endif
