#include "SerialLCD.h"

SerialLCD::SerialLCD(PinName tx, uint8_t type) : 
    Serial(tx, NC)
{
    // Init
    baud(LCD_BAUD);
    setType(type);
    
    _rowoffset = 0;
    
    // Reset
    clear();
    setBrightness(30);
}

void SerialLCD::clear()
{
    command(LCD_CLEARDISPLAY);
}

void SerialLCD::clearLine(uint8_t line)
{
    if(line > 0 && line <= _numlines){
        setCursor(line, 1);
        printf("                ");
        setCursor(line, 1);
    }
}

void SerialLCD::selectLine(uint8_t line)
{
    if(line > 0 && line <= _numlines){
        setCursor(line, 1);
    }
}

void SerialLCD::setBrightness(uint8_t num)
{
    if(num >= 1 && num <= 30){
        specialCommand(LCD_BACKLIGHT | (num - 1));
    }
}

void SerialLCD::home()
{
    command(LCD_RETURNHOME);
}

void SerialLCD::setSplash()
{
    specialCommand(LCD_SETSPLASHSCREEN);
}

void SerialLCD::setType(uint8_t type)
{
/*
  3: type 2x16
  4: type 2x20
  5: type 4x16
  6: type 4x20
*/
    specialCommand(type);
    switch (type) {
      case 3:
        _numlines = LCD_2LINE;
        _numchars = LCD_16CHAR;

        break;
      case 4:
        _numlines = LCD_2LINE;
        _numchars = LCD_20CHAR;

        break;
      case 5:
        _numlines = LCD_4LINE;
        _numchars = LCD_16CHAR;

        break;
      case 6:
        _numlines = LCD_4LINE;
        _numchars = LCD_20CHAR;

        break;

      default:
        _numlines = LCD_2LINE;
        _numchars = LCD_16CHAR;
    }
}

void SerialLCD::toggleSplash()
{
    specialCommand(LCD_SPLASHTOGGLE);
}

void SerialLCD::leftToRight()
{
    _displaymode |= LCD_ENTRYLEFT;
    command(LCD_ENTRYMODESET | _displaymode);
}

void SerialLCD::rightToLeft()
{
    _displaymode &= ~LCD_ENTRYLEFT;
    command(LCD_ENTRYMODESET | _displaymode);
}

void SerialLCD::blink()
{
    _displaycontrol |= LCD_BLINKON;
    command(LCD_DISPLAYCONTROL | _displaycontrol);
}

void SerialLCD::noBlink()
{
    _displaycontrol &= ~LCD_BLINKON;
    command(LCD_DISPLAYCONTROL | _displaycontrol);
}

void SerialLCD::cursor()
{
    _displaycontrol |= LCD_CURSORON;
    command(LCD_DISPLAYCONTROL | _displaycontrol);
}

void SerialLCD::noCursor()
{
    _displaycontrol &= ~LCD_CURSORON;
    command(LCD_DISPLAYCONTROL | _displaycontrol);
}

void SerialLCD::display()
{
    _displaycontrol |= LCD_DISPLAYON;
    command(LCD_DISPLAYCONTROL | _displaycontrol);
}

void SerialLCD::noDisplay()
{
    _displaycontrol &= ~LCD_DISPLAYON;
    command(LCD_DISPLAYCONTROL | _displaycontrol);
}

void SerialLCD::setCursor(uint8_t row, uint8_t col)
{
    int row_offsets[2][4] = {
        { 0x00, 0x40, 0x10, 0x50 },
        { 0x00, 0x40, 0x14, 0x54 }
    };
    if((row > 0 && row < 3) && (col > 0 && col < 17)){
        command(LCD_SETDDRAMADDR | ((col - 1) + row_offsets[_rowoffset][(row - 1)]));
    }
}

void SerialLCD::createChar(uint8_t location, uint8_t charmap[])
{
    location -= 1;
    location &= 0x07;
    for (int i=0; i<8; i++){
        command(LCD_SETCGRAMADDR | (location << 3) | i);
        putc(charmap[i]);
    }
}

void SerialLCD::printCustomChar(uint8_t num)
{
    putc((num - 1));
}

void SerialLCD::scrollLeft()
{
    command(0x18);
}

void SerialLCD::scrollRight()
{
    command(0x1C);
}

// Private Functions

void SerialLCD::command(uint8_t value)
{
    putc(0xFE);
    putc(value);
    wait_ms(5);
}

void SerialLCD::specialCommand(uint8_t value)
{
    putc(0x7C);
    putc(value);
    wait_ms(5);
}
