#include "States.h"
#include "mbed.h"

////////////////////////////////////////////////////////////////
// CONSTRUCTOR
////////////////////////////////////////////////////////////////
Gameplay::Gameplay(uLCD_4DGL &uLCD, int numberOfPlayers, PinDetect &P1_Left, PinDetect &P1_Right, PinDetect &P2_Left, PinDetect &P2_Right)
{   
    uLCD.background_color(LGREY);
    uLCD.cls();
    //random seed setup
    srand(time(NULL));
    
    numPlayers = numberOfPlayers;
    points = new int[2];
    CPU cpu;
    
    //LED's used to indicate which player wins each round
    DigitalOut led1(LED1);
    DigitalOut led4(LED4);

    //LCD setup
    uLCD.color(BLACK);
    uLCD.textbackground_color(LGREY);
    uLCD.set_font(FONT_7X8);

    //points setup
    resetPoints(uLCD);
    
    //it takes 5-6 seconds for guns to get drawn, but we only draw
    //them once at the beginning of the game-- so that's okay
    drawGun(uLCD, 10, 30, true);
    drawGun(uLCD, 10, 92, true);
    
    drawGun(uLCD, 127 - 10 - 42, 30, false);
    drawGun(uLCD, 127 - 10 - 42, 92, false);
    
    //loop until someone runs out of points
    while(getPoints(1) != 0 && getPoints(2) != 0)
    {
        //gameplay here
        
        //generate which prompt will be given
        Prompt prompt = (Prompt)(rand() % 4);
        
        //display countdown
        countdown(uLCD);   
        
        //clear the countdown     
        clearCountdown(uLCD);
        
        //show the prompt, and listen for inputs
        if(prompt == HoldFire)
        {
            //hold fire
            displayXs(uLCD);
            
            Timer timer;
            timer.start();
            
            //hold fire lasts for 2 seconds
            //if a player fires during this 2 seconds
            //they lose a point
            while(timer.read_ms() < 2000)
            {
                if(!P1_Left || !P1_Right)
                {
                    setPoints(1, getPoints(1) - 1, uLCD);
                    led4 = 1;
                    break;
                }
                
                //AI will never mis-fire on hold-fire... just for simplicity's sake
                if(numPlayers == 2 && (!P2_Left || !P2_Right))
                {
                    setPoints(2, getPoints(2) - 1, uLCD);
                    led1 = 1;
                    break;
                }
            }
        }
        else
        {
            if(prompt == Either)
            {
                displayCircle(uLCD);
            }
            else if(prompt == Down)
            {
                displayDownArrow(uLCD);
            }
            else if(prompt == Up)
            {
                displayUpArrow(uLCD);
            }
            
            int aiShootTime = 0;
            
            if(numPlayers == 1)
                aiShootTime = (int)(cpu.shootTime() * 1000);
                
            Timer timer;
            timer.start();
            
            //these variables act as a layer of indirection between the button being pressed
            //and the value being read by the program. This allows us to "inject" fake
            //button presses for the AI without having to actually have a button signal
            //be sent by the hardware
            bool p1l = false, p1r = false, p2l = false, p2r = false;
            
            //wait for button to be pressed
            while(true)
            {
                //if the predetermined ai wait time has elapsed, input the AI's button press
                if(numPlayers == 1 && timer.read_ms() > aiShootTime)
                {
                    bool aiCorrect = cpu.shootAnswer(prompt);
                    
                    if(prompt == Up)
                    {
                        if(aiCorrect)
                        {
                            p2r = true;
                            p2l = false;
                        }
                        else
                        {
                            p2l = true;
                            p2r = false;
                        }
                    }
                    else if (prompt == Down)
                    {
                        if(aiCorrect)
                        {
                            p2l = true;
                            p2r = false;
                        }
                        else
                        {
                            p2r = true;
                            p2l = false;
                        }
                    }
                    else
                    {
                        //prompt = either, just press left since either one works for this prompt
                        p2l = true;
                        p2r = false;
                    }
                    
                    break;
                }
                
                if(!P1_Left)
                {
                    p1l = true;
                    p1r = false;
                    p2l = false;
                    p2r = false;
                    break;
                }
                
                if(!P1_Right)
                {
                    p1l = false;
                    p1r = true;
                    p2l = false;
                    p2r = false;
                    break;
                }
                
                //only read p2 buttons if its a 2 player game
                if(numPlayers == 2 && !P2_Left)
                {
                    p1l = false;
                    p1r = false;
                    p2l = true;
                    p2r = false;
                    break;
                }
                
                if(numPlayers == 2 && !P2_Right)
                {
                    p1l = false;
                    p1r = false;
                    p2l = false;
                    p2r = true;
                    break;
                }
            }
            
            
            //check the button presses against the prompt given. decrement points accordingly
            if(p1l)
            {
                if(prompt == Either || prompt == Up)
                {
                    setPoints(2, getPoints(2) - 1, uLCD);
                    led1 = 1;
                }
                else
                {
                    setPoints(1, getPoints(1) - 1, uLCD);
                    led4 = 1;
                }
            }
            else if(p1r)
            {
                if(prompt == Either || prompt == Down)
                {
                    setPoints(2, getPoints(2) - 1, uLCD);
                    led1 = 1;
                }
                else
                {
                    setPoints(1, getPoints(1) - 1, uLCD);
                    led4 = 1;
                }
            }
            else if(p2l)
            {
                if(prompt == Either || prompt == Down)
                {
                    setPoints(1, getPoints(1) - 1, uLCD);
                    led4 = 1;
                }
                else
                {
                    setPoints(2, getPoints(2) - 1, uLCD);
                    led1 = 1;
                }
            }
            else if(p2r)
            {
                if(prompt == Either || prompt == Up)
                {
                    setPoints(1, getPoints(1) - 1, uLCD);
                    led4 = 1;
                }
                else
                {
                    setPoints(2, getPoints(2) - 1, uLCD);
                    led1 = 1;
                }
            }
        }
        
        //remove the displayed prompt. Wait 2 seconds to give the players some
        //amount of rest time between rounds
        clearPrompt(uLCD);
        Thread::wait(2000);
        led1 = 0;
        led4 = 0;
    }
    
    //store the winning player in a variable so the main method can create the correct GameOver screen
    if(getPoints(1) == 0)
        winningPlayer = 2;
    else
        winningPlayer = 1;
}

////////////////////////////////////////////////////////////////
// GET FUNCTIONS
////////////////////////////////////////////////////////////////
int Gameplay::getPoints(int player)
{
    return points[player - 1];
}

int Gameplay::getWinningPlayer()
{
    return winningPlayer;
}

////////////////////////////////////////////////////////////////
// SET FUNCTIONS
////////////////////////////////////////////////////////////////
void Gameplay::resetPoints(uLCD_4DGL &uLCD)
{
    points[0] = 5;
    points[1] = 5;
    renderScore(uLCD);
}

void Gameplay::setPoints(int player, int value, uLCD_4DGL &uLCD)
{
    points[player - 1] = value;
    renderScore(uLCD);
}

////////////////////////////////////////////////////////////////
// PROTECTED FUNCTIONS
////////////////////////////////////////////////////////////////

void Gameplay::drawGun(uLCD_4DGL &uLCD, int x, int y, bool facingRight)
{
    //image arrays generated using a separate Java program (for ease of operating with images)
    if(facingRight)
    {
        int colors[] = {12566463, 12566463, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -5023488, -5023488, -5023488, -5023488, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -5023488, -5023488, -5023488, -5023488, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -5023488, -5023488, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, -5023488, -5023488, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -8372224, -8372224, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -8372224, -8372224, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -11382190, -11382190, -11382190, -11382190, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -11382190, -11382190, -11382190, -11382190, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463};
        uLCD.BLIT(x, y, 42, 20, colors);
    }
    else
    {
        int colors[] = {12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -11382190, -11382190, -11382190, -11382190, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -11382190, -11382190, -11382190, -11382190, -11382190, -11382190, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -8372224, -8372224, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -8372224, -8372224, -13158601, -13158601, -13158601, -13158601, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -5023488, -5023488, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, -5023488, -5023488, -8372224, -8372224, -8372224, -8372224, -8372224, -8372224, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, 12566463, 12566463, -16777216, -16777216, -5023488, -5023488, -5023488, -5023488, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -5023488, -5023488, -5023488, -5023488, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, -16777216, -16777216, -16777216, -16777216, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463, 12566463};
        uLCD.BLIT(x, y, 42, 20, colors);
    }
}

void Gameplay::displayUpArrow(uLCD_4DGL &uLCD)
{
    uLCD.line(64, 64, 64, 30, GREEN);
    uLCD.line(54, 40, 64, 30, GREEN);
    uLCD.line(74, 40, 64, 30, GREEN);
}

void Gameplay::displayDownArrow(uLCD_4DGL &uLCD)
{
    uLCD.line(64, 64, 64, 98, GREEN);
    uLCD.line(54, 88, 64, 98, GREEN);
    uLCD.line(74, 88, 64, 98, GREEN);
}

void Gameplay::displayCircle(uLCD_4DGL &uLCD)
{
    uLCD.circle(64, 64, 15, GREEN);   
}

void Gameplay::displayXs(uLCD_4DGL &uLCD)
{
    //top X
    uLCD.line(54, 40, 74, 20, RED);
    uLCD.line(54, 20, 74, 40, RED);
    
    //bottom X
    uLCD.line(54, 88, 74, 108, RED);
    uLCD.line(54, 108, 74, 88, RED);
}

//clears the arrow/circle/x currently on the screen
void Gameplay::clearPrompt(uLCD_4DGL &uLCD)
{
    //up arrow
    uLCD.line(64, 64, 64, 30, LGREY);
    uLCD.line(54, 40, 64, 30, LGREY);
    uLCD.line(74, 40, 64, 30, LGREY);
    
    //down arrow
    uLCD.line(64, 64, 64, 98, LGREY);
    uLCD.line(54, 88, 64, 98, LGREY);
    uLCD.line(74, 88, 64, 98, LGREY);
    
    //circle
    uLCD.circle(64, 64, 15, LGREY);
    
    //top X
    uLCD.line(54, 40, 74, 20, LGREY);
    uLCD.line(54, 20, 74, 40, LGREY);
    
    //bottom X
    uLCD.line(54, 88, 74, 108, LGREY);
    uLCD.line(54, 108, 74, 88, LGREY);
}

//Displays 3..2..1.. on screen
void Gameplay::countdown(uLCD_4DGL &uLCD)
{
    uLCD.locate(5, 8);
    uLCD.printf("3");
    Thread::wait(333);
    uLCD.printf(".");
    Thread::wait(333);
    uLCD.printf(".");
    Thread::wait(333);
    uLCD.printf("2");
    Thread::wait(333);
    uLCD.printf(".");
    Thread::wait(333);
    uLCD.printf(".");
    Thread::wait(333);
    uLCD.printf("1");
    Thread::wait(333);
    uLCD.printf(".");
    Thread::wait(333);
    uLCD.printf(".");
    Thread::wait(333);
}

//clears the 3..2..1.. display
void Gameplay::clearCountdown(uLCD_4DGL &uLCD)
{
    uLCD.filled_rectangle(30, 50, 98, 78, LGREY);
}

//re-draws the score on the screen
void Gameplay::renderScore(uLCD_4DGL &uLCD)
{    
    uLCD.locate(3, 0);
    uLCD.printf("P1: %d", getPoints(1));
    
    uLCD.locate(10, 0);
    uLCD.printf("P2: %d", getPoints(2));
}