/*
 * - Proyecto:      SABMiller ILC
                    Velocímetro digital
                    CONDUSE

 * - Lenguaje:      ANSI C/C++ (mbed)
 * - Tarjeta:       Nucleo F091RC
 * - Referencias:
 * - Fecha:         2016/Agosto
 * - Autor(es):     Felícito Manzano /
                    Mario Vargas
 * - Compañia:      V.S.R. de Centroamérica
 * - País:          SV / CR
*/

/* ***********************************************
******* ARCHIVO CON FUNCIONES PARA CONDUSE *******
************************************************** */

#include "constantes.hpp"
#include "BufferedSerial.h"

/* ***********************************************
*******   CONTROL PARA COLORES RGB EN LED  *******
************************************************** */

// https://developer.mbed.org/users/4180_1/notebook/rgb-leds/

//Class to control an RGB LED using three PWM pins
class RGBLed
{
public:
    RGBLed(PinName redpin, PinName greenpin, PinName bluepin);
    void write(float red,float green, float blue);
private:
    PwmOut _redpin;
    PwmOut _greenpin;
    PwmOut _bluepin;
};

RGBLed::RGBLed (PinName redpin, PinName greenpin, PinName bluepin)
    : _redpin(redpin), _greenpin(greenpin), _bluepin(bluepin)
{
    //50Hz PWM clock default a bit too low, go to 2000Hz (less flicker)
    _redpin.period(0.0005);
}

void RGBLed::write(float red,float green, float blue)
{
    _redpin = red;
    _greenpin = green;
    _bluepin = blue;
}

/* ***********************************************
*******    LISTADO DE FUNCIONES CONDUSE    *******
************************************************** */

int iniciar_data_cp(int data[35])
{
    /*
    Esta función recibe un arreglo, donde se contendrá la trama entrante del CP
    la función se encarga de almacenar 0 en cada posición del arreglo. La función
    siempre retorna 0.
    */
    int x;
    for (x = 0; x < 35; x++) {
        data[x] = 0;
    }
    return(0);
}

int iniciar_display7s(DigitalOut *d7_cen, DigitalOut *d7_dec, DigitalOut *d7_uni)
{
    /*
    Esta función se encarga de presentar en el display de 7 segmentos el mensaje:
    " --" para indicar que se está iniciando el sensor. La función siempre retorna 0
    */
    int x;

    // Apagar el digito más significativo
    d7_cen[0] = CIEN_MENOR[0];
    d7_cen[1] = CIEN_MENOR[1];

    // DECENAS
    for (x=0; x<7; x++) {
        d7_dec [x] = GUION[x];
    }

    // Digito menos significativo (UNIDADES)
    for (x=0; x<7; x++) {
        d7_uni [x] = GUION[x];
    }

    return(0);
}

int consultar_conduse(BufferedSerial *puerto_CP)
{
    /*
    Esta función recibe el puerto serial del CP ConduSe y se encarga de transmitir
    la trama de consulta por la velocidad.
    Siempre retorna 0
    */
    int x;
    for (x = 0; x < 9; x++) {
        puerto_CP -> putc(QUERRY_CONDUSE[x]);
    }

    return(0);
}

int recibir_respuesta_conduse(BufferedSerial *puerto_CP, int data[35])
{
    /*
    Esta función recibe el puerto serial del CP conduSe más un arreglo entero
    para almacenar los datos recibidos.
    Esta función retorna 1 cuando se reciben datos. Retorna 0 cuando no hay
    datos por recibir en el puerto serial.
    */

    int x=0;
    int y=0;

    if (puerto_CP -> readable()) {
        y=1;
        wait_ms(30);
        while (puerto_CP -> readable()) {
            data[x] = puerto_CP -> getc();
            x++;

            if (x >= 34) {
                x=0;
            }
        }
    }
    return(y);

}

int validar_trama_conduse(int data[35])
{
    /*
    Esta función recibe los datos recibidos en la trama conduse y verifica si es
    una trama valida para continuar analizandola. Si la trama es valida devuelve
    1 pero si es invalida devuelve 0.
    */

    if ((data[CONDUSE_VAL0] == BYTE0_CONDUSE) &&
        (data[CONDUSE_VAL1] == BYTE1_CONDUSE)) {
        return(1);
    } else {
        return(0);
    }
}


int extraer_velocidad_conduse(int data[35], float *velocidad)
{
    /*
    Esta función se encarga de extraer el byte de la trama recibida que corresponde
    a la velocidad GPS y multiplicarlo por el factor de conversión de KNOTS (Nudos)
    a Km/h. La función siempre retorna 0.
    */
    *velocidad = (float)data[CONDUSE_VELOCIDAD] * KNOTS;
    return(0);
}

int analizar_velocidad_conduse(float *velocidad, int *cent, int *dec, int *uni)
{
    /*
    Esta función recibe una variable flotante con la velocidad gps expresada en
    Km/h y la separa en 3 enteros que representan las centenas, decenas y unidades.
    La función siempre retorna 0.
    */

    if (*velocidad >= 100) {
        *cent =  (((int)*velocidad)/ 100 % 10);
        *dec = (((int)*velocidad)/ 10 % 10);
        *uni = (((int)*velocidad) % 10);

    } else if (*velocidad >= 10) {
        *cent = 0;
        *dec = (((int)*velocidad) / 10 % 10);
        *uni = (((int)*velocidad) % 10);
    } else {
        *cent = 0;
        *dec = 0;
        *uni = (int) *velocidad;
    }
    return(0);
}

int presentar_velocidad(int *cen, int *dec, int *uni, DigitalOut *d7_cen,
                        DigitalOut *d7_dec, DigitalOut *d7_uni)
{
    /*
    Esta función recibe los digitos correspondientes a las decenas, centenas y
    unidades que representan la velocidad y que deben presentarse en los tres
    display de siete segmentos. La función siempre retorna 0.
    */

    int x;

    // Digito más significativo (CENTERNARES)
    if (*cen) {
        d7_cen[0] = CIEN_MAYOR[0];
        d7_cen[1] = CIEN_MAYOR[1];
    } else {
        d7_cen[0] = CIEN_MENOR[0];
        d7_cen[1] = CIEN_MENOR[1];
    }

    // DECENAS
    for (x=0; x<7; x++) {
        d7_dec [x] = DIGITOS[*dec] [x];
    }

    // Digito menos significativo (UNIDADES)
    for (x=0; x<7; x++) {
        d7_uni [x] = DIGITOS[*uni] [x];
    }

    return(0);
}


int mostrar_error7s(DigitalOut *d7_cen, DigitalOut *d7_dec, DigitalOut *d7_uni)
{
    /*
    Esta función se encarga de presentar en el display de 7 segmentos el mensaje:
    " EE" para indicar que se ha generado un Error. La función siempre retorna 0
    */
    int x;

    // Apagar el digito más significativo
    d7_cen[0] = CIEN_MENOR[0];
    d7_cen[1] = CIEN_MENOR[1];

    // DECENAS
    for (x=0; x<7; x++) {
        d7_dec [x] = ERROR_D7S[x];
    }

    // Digito menos significativo (UNIDADES)
    for (x=0; x<7; x++) {
        d7_uni [x] = ERROR_D7S[x];
    }

    return(0);
}

int actualizar_led(float *velocidad, DigitalOut *buzz, RGBLed *myRGBled)
{
    /*
    Esta función recibe el valor de la velocidad y en base al valor activa las
    salidas PWM y el Buzzer. La función siempre retorna 0.
    */

    if (*velocidad < VEL_RALENTI) { // Color azul

        myRGBled -> write(COLORES [0] [0], COLORES [0] [1], COLORES [0] [2]);
        //buzz = 0;

    } else if ((*velocidad >= (VEL_ADVERTENCIA-10)) &&
               (*velocidad < VEL_ADVERTENCIA)) { // Color amarillo

        myRGBled -> write(COLORES [2] [0], COLORES [2] [1], COLORES [2] [2]);
        //buzz = 0;

    } else if ((*velocidad >= VEL_ADVERTENCIA) &&
               (*velocidad < VEL_LIMITE)) { // Color ambar + slow beep

        myRGBled -> write(COLORES [3] [0], COLORES [3] [1], COLORES [3] [2]);
        //buzz = 1;

    } else if (*velocidad >= VEL_LIMITE) { // Color rojo + fast beep

        myRGBled -> write(COLORES [4] [0], COLORES [4] [1], COLORES [4] [2]);
        //buzz = 1;

    } else { // color verde

        myRGBled -> write(COLORES [1] [0], COLORES [1] [1], COLORES [1] [2]);
        buzz = 0;
    }

    return(0);
}