/**
 * @file f_basicas.cpp
 * @author Felícito Manzano (felicito.manzano@detektor.com.sv)
 * @brief 
 * @version 0.1
 * @date 2021-05-23
 * 
 * @copyright Copyright (c) 2021
 * 
 */

#include "mbed.h"
#include "BufferedSerial.h"
#include "constantes.hpp"
#include "tagid.hpp"
#include "f_basicas.hpp"



/**
 * @brief 
 *  Esta función recibe un entero que es el contador de tramas
    realiza el incremento en uno y verifica si el valor es mayor
    o igual que 65535 que es equivalente a 0xFFFF. Si se cumple
    la condución se regresa el valor a 1.
    Esta función siempre retorna 0.
 * 
 * @param tramas 
 * @return true 
 * @return false 
 */
bool incrementar_trama(int *tramas)
{

    int actual = *tramas;
    int limite = 65535;
    actual++;

    if (actual > limite) {
        *tramas = 1;
    } else {
        *tramas = actual;
    }
    return(false);
}



/**
 * @brief 
 * 
 * @param puerto_usb 
 * @param puerto_cp 
 * @return true 
 * @return false 
 */
bool booting_gtdat(Serial *puerto_usb, Serial *puerto_cp) {
    wait_us(1000000);
    puerto_usb -> printf("\r\n*** DETEKTOR LATAM ***\r\n");
    puerto_usb -> printf("INICIANDO NUCLEO STM32\r\n");
    puerto_usb -> printf("CCN TIEMPO EN PLANTA\r\n");
    puerto_usb -> printf("R.0A.1 - 2021-05-24\r\n\r\n");
    puerto_usb -> printf("BAHIA-%d\r\n", BAY_ADDRESS);
    puerto_cp  -> printf("%s*INICIA BAHIA-%d*%s0001%s\r\n", 
                    CABECERA_TX, BAY_ADDRESS, FIN_CABECERA_TX, ULTIMO_CARACTER);
    wait_us(T_TX);
    return(false);
}



/**
 * @brief 
 * 
 *  Esta función recibe los digitos correspondientes a las horas, decenas y
    minutos que representan la hora y que deben presentarse en los tres
    display de siete segmentos. La función siempre retorna 0.
 * @param horas 
 * @param decenas 
 * @param minutos 
 * @param d7_h 
 * @param d7_d 
 * @param d7_m 
 * @return int 
 */
int presentar_tiempo(int *horas, int *decenas, int *minutos,
                     BusOut *d7_h, BusOut *d7_d, BusOut *d7_m)
{
    // Digito más significativo (HORAS)
    switch (*horas) {
        case 0 :
            d7_h -> write(DIGITOS[CERO_7S]);
            break;
        case 1 :
            d7_h -> write(DIGITOS[UNO_7S]);
            break;
        case 2 :
            d7_h -> write(DIGITOS[DOS_7S]);
            break;
        case 3 :
            d7_h -> write(DIGITOS[TRES_7S]);
            break;
        case 4 :
            d7_h -> write(DIGITOS[CUATRO_7S]);
            break;
        case 5 :
            d7_h -> write(DIGITOS[CINCO_7S]);
            break;
        case 6 :
            d7_h -> write(DIGITOS[SEIS_7S]);
            break;
        case 7 :
            d7_h -> write(DIGITOS[SIETE_7S]);
            break;
        case 8 :
            d7_h -> write(DIGITOS[OCHO_7S]);
            break;
        case 9 :
            d7_h -> write(DIGITOS[NUEVE_7S]);
            break;
        case 10:
            d7_h -> write(DIGITOS[APAGADO_7S]);
            break;
    }

    // DECENAS DE MINUTOS
    switch (*decenas) {
        case 0 :
            d7_d -> write(DIGITOS[CERO_7S]);
            break;
        case 1 :
            d7_d -> write(DIGITOS[UNO_7S]);
            break;
        case 2 :
            d7_d -> write(DIGITOS[DOS_7S]);
            break;
        case 3 :
            d7_d -> write(DIGITOS[TRES_7S]);
            break;
        case 4 :
            d7_d -> write(DIGITOS[CUATRO_7S]);
            break;
        case 5 :
            d7_d -> write(DIGITOS[CINCO_7S]);
            break;
        case 6 :
            d7_d -> write(DIGITOS[SEIS_7S]);
            break;
        case 7 :
            d7_d -> write(DIGITOS[SIETE_7S]);
            break;
        case 8 :
            d7_d -> write(DIGITOS[OCHO_7S]);
            break;
        case 9 :
            d7_d -> write(DIGITOS[NUEVE_7S]);
            break;
        case 10:
            d7_d -> write(DIGITOS[APAGADO_7S]);
            break;
    }

    // Digito menos significativo (MINUTOS  )
    switch (*minutos) {
        case 0 :
            d7_m -> write(DIGITOS[CERO_7S]);
            break;
        case 1 :
            d7_m -> write(DIGITOS[UNO_7S]);
            break;
        case 2 :
            d7_m -> write(DIGITOS[DOS_7S]);
            break;
        case 3 :
            d7_m -> write(DIGITOS[TRES_7S]);
            break;
        case 4 :
            d7_m -> write(DIGITOS[CUATRO_7S]);
            break;
        case 5 :
            d7_m -> write(DIGITOS[CINCO_7S]);
            break;
        case 6 :
            d7_m -> write(DIGITOS[SEIS_7S]);
            break;
        case 7 :
            d7_m -> write(DIGITOS[SIETE_7S]);
            break;
        case 8 :
            d7_m -> write(DIGITOS[OCHO_7S]);
            break;
        case 9 :
            d7_m -> write(DIGITOS[NUEVE_7S]);
            break;
        case 10:
            d7_m -> write(DIGITOS[APAGADO_7S]);
            break;
    }

    return(0);
}

int presentar_torre(const int *color, BusOut *torre_luz)
{
    /*
    Esta función recibe el código de color correspondiente a la torre de luz
    estilo semaforo para activar el LED correspondiente.
    La función siempre retorna 0
    */

    switch (*color) {
        case 0 :
            torre_luz -> write(COLOR_TORRE_LUZ[APAGADO_TL]);
            break;
        case 1 :
            torre_luz -> write(COLOR_TORRE_LUZ[VERDE]);
            break;
        case 2 :
            torre_luz -> write(COLOR_TORRE_LUZ[AMARILLO]);
            break;
        case 3 :
            torre_luz -> write(COLOR_TORRE_LUZ[ROJO]);
            break;
    }

    return(0);
}



/**
 * @brief 
 *  Esta función se encarga de buscar el grupo al que está asignado el TAGRFiD
       que ha sido leido por el lector. Se ha predefinido 3 arreglos con los TAG
       validos. En función del arreglo en que se genere la coincidencia se retorna
       el tiempo para cambiar.
       La función retorna 1 cuando encuentra coincidencia. 0 cuando no encuentra coincidencia y
       se define el tiempo de la torre de luz amarillo y rojo en 50 minutos y 1 hora.
 * 
 * @param id 
 * @param amarillo 
 * @param rojo 
 * @return int 
 */
int tiempo_asignado(char id[9], int *amarillo, int *rojo)
{
    int i = 0;
    int v = 0;
    int busca_id = 0;

    for (i=0; i<TOTAL_PLACAS_A; i++) {
        busca_id = strncmp(PLACAS_A[i], id, 10);
        if (busca_id == 0) {
            *amarillo = GRUPO_A_AMARILLO;
            *rojo = GRUPO_A_ROJO;
            v = 1; return(1);
        }
    }

    for (i=0; i<TOTAL_PLACAS_B; i++) {
        busca_id = strncmp(PLACAS_B[i], id, 10);
        if (busca_id == 0) {
            *amarillo = GRUPO_B_AMARILLO;
            *rojo = GRUPO_B_ROJO;
            v = 1; return(1);
        }
    }

    for (i=0; i<TOTAL_PLACAS_C; i++) {
        busca_id = strncmp(PLACAS_C[i], id, 10);
        if (busca_id == 0) {
            *amarillo = GRUPO_C_AMARILLO;
            *rojo = GRUPO_C_ROJO; 
            v = 1; return(1);
        }
    }

    if (v) {
        return(1);
    } else {
        *amarillo = GRUPO_DEFAULT_AMARILLO;
        *rojo = GRUPO_DEFAULT_ROJO;
        return(0);
    }
}



/**
 * @brief 
 * 
 *  Esta función se encarga de analisar el payload de una
    huella que es enviada dentro de una cadena (char array)
    expresada de manera hexadecimal. en su equivalente de código
    ASCII que es guardado en otra cadena.

 * @param id_original 
 * @param id_hex 
 * @param tamano_id 
 * @return int 
 */
int convertir_id(char id_original[256], char id_hex[15], int *tamano_id)
{
    int i = 0;
    int j = 0;
    int limite = *tamano_id;
    int k = 0;
    char tmp_buffer_ascii[77];

    memset(tmp_buffer_ascii, '\0', sizeof tmp_buffer_ascii);
    memset(id_hex, '\0', sizeof id_hex);

    for (i=0; i<limite; i++) {
        snprintf(&tmp_buffer_ascii[j*2], 3,"%02X", id_original[i]);
        j++;
    }

    limite = strlen(tmp_buffer_ascii);
    k = limite - 14;

    strncpy(id_hex, &tmp_buffer_ascii[k], 14);
    id_hex[15] = '\0';

    return(1);
}



/**
 * @brief 
 *  Esta funsión se utiliza para limpiar el buffer de recepción del puerto seleccionado.
    No se guarda ninguna información. Siempre devuelve true
 * 
 * @param uart 
 * @return true 
 * @return false 
 */
bool flush_uart_rx(BufferedSerial *uart)
{
    while (uart -> readable()) {
        uart -> getc();
    }    
    return(true);
}






/**
 * @brief 
 * 
 */
extern Serial  pcusb;
extern BusOut  torreLuz;
extern BusOut  display_H;
extern BusOut  display_dM;
extern BusOut  display_uM;
extern Ticker  ticker_minuto; 
extern Timer   t_ocupado;
extern Timer   t_apagado;
extern float   tiempo_actual;
extern bool    mostrar_tiempo;
extern char    actual_trailerID_HEX[];
extern char    antena_trailerID_HEX[];
extern int     rtc_delta;
extern int     uminutos;
extern int     dminutos;
extern int     horas;
extern int     tiempo_luz_amarilla;
extern int     tiempo_luz_roja;

void actualizar_minuto() {
    // ACTUALIZACIÓN DE PANTALLA DE CRONOMETRO
    uminutos++; // Acumular un minuto
    if (uminutos >= 10) { // Si es 10 cambiar 0
        uminutos = 0;
        dminutos++; // Acumularse las decenas de minutos
    }
    if (dminutos >= 6) { // Si es 6 cambiar 0
        dminutos = 0; // Acumular una hora
        horas++;
    }

    if (mostrar_tiempo) {
        rtc_delta = (int)t_ocupado.read();
        pcusb.printf("\r\nTiempo total en Bahia: %d segundos", rtc_delta);

        if ((rtc_delta >= tiempo_luz_amarilla) && (rtc_delta < tiempo_luz_roja)) { // Si estamos en el rango de 50 - 59 minutos
            pcusb.printf("\r\nSemaforo AMARILLO...");
            presentar_torre(&AMARILLO, &torreLuz); // Activar luz ambar
        } else if (rtc_delta >= tiempo_luz_roja) {
            pcusb.printf("\r\nSemaforo ROJO...");
            presentar_torre(&ROJO, &torreLuz); // Activar luz roja
        } else {
            presentar_torre(&VERDE, &torreLuz); // Activar luz ambar
        }
        presentar_tiempo(&horas, &dminutos, &uminutos, &display_H, &display_dM, &display_uM);// Actualizar la vista del cronometro
        pcusb.printf("\r\nTiempo actual [H:MM] = %d:%d%d\r\n\r\n", horas, dminutos, uminutos);
    } else {
        tiempo_actual =  t_apagado.read();
        if (tiempo_actual >= (TIEMPO_RECARGA)) {
            pcusb.printf("\r\nSobrepasa limite de %d segundos con pantallas apagadas", TIEMPO_RECARGA);
            pcusb.printf("\r\nSe descarta TAG actual y reinicia contadores.\r\n");
            ticker_minuto.detach();
            t_apagado.stop();
        }
    }


}


/**
 * @brief 
 * 
 */
extern Serial   gv300;
extern char     skytrack_frame[];
extern int      contador_tramas;
void tx_skytrack()
{
    sprintf(skytrack_frame, "%s%s%s%s%04X%s",
            CABECERA_TX,CODIGO_INT,ALIVE, FIN_CABECERA_TX, contador_tramas, ULTIMO_CARACTER);
    gv300.printf("%s\r\n", skytrack_frame);
    
    incrementar_trama(&contador_tramas);
    pcusb.printf("\r\n%s", ALIVE);
}