/**
 * Interrupt Algorithm
 * Biofeedback Badgers - UW Madison
 * Fall 2020
 * Start Date: 2020/10/22
 * Last Edit: 2020/11/20
 *
 * The purpose of this script is to use the STM32 F303k8 microcontroller to 
 * record and transmit EMG Data with a sampling frequency above 1kHz and
 * transmit this data via serial communications to a connected computer.
 * 
 * This script will also sample and transmist temperature data recorded from a
 * thermistor at ~4 Hz.
 * 
 * This algorithm uses interrupts to record data and alert the system when
 * the buffers are ready for signal processing or transmission.
 * 
 * Version 01.3
 * 
 * Contributors: Will Wightman
 */
#include "mbed.h"
#include "Ticker.h"
#include "BFB_Configs_v01.hpp"   // Includes predefined filter and constants
#include <stdint.h>

// Objects and ports used for potentiometer code
SPI spi(D11, D12, D13);
DigitalOut cs(D7);

// Function declaration for function used to set gain
void setGain(float newGain);
void setGainN(int N);

// Constants used when calculating gain
extern const float MAX_GAIN; // V/V
extern const float MIN_GAIN; // V/V
extern const int MAX_GAIN_N; // Maximum integer value to calculat R_wb
extern const int MIN_GAIN_N; // Minimum integer value to calculat R_wb

// Objects needed for data collection and transmission
Serial pc(USBTX, USBRX);    // Serial transmission object to computer/PC

AnalogIn EMG(A0);           // Reads input of EMG
AnalogIn Temp(A1);          // Reads input of thermistor

Ticker emgSample;           // Timer interrupt object for collecting EMG data
Ticker tempSample;          // Timer interrupt object for collecting temp data

// Constants to define program's function
extern const int BUF_LEN;           // Length of transmission buffer
extern const float T_EMG;           // EMG sampling period in seconds
extern const float T_TEMP;          // Temperature sampling period in seconds
extern const unsigned short DATA_BIT;    // The bit identfying a data block

// Global variables to be used in this process
volatile int bufIndex;              // Keeps track of index in sample buffer
volatile bool bufSwitch;            // Indicates if the current recording array is full
volatile bool transmitting;         // Indicates if transmission is currently occuring

unsigned short buffer1[BUF_LEN];    // Buffer used to record data
unsigned short buffer2[BUF_LEN];    // Buffer used to process data
unsigned short *recordBuf; // Pointer to the recording buffer

unsigned short *transBuf;  // Pointer to the transmission buffer

// Function declarations to be used in interrupts
void emgInterrupt();                // Handles EMG timer interrupt
void tempInterrupt();               // Handles temperature interrupt
void handleInput();                 // Handles input from the serial line

int main()
{
    // Sets up serial transmission object
    pc.baud(115200);        // Rate of 11kBytes (~5k data points) per second
    
    // Initializes global variables
    bufIndex = 0;          // First index of recording buffer
    bufSwitch = false;     // Indicates buffers dont need to be rotated
    recordBuf = buffer1;   // Initializes the recording buffer
    transBuf = buffer2;    // Initializes the transmission buffer
    transmitting = false;  // indicates transmission should not be occuring
    
    // Initializes interrupts
    pc.attach(&handleInput);    // Attaches a way to process input
    
    // Sets up SPI to handle peripherals
    cs = 1;                     // Deselects the chip
    spi.format(8,0);            // Setup for 8 bit data, high steady state clock
    spi.frequency(10000000);    // Set SCLK to 10MHz
    
    while(1)
    {
        if (bufSwitch) // Processes and transmits EMG Data
        {   
            // Prints the buffer byte by byte
            for (int i = 0; i < BUF_LEN; ++i)
            {
                pc.putc(transBuf[i]);
                pc.putc(transBuf[i] >> 8);
            }
            
            // Prints the footer
            pc.putc(FOOTER);
            pc.putc(FOOTER >> 8);
            
            // Indicates the buffer is fully transmitted
            bufSwitch = false;
        }
        
    }    
}

/**
 * This function records EMG data to the buffer and calls for a buffer switch
 * once the buffer reaches capacity.
 */
void emgInterrupt()
{   
    // Switches the buffers
    if (bufIndex == BUF_LEN) {
        if (bufSwitch)
        {
            pc.printf("\nEB\n");
            return;
        }
        else
        {
            // Switches buffers and signals the buffers are available
            unsigned short *temp = recordBuf;
            recordBuf = transBuf;
            transBuf = temp;
            bufIndex = 0;
            bufSwitch = true;   // Indicates buffers have been switched
        }
    }
    
    uint16_t valIn = (0x0FFF & (int)(EMG * 0x0FFF));  // Reads input as unsigned short
    recordBuf[bufIndex] = valIn + DATA_BIT + EMG_TAG; // Saves and tags data
    bufIndex++;
}

/**
 * This funtion handles the temperature interrupt by recording the current
 * temperature.
 */
void tempInterrupt()
{
    // Switches the buffers
    if (bufIndex == BUF_LEN) {
        if (bufSwitch)
        {
            pc.printf("\nEB\n");
            return;
        }
        else
        {
            // Switches buffers and signals the buffers are available
            unsigned short *temp = recordBuf;
            recordBuf = transBuf;
            transBuf = temp;
            bufIndex = 0;
            bufSwitch = true;   // Indicates buffers have been switched
        }
    }
    
    uint16_t valIn = (0x0FFF & (uint16_t)(Temp *0x0FFF));  // Reads input as unsigned short
    recordBuf[bufIndex] = valIn + DATA_BIT + TEMP_TAG; // Saves and tags data
    bufIndex++;
}

/**
 * This interupt method handles input from the user's device to start or stop
 * sending data
 */
void handleInput()
{
    switch (pc.getc())
    {
        case 'r':
            bufIndex = 0;          // First index of recording buffer
            bufSwitch = false;     // Indicates buffers dont need to be rotated
            recordBuf = buffer1;   // Initializes the recording buffer
            transBuf = buffer2;    // Initializes the transmission buffer
            transmitting = false;  // indicates transmission should not be occuring
            emgSample.attach(&emgInterrupt, T_EMG);
            tempSample.attach(&tempInterrupt, T_TEMP);
            break;
        case 'q':
            emgSample.detach();
            tempSample.detach();
            break;
        case 'g':
            setGainN(pc.getc());
            break;
        default:
            break;
    }
}


/**
 * This method is called to change the gain of the entire circuit
 * 
 * Gain of circuit is primarily controlled by MCP4132-103 Potentiometer
 * Data sheet: https://www.mouser.com/datasheet/2/268/22059a-51937.pdf
 * - Rwb is the resistance between pin 6 and 7
 * - MOSI(D11, Nucleo) ---> SDI(4132-103, chip)
 * - MISO(D12, Nucleo) ---> SDO(4132-103, chip)
 * - sclk(D13, Nucleo) ---> sclk(4132-103, chip)
 * - CS(D7, Nucleo) ---> CS(4132-103, CS)
 * 
 * Rwb = [(Rab/128)*N + Rw] Ohms = [39.0265*N + 75] Ohms
 * 
 * Rab = 10 kOhms
 * Rw  = 75 Ohms
 * alpha = 9.23*2*2*(-0.5)*(-30E3) Ohms
 * 
 * G(Preamp) = 9.23
 * G(HPF) = 2
 * G(LPF) = 2
 * G(Level Shifter) = -0.5
 * G(Inverting Amp) = -30kOhm/Rwb
 * Gtotal = [] V/V
 *
 * N = (128*alpha/Rab)*(1/newGain) - Rw*128/Rab
 * N = 7626.2/newGain - 0.96
 */
void setGain(float newGain)
{
    if (newGain > MAX_GAIN) newGain = MAX_GAIN;
    if (newGain < MIN_GAIN) newGain = MIN_GAIN;
    
    int N = rint(7626.2f/newGain - 0.96f);
    
    if (N < MIN_GAIN_N) N = MIN_GAIN_N;
    if (N > MAX_GAIN_N) N = MAX_GAIN_N;
    
    cs = 0; // Selects the chip
    
    // Sets rheostat the proper resistance
    spi.write(0x00);
    spi.write(N);
    
    cs = 1; // Deselects the chip
}

/**
 * This method is called to change the gain of the entire circuit
 * 
 * Gain of circuit is primarily controlled by MCP4132-103 Potentiometer
 * Data sheet: https://www.mouser.com/datasheet/2/268/22059a-51937.pdf
 * - Rwb is the resistance between pin 6 and 7
 * - MOSI(D11, Nucleo) ---> SDI(4132-103, chip)
 * - MISO(D12, Nucleo) ---> SDO(4132-103, chip)
 * - sclk(D13, Nucleo) ---> sclk(4132-103, chip)
 * - CS(D7, Nucleo) ---> CS(4132-103, CS)
 * 
 * Sets the gain based on input
 */
void setGainN(int N)
{
    if (N < MIN_GAIN_N) N = MIN_GAIN_N;
    if (N > MAX_GAIN_N) N = MAX_GAIN_N;
    
    cs = 0; // Selects the chip
    
    // Sets rheostat the proper resistance
    spi.write(0x00);
    spi.write(N);
    
    cs = 1; // Deselects the chip
}