/******************************************************************************
    @file       main.cpp
    @brief      メイン処理
    @date       2015/11/14
    @author     Akira Minomo
    @par        History
    - 2015/11/14 Akira Minomo
      -# Initial Version
******************************************************************************/

#include "mbed.h"
#include "motor.h"

// if you DO NOT want to use Color Sensor Comment out next line 
//#define COLOR_SENSOR_USE

//PC接続によるデバッグ出力設定(不要の際にはコメントアウトする)
//#define DEBUG

// Color Sensor Config
#ifdef COLOR_SENSOR_USE
#define sensor_addr    (0x29 << 1) // slave adr 7bit(0x29)
#define sens_bur    100000      // Clock frequency(100khz)
#endif

// H8-mbed間のシリアル通信のボーレート
#define H8_SERIAL_BAUDRATE  38400

//タイマー割り込みの処理間隔を設定する
#define ATTACH_TIMER_PERIOD 5000 //100ms周期で割り込み処理を実施  

//H8から受信するデータに対するマスク
//H8通信データは8方位にセンサにおいて黒いラインがあるかどうかを示す。
#define BIT_MASK_SENSOR_F   0x01
#define BIT_MASK_SENSOR_FR  0x02
#define BIT_MASK_SENSOR_R   0x04
#define BIT_MASK_SENSOR_BR  0x08
#define BIT_MASK_SENSOR_B   0x10
#define BIT_MASK_SENSOR_BL  0x20
#define BIT_MASK_SENSOR_L   0x40
#define BIT_MASK_SENSOR_FL  0x80

//LED出力のON/OFF設定
#define LED_ON              1
#define LED_OFF             0

//#define DEBUG 

DigitalOut ledL(LED1);  // LED表示　デバッグ用
DigitalOut ledB(LED2);  // LED表示　デバッグ用
DigitalOut ledR(LED3);  // LED表示　デバッグ用
DigitalOut ledF(LED4);  // LED表示　デバッグ用

DigitalIn startSw(p20); // Startスイッチ　押すことでライントレースを開始する

Serial pc(USBTX, USBRX);    // PCとのシリアル通信[デバッグ用](送信・受信：USB接続)
Serial serialH8(p9, p10);   // H8とのシリアル通信(送信：p9、受信：p10)

#ifdef COLOR_SENSOR_USE
I2C i2c(p28,p27);           // カラーセンサ情報用I2C通信(SDA:p28, SCL:p29)
#endif

Ticker timer;               // 未使用

unsigned short runMode;

/* H8による受信割り込み処理 */
void attachRx()
{
    unsigned char data=0;
    int mode;
    
    __disable_irq(); // 受信処理中の割り込み禁止
    if(serialH8.readable())
    {
        data = serialH8.getc();

        // 前後左右のセンサ結果表示   
        if(data&BIT_MASK_SENSOR_FR){
            ledF = LED_ON;
        }else{
            ledF = LED_OFF;   
        }
        if(data&BIT_MASK_SENSOR_BR){
            ledR = LED_ON;
        }else{
            ledR = LED_OFF;   
        }
        if(data&BIT_MASK_SENSOR_BL){
            ledB = LED_ON;
        }else{
            ledB = LED_OFF;   
        }
        if(data&BIT_MASK_SENSOR_FL){
            ledL = LED_ON;
        }else{
            ledL = LED_OFF;   
        }
        //モードに合わせて動作の制御
        switch(runMode){
        case STARTUP_MODE:
            moveStop();            
            break;
        case RUN_MODE:
            mode = setMove(data);
#ifdef DEBUG
            pc.printf("S=%x, M=%d\n",data,mode);
#endif
            break;
        case BONUS_MODE:
            BonusTime();
            runMode = STOP_MODE;            
            break;
        case STOP_MODE:
        default:
            moveStop();            
            break;
        }
    }
    __enable_irq(); // 割り込み禁止を解除する
}

// 初期化処理
void InitTentomon()
{
    runMode = STARTUP_MODE; //走行モードを始動前のモードに設定する。
    moveStop();             // タイヤ初期化(停止)
    ArmUp();                // アーム初期化(持ち上げ状態)
    serialH8.baud(H8_SERIAL_BAUDRATE);  // H8ボーレート設定
    serialH8.format(8,Serial::None,1);  //H8通信データフォーマット設定(DataBit:8bit,Parity:なし,StopBit:1)
}

// main処理
int main() {
    InitTentomon();
     
    // H8シリアル受信に割り込みをする
    // (受信割り込みにより制御を実施)
    serialH8.attach(&attachRx,Serial::RxIrq);
        
    while(1) {
        //スタートスイッチが押された時、
        //モードをライントレースモードに変更する。()
        if(!startSw){
            if(runMode==STARTUP_MODE){
                runMode = RUN_MODE;
            }
        }
    }
}

