/* mbed Microcontroller Library
 * Copyright (c) 2006-2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "ble/BLE.h"
#include "ble/services/DeviceInformationService.h"
#include "MicroBitAccelerometerService.h"
#include "nrf_soc.h"

const char* DEVICE_NAME = "BBC micro:bit [zevug]";
const char* MICROBIT_BLE_MANUFACTURER = "The Cast of W1A";
const char* MICROBIT_BLE_MODEL = "BBC micro:bit";
const char* MICROBIT_BLE_HARDWARE_VERSION = "1.0";
const char* MICROBIT_BLE_FIRMWARE_VERSION = "BODGE";
const char* MICROBIT_BLE_SOFTWARE_VERSION = "1.0";

#define MICROBIT_BLE_ENABLE_BONDING     true
#define MICROBIT_BLE_REQUIRE_MITM       true

MicroBitAccelerometerService *accelService;
DeviceInformationService *deviceInfo;
Serial pc(USBTX, USBRX);

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    BLE::Instance(BLE::DEFAULT_INSTANCE).gap().startAdvertising(); // restart advertising
}

void periodicCallback(void)
{
    pc.printf(".");
    wait_ms(3);
}

static void passkeyDisplayCallback(Gap::Handle_t handle, const SecurityManager::Passkey_t passkey)
{
    pc.printf("\nPASSKEY: ");
    
    for (int i = 0; i<SecurityManager::PASSKEY_LEN; i++)
        pc.printf("%c", passkey[i]);

    pc.printf("\n");
}

static void securitySetupCompletedCallback(Gap::Handle_t handle, SecurityManager::SecurityCompletionStatus_t status)
{
    pc.printf("paring complete\n");
}

void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE &ble          = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);

    // Setup our security requirements.
    ble.securityManager().onPasskeyDisplay(passkeyDisplayCallback);
    ble.securityManager().onSecuritySetupCompleted(securitySetupCompletedCallback);
    ble.securityManager().init(MICROBIT_BLE_ENABLE_BONDING, MICROBIT_BLE_REQUIRE_MITM, SecurityManager::IO_CAPS_DISPLAY_ONLY);

    /* Setup auxiliary service. */
    deviceInfo = new DeviceInformationService (ble, MICROBIT_BLE_MANUFACTURER, MICROBIT_BLE_MODEL, "SN1", MICROBIT_BLE_HARDWARE_VERSION, MICROBIT_BLE_FIRMWARE_VERSION, MICROBIT_BLE_SOFTWARE_VERSION);

    /* Setup primary service. */
    accelService = new MicroBitAccelerometerService(ble);

    // Configure for high speed mode where possible.
    Gap::ConnectionParams_t fast;
    ble.getPreferredConnectionParams(&fast);
    fast.minConnectionInterval = 8; // 10 ms
    fast.maxConnectionInterval = 16; // 20 ms
    fast.slaveLatency = 0;
    ble.setPreferredConnectionParams(&fast);

    /* Setup advertising. */
    //ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    //ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    //ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_HEART_RATE_SENSOR);
    //ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    //ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    //ble.gap().setAdvertisingInterval(200); /* 1000ms */
    //ble.gap().startAdvertising();

    // Setup advertising.
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, strlen(DEVICE_NAME));
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.setAdvertisingInterval(40);
    ble.startAdvertising();  
}

int main(void)
{
    Ticker ticker;
    
    pc.baud(115200);
    
    ticker.attach(periodicCallback, 0.006); // 6ms

    BLE& ble = BLE::Instance(BLE::DEFAULT_INSTANCE);
    ble.init(bleInitComplete);

    /* SpinWait for initialization to complete. This is necessary because the
     * BLE object is used in the main loop below. */
    while (ble.hasInitialized() == false) { /* spin loop */ }
 
    // TEST
 
    uint16_t i=0;
    uint16_t accelerometerDataCharacteristicBuffer[3];

    // Disable all the contents of the ISR.
    //uBit.systemTicker.detach();

    for (int irq = 0; irq < 23; irq++)
        sd_nvic_SetPriority((IRQn_Type)irq, 3);
    
    while(1)
    {   
        /*
        if (i % 500 == 0)
        {
            int sp = (int) __get_MSP();
            uBit.serial.printf("MSP: 0x%.8x\n", sp);

            uint32_t pm = __get_PRIMASK();
            uBit.serial.printf("PG: 0x%.8x\n", pm);
        
            nrf_app_irq_priority_t pri;
            
            for (int irq = 0; irq <= 25; irq++)
            {
                sd_nvic_GetPriority((IRQn_Type)irq, &pri);
                uBit.serial.printf("irq: %2d priority: %d\n", irq, pri);
            }

        }
        */

        wait_ms(20);
        accelerometerDataCharacteristicBuffer[0] = i;
        i++;
        
        if (ble.getGapState().connected)
        {
            ble.gattServer().write(accelerometerDataCharacteristicHandleGlobal,(uint8_t *)accelerometerDataCharacteristicBuffer, sizeof(accelerometerDataCharacteristicBuffer));
        }
    }
}

