/******************** (C) COPYRIGHT 2014 STMicroelectronics ********************
* File Name          : bluenrg_gap_aci.h
* Author             : AMS - AAS
* Version            : V1.0.0
* Date               : 26-Jun-2014
* Description        : Header file with GAP commands for BlueNRG FW6.3.
********************************************************************************
* THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
* WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
* AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT,
* INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE
* CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING
* INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/

#ifndef __BLUENRG_GAP_ACI_H__
#define __BLUENRG_GAP_ACI_H__

#include <sm.h>

/**
 *@defgroup GAP_Functions GAP functions
 *@brief API for GAP layer.
 *@{
 */

/**
  * @brief  Initialize the GAP layer.
  * @note   Register the GAP service with the GATT. If the role is peripheral,
  *         all the standard GAP characteristic will be also added:
  *         @li Device Name
  *         @li Appearance
  *         @li Peripheral Privacy Flag
  *         @li Reconnection Address
  *         @li Peripheral Preferred Connection Parameters
  *         @code

              tBleStatus ret;
              uint16_t service_handle, dev_name_char_handle, appearance_char_handle;

              ret = aci_gap_init(1, &service_handle, &dev_name_char_handle, &appearance_char_handle);
              if(ret){
                PRINTF("GAP_Init failed.\n");
                reboot();    
              }  
              const char *name = "BlueNRG";  
              ret = aci_gatt_update_char_value(service_handle, dev_name_char_handle, 0, strlen(name), (uint8_t *)name);        
              if(ret){
                PRINTF("aci_gatt_update_char_value failed.\n");
              }  
  *         @endcode
  * @param       role     One of the allowed roles: PERIPHERAL or CENTRAL.
  * @param[out]  service_handle  Handle of the GAP service.
  * @param[out]  dev_name_char_handle  Device Name Characteristic handle
  * @param[out]  appearance_char_handle Appearance Characteristic handle
  * @retval tBleStatus Value indicating success or error code.
  */
tBleStatus aci_gap_init(uint8_t role,
                 uint16_t* service_handle,
                 uint16_t* dev_name_char_handle,
                 uint16_t* appearance_char_handle);

/**
  * @brief   Set the Device in non-discoverable mode.
  * @note    This command will disable the LL advertising.
  * @retval  tBleStatus Value indicating success or error code.
  */
tBleStatus aci_gap_set_non_discoverable(void);

/**
 * @brief  Put the device in limited discoverable mode
 *         (as defined in GAP specification volume 3, section 9.2.3).
 * @note    The device will be discoverable for TGAP (lim_adv_timeout) = 180 seconds.
 *          The advertising can be disabled at any time by issuing
 *          aci_gap_set_non_discoverable() command.
 *          The AdvIntervMin and AdvIntervMax parameters are optional. If both
 *          are set to 0, the GAP will use default values (250 ms and 500 ms respectively).
 *          Host can set the Local Name, a Service UUID list and the Slave Connection
 *          Minimum and Maximum. If provided, these data will be inserted into the
 *          advertising packet payload as AD data. These parameters are optional
 *          in this command. These values can be also set using aci_gap_update_adv_data()
 *          separately.
 *          The total size of data in advertising packet cannot exceed 31 bytes.
 *          With this command, the BLE Stack will also add automatically the following
 *          standard AD types:
 *          @li AD Flags
 *          @li TX Power Level
 *
 *          When advertising timeout happens (i.e. limited discovery period has elapsed), controller generates
 *          @ref EVT_BLUE_GAP_LIMITED_DISCOVERABLE event.
 *
 *          Example:
 * @code
 *
 *              #define  ADV_INTERVAL_MIN_MS  100
 *              #define  ADV_INTERVAL_MAX_MS  200
 *
 *              tBleStatus ret;
 *
 *              const char local_name[] = {AD_TYPE_COMPLETE_LOCAL_NAME,'B','l','u','e','N','R','G'};
 *              const uint8_t serviceUUIDList[] = {AD_TYPE_16_BIT_SERV_UUID,0x34,0x12};
 *
 *              ret = aci_gap_set_limited_discoverable(ADV_IND, (ADV_INTERVAL_MIN_MS*1000)/0.625,
 *                                                     (ADV_INTERVAL_MAX_MS*1000)/0.625,
 *                                                     RANDOM_ADDR, NO_WHITE_LIST_USE,
 *                                                    sizeof(local_name), local_name,
 *                                                     sizeof(serviceUUIDList), serviceUUIDList,
 *                                                     0, 0);
 * @endcode
 *
 * @param       AdvType     One of the advertising types:
 *               @arg @ref ADV_IND Connectable undirected advertising
 *               @arg @ref ADV_SCAN_IND Scannable undirected advertising
 *               @arg @ref ADV_NONCONN_IND Non connectable undirected advertising
 * @param       AdvIntervMin     Minimum advertising interval
 * @param       AdvIntervMax     Maximum advertising interval
 * @param       OwnAddrType     Type of our address used during advertising
 *                              (@ref PUBLIC_ADDR,@ref RANDOM_ADDR).
 * @param       AdvFilterPolicy  Filter policy:
 *                               @arg NO_WHITE_LIST_USE
 *                               @arg WHITE_LIST_FOR_ONLY_SCAN
 *                               @arg WHITE_LIST_FOR_ONLY_CONN
 *                               @arg WHITE_LIST_FOR_ALL
 * @param  LocalNameLen  Length of LocalName array.
 * @param  LocalName  Array containing the Local Name AD data. First byte is the AD type:
 *                       @ref AD_TYPE_SHORTENED_LOCAL_NAME or @ref AD_TYPE_COMPLETE_LOCAL_NAME.
 * @param  ServiceUUIDLen Length of ServiceUUIDList array.
 * @param  ServiceUUIDList  This is the list of the UUIDs AD Types as defined in Volume 3,
 *  							Section 11.1.1 of GAP Specification. First byte is the AD Type.
 *  							@arg @ref AD_TYPE_16_BIT_SERV_UUID
 *  							@arg @ref AD_TYPE_16_BIT_SERV_UUID_CMPLT_LIST
 *  							@arg @ref AD_TYPE_128_BIT_SERV_UUID
 *  							@arg @ref AD_TYPE_128_BIT_SERV_UUID_CMPLT_LIST
 * @param  SlaveConnIntervMin Slave connection interval minimum value suggested by Peripheral.
 * 							 If SlaveConnIntervMin and SlaveConnIntervMax are not 0x0000,
 * 							 Slave Connection Interval Range AD structure will be added in advertising
 * 							 data.
 * 							 Connection interval is defined in the following manner:
 * 							 connIntervalmin = Slave_Conn_Interval_Min x 1.25ms
 * 							 Slave_Conn_Interval_Min range: 0x0006 to 0x0C80
 * 							 Value of 0xFFFF indicates no specific minimum.
 * @param  SlaveConnIntervMax Slave connection interval maximum value suggested by Peripheral.
 * 							 If SlaveConnIntervMin and SlaveConnIntervMax are not 0x0000,
 * 							 Slave Connection Interval Range AD structure will be added in advertising
 * 							 data.
 * 							 ConnIntervalmax = Slave_Conn_Interval_Max x 1.25ms
 * 							 Slave_Conn_Interval_Max range: 0x0006 to 0x0C80
 * 							 Slave_ Conn_Interval_Max shall be equal to or greater than the Slave_Conn_Interval_Min.
 * 							 Value of 0xFFFF indicates no specific maximum.
 *
 * @retval tBleStatus Value indicating success or error code.
 */
tBleStatus aci_gap_set_limited_discoverable(uint8_t AdvType, uint16_t AdvIntervMin, uint16_t AdvIntervMax,
					    uint8_t OwnAddrType, uint8_t AdvFilterPolicy, uint8_t LocalNameLen,
					    const char *LocalName, uint8_t ServiceUUIDLen, uint8_t* ServiceUUIDList,
					    uint16_t SlaveConnIntervMin, uint16_t SlaveConnIntervMax);
/**
 * @brief Put the Device in general discoverable mode (as defined in GAP specification volume 3, section 9.2.4).
 * @note  The device will be discoverable until the Host issue Bluehci_Gap_Set_Non_Discoverable command.
 * 		  The Adv_Interval_Min and Adv_Interval_Max parameters are optional. If both are set to 0, the GAP uses
 * 		  the default values for advertising intervals (1.28 s and 2.56 s respectively).
 * 		  Host can set the Local Name, a Service UUID list and the Slave Connection Interval Range. If provided,
 * 		  these data will be inserted into the advertising packet payload as AD data. These parameters are optional
 * 		  in this command. These values can be also set using aci_gap_update_adv_data() separately.
 * 		  The total size of data in advertising packet cannot exceed 31 bytes.
 * 		  With this command, the BLE Stack will also add automatically the following standard AD types:
 * 		  @li AD Flags
 * 		  @li TX Power Level
 *
 * 		  Usage example:
 *
 * 		  @code
 *
 *              #define  ADV_INTERVAL_MIN_MS  800
 *              #define  ADV_INTERVAL_MAX_MS  900
 *              #define  CONN_INTERVAL_MIN_MS 100
 *              #define  CONN_INTERVAL_MAX_MS 300
 *
 *              tBleStatus ret;
 *
 *              const char local_name[] = {AD_TYPE_COMPLETE_LOCAL_NAME,'B','l','u','e','N','R','G'};
 *              const uint8_t serviceUUIDList[] = {AD_TYPE_16_BIT_SERV_UUID,0x34,0x12};
 *
 *              ret = aci_gap_set_discoverable(ADV_IND, (ADV_INTERVAL_MIN_MS*1000)/0.625,
 *                                                     (ADV_INTERVAL_MAX_MS*1000)/0.625,
 *                                                     RANDOM_ADDR, NO_WHITE_LIST_USE,
 *                                                     sizeof(local_name), local_name,
 *                                                     0, NULL,
 *                                                     (CONN_INTERVAL_MIN_MS*1000)/1250,
 *                                                     (CONN_INTERVAL_MAX_MS*1000)/1250);
 * 			@endcode
 *
 * @param AdvType One of the advertising types:
 *                @arg @ref ADV_IND Connectable undirected advertising
 *                @arg @ref ADV_SCAN_IND Scannable undirected advertising
 *                @arg @ref ADV_NONCONN_IND Non connectable undirected advertising
 * @param       AdvIntervMin     Minimum advertising interval
 * @param       AdvIntervMax     Maximum advertising interval
 * @param       OwnAddrType     Type of our address used during advertising
 *                              (@ref PUBLIC_ADDR,@ref RANDOM_ADDR).
 * @param       AdvFilterPolicy  Filter policy:
 *                               @arg @ref NO_WHITE_LIST_USE
 *                               @arg @ref WHITE_LIST_FOR_ONLY_SCAN
 *                               @arg @ref WHITE_LIST_FOR_ONLY_CONN
 *                               @arg @ref WHITE_LIST_FOR_ALL
 * @param  LocalNameLen  Length of LocalName array.
 * @param  LocalName  Array containing the Local Name AD data. First byte is the AD type:
 *                       @ref AD_TYPE_SHORTENED_LOCAL_NAME or @ref AD_TYPE_COMPLETE_LOCAL_NAME.
 * @param  ServiceUUIDLen Length of ServiceUUIDList array.
 * @param  ServiceUUIDList  This is the list of the UUIDs AD Types as defined in Volume 3,
 *  							Section 11.1.1 of GAP Specification. First byte is the AD Type.
 *  							@arg @ref AD_TYPE_16_BIT_SERV_UUID
 *  							@arg @ref AD_TYPE_16_BIT_SERV_UUID_CMPLT_LIST
 *  							@arg @ref AD_TYPE_128_BIT_SERV_UUID
 *  							@arg @ref AD_TYPE_128_BIT_SERV_UUID_CMPLT_LIST
 * @param  SlaveConnIntervMin Slave connection interval minimum value suggested by Peripheral.
 * 							 If SlaveConnIntervMin and SlaveConnIntervMax are not 0x0000,
 * 							 Slave Connection Interval Range AD structure will be added in advertising
 * 							 data.
 * 							 Connection interval is defined in the following manner:
 * 							 connIntervalmin = Slave_Conn_Interval_Min x 1.25ms
 * 							 Slave_Conn_Interval_Min range: 0x0006 to 0x0C80
 * 							 Value of 0xFFFF indicates no specific minimum.
 * @param  SlaveConnIntervMax Slave connection interval maximum value suggested by Peripheral.
 * 							 If SlaveConnIntervMin and SlaveConnIntervMax are not 0x0000,
 * 							 Slave Connection Interval Range AD structure will be added in advertising
 * 							 data.
 * 							 ConnIntervalmax = Slave_Conn_Interval_Max x 1.25ms
 * 							 Slave_Conn_Interval_Max range: 0x0006 to 0x0C80
 * 							 Slave_ Conn_Interval_Max shall be equal to or greater than the Slave_Conn_Interval_Min.
 * 							 Value of 0xFFFF indicates no specific maximum.
 *
 * @retval tBleStatus Value indicating success or error code.
 */
tBleStatus aci_gap_set_discoverable(uint8_t AdvType, uint16_t AdvIntervMin, uint16_t AdvIntervMax,
                             uint8_t OwnAddrType, uint8_t AdvFilterPolicy, uint8_t LocalNameLen,
                             const char *LocalName, uint8_t ServiceUUIDLen, uint8_t* ServiceUUIDList,
                             uint16_t SlaveConnIntervMin, uint16_t SlaveConnIntervMax);

/**
 * @brief Set the Device in direct connectable mode (as defined in GAP specification Volume 3, Section 9.3.3).
 * @note  If the privacy is enabled, the reconnection address is used for advertising, otherwise the address
 * 		  of the type specified in OwnAddrType is used. The device will be in directed connectable mode only
 * 		  for 1.28 seconds. If no connection is established within this duration, the device enters non
 * 		  discoverable mode and advertising will have to be again enabled explicitly.
 * 		  The controller generates a @ref EVT_LE_CONN_COMPLETE event with the status set to @ref HCI_DIRECTED_ADV_TIMEOUT
 * 		  if the connection was not established and 0x00 if the connection was successfully established.
 *
 * 		  Usage example:
 * 		  @code
 *
 * 		  tBleStatus ret;
 *
 * 		  const uint8_t central_address = {0x43,0x27,0x84,0xE1,0x80,0x02};
 * 		  ret = aci_gap_set_direct_connectable(PUBLIC_ADDR, PUBLIC_ADDR, central_address);
 * 		  @endcode
 *
 *
 *
 * @param OwnAddrType  Type of our address used during advertising (@ref PUBLIC_ADDR,@ref RANDOM_ADDR).
 * @param InitiatorAddrType Type of peer address (@ref PUBLIC_ADDR,@ref RANDOM_ADDR).
 * @param InitiatorAddr     Initiator's address (Little Endian).
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_set_direct_connectable(uint8_t OwnAddrType, uint8_t InitiatorAddrType, const uint8_t *InitiatorAddr);

/**
 * @brief Set the IO capabilities of the device.
 * @note This command has to be given only when the device is not in a connected state.
 * @param io_capability One of the allowed codes for IO Capability:
 * 		  @arg @ref IO_CAP_DISPLAY_ONLY
 * 		  @arg @ref IO_CAP_DISPLAY_YES_NO
 * 		  @arg @ref IO_CAP_KEYBOARD_ONLY
 * 		  @arg @ref IO_CAP_NO_INPUT_NO_OUTPUT
 * 		  @arg @ref IO_CAP_KEYBOARD_DISPLAY
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_set_io_capabilitiy(uint8_t io_capability);

/**
 * @brief Set the authentication requirements for the device.
 * @note  If the oob_enable is set to 0, oob_data will be ignored.
 *        This command has to be given only when the device is not in a connected state.
 * @param mitm_mode MITM mode:
 * 					@arg @ref MITM_PROTECTION_NOT_REQUIRED
 * 					@arg @ref MITM_PROTECTION_REQUIRED
 * @param oob_enable If OOB data are present or not:
 * 					 @arg @ref OOB_AUTH_DATA_ABSENT
 * 					 @arg @ref OOB_AUTH_DATA_PRESENT
 * @param oob_data   Out-Of-Band data
 * @param min_encryption_key_size Minimum size of the encryption key to be used during the pairing process
 * @param max_encryption_key_size Maximum size of the encryption key to be used during the pairing process
 * @param use_fixed_pin If application wants to use a fixed pin or not:
 * 						@arg @ref USE_FIXED_PIN_FOR_PAIRING
 * 						@arg @ref DONOT_USE_FIXED_PIN_FOR_PAIRING
 * 						If a fixed pin is not used, it has to be provided by the application with
 * 						aci_gap_pass_key_response() after @ref EVT_BLUE_GAP_PASS_KEY_REQUEST event.
 * @param fixed_pin If use_fixed_pin is USE_FIXED_PIN_FOR_PAIRING, this is the value of the pin that will
 * 					be used during pairing if MIMT protection is enabled. Any value between 0 to 999999 is
 * 					accepted.
 * @param bonding_mode One of the bonding modes:
 * 					   @arg @ref BONDING
 * 					   @arg @ref NO_BONDING
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_set_auth_requirement(uint8_t mitm_mode,
                                        uint8_t oob_enable,
                                        uint8_t oob_data[16],
                                        uint8_t min_encryption_key_size,
                                        uint8_t max_encryption_key_size,
                                        uint8_t use_fixed_pin,
                                        uint32_t fixed_pin,
                                        uint8_t bonding_mode);
 /**
  * @brief Set the authorization requirements of the device.
  * @note This command has to be given only when the device is not in a connected state.
  * @param conn_handle Handle of the connection in case BlueNRG is configured as a master (otherwise it can be also 0).
  * @param authorization_enable @arg @ref AUTHORIZATION_NOT_REQUIRED : Authorization not required
  * 							@arg @ref AUTHORIZATION_REQUIRED : Authorization required. This enables
  * 							the authorization requirement in the device and when a remote device
  * 							tries to connect to GATT server, @ref EVT_BLUE_GAP_AUTHORIZATION_REQUEST event
  * 							will be sent to the Host.
  *
  * @return Value indicating success or error code.
  */
tBleStatus aci_gap_set_author_requirement(uint16_t conn_handle, uint8_t authorization_enable);

/**
 * @brief Provide the pass key that will be used during pairing.
 * @note This command should be sent by the Host in response to @ref EVT_BLUE_GAP_PASS_KEY_REQUEST event.
 * @param conn_handle Connection handle
 * @param passkey	  Pass key that will be used during the pairing process. Must be a number between
 * 					  0 and 999999.
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_pass_key_response(uint16_t conn_handle, uint32_t passkey);

/**
 * @brief Authorize a device to access attributes.
 * @note Application should send this command after it has received a @ref EVT_BLUE_GAP_AUTHORIZATION_REQUEST.
 *
 * @param conn_handle Connection handle
 * @param authorize   @arg @ref CONNECTION_AUTHORIZED : Authorize (accept connection)
 * 					  @arg @ref CONNECTION_REJECTED : Reject (reject connection)
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_authorization_response(uint16_t conn_handle, uint8_t authorize);

/**
 * @brief Put the device into non-connectable mode.
 * @param adv_type One of the allowed advertising types:
 *                 @arg @ref ADV_SCAN_IND : Scannable undirected advertising
 *                 @arg @ref ADV_NONCONN_IND : Non-connectable undirected advertising
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_set_non_connectable(uint8_t adv_type);

/**
 * @brief Put the device into undirected connectable mode.
 * @note  If privacy is enabled in the device, a resolvable private address is generated and used
 *        as the advertiser's address. If not, the address of the type specified in own_addr_type
 *        is used for advertising.
 * @param own_addr_type Type of our address used during advertising:
 * 			@arg @ref PUBLIC_ADDR.
 * 			@arg @ref RANDOM_ADDR.
 * @param adv_filter_policy  Filter policy:
 *                         @arg @ref NO_WHITE_LIST_USE
 *                         @arg @ref WHITE_LIST_FOR_ALL
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_set_undirected_connectable(uint8_t own_addr_type, uint8_t adv_filter_policy);

/**
 * @brief Send a slave security request to the master.
 * @note This command has to be issued to notify the master of the security requirements of the slave.
 * 		 The master may encrypt the link, initiate the pairing procedure, or reject the request.
 * @param conn_handle Connection handle
 * @param bonding	   One of the bonding modes:
 * 					   @arg @ref BONDING
 * 					   @arg @ref NO_BONDING
 * @param mitm_protection  If MITM protection is required or not:
 * 						   @arg @ref MITM_PROTECTION_NOT_REQUIRED
 * 					       @arg @ref MITM_PROTECTION_REQUIRED
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_slave_security_request(uint16_t conn_handle, uint8_t bonding, uint8_t mitm_protection);

/**
 * @brief Update advertising data.
 * @note This command can be used to update the advertising data for a particular AD type.
 *       If the AD type specified does not exist, then it is added to the advertising data.
 *       If the overall advertising data length is more than 31 octets after the update, then
 *       the command is rejected and the old data is retained.
 * @param AdvLen Length of AdvData array
 * @param AdvData Advertisement Data,  formatted as specified in Bluetooth specification
 *        (Volume 3, Part C, 11), including data length. It can contain more than one AD type.
 *        Example
 * @code
 *  tBleStatus ret;
 *  const char local_name[] = {AD_TYPE_COMPLETE_LOCAL_NAME,'B','l','u','e','N','R','G'};
 *  const uint8_t serviceUUIDList[] = {AD_TYPE_16_BIT_SERV_UUID,0x34,0x12};
 *  const uint8_t manuf_data[] = {4, AD_TYPE_MANUFACTURER_SPECIFIC_DATA, 0x05, 0x02, 0x01};
 *
 *  ret = aci_gap_set_discoverable(ADV_IND, 0, 0, RANDOM_ADDR, NO_WHITE_LIST_USE,
 *                                 8, local_name, 3, serviceUUIDList, 0, 0);
 *  ret = aci_gap_update_adv_data(5, manuf_data);
 * @endcode
 *
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_update_adv_data(uint8_t AdvLen, uint8_t *AdvData);

/**
 * @brief Add addresses of bonded devices into the controller's whitelist.
 * @note  The command will return an error if there are no devices in the database or if it was unable
 * 		  to add the device into the whitelist.
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_configure_whitelist(void);

/**
 * @brief Terminate a connection.
 * @note  A @ref EVT_DISCONN_COMPLETE event will be generated when the link is disconnected.
 * @param conn_handle Connection handle
 * @param reason  Reason for requesting disconnection. The error code can be any of ones as specified
 * 				  for the disconnected command in the HCI specification (See @ref HCI_Error_codes).
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_terminate(uint16_t conn_handle, uint8_t reason);

/**
 * @brief Clear the security database.
 * @note  All the devices in the security database will be removed.
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_clear_security_database(void);

/**
 * @brief Allows the security manager to complete the pairing procedure and re-bond with the master.
 * @note This command can be issued by the application if a @ref EVT_BLUE_GAP_BOND_LOST event is generated.
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_allow_rebond(void);

/**
 * @brief Start the limited discovery procedure.
 * @note  The controller is commanded to start active scanning. When this procedure is started,
 *        only the devices in limited discoverable mode are returned to the upper layers.
 *        The procedure is terminated when either the upper layers issue a command to terminate the
 *        procedure by issuing the command aci_gap_terminate_gap_procedure() with the procedure code
 *        set to @ref GAP_LIMITED_DISCOVERY_PROC or a timeout happens. When the procedure is terminated
 *        due to any of the above  reasons, @ref EVT_BLUE_GAP_PROCEDURE_COMPLETE event is returned with
 *        the procedure code set to @ref GAP_LIMITED_DISCOVERY_PROC.
 *        The device found when the procedure is ongoing is returned to the upper layers through the
 *        event @ref EVT_BLUE_GAP_DEVICE_FOUND.
 * @param scanInterval Time interval from when the Controller started its last LE scan until it begins
 * 					   the subsequent LE scan. The scan interval should be a number in the range
 * 					   0x0004 to 0x4000. This corresponds to a time range 2.5 msec to 10240 msec.
 * 					   For a number N, Time = N x 0.625 msec.
 * @param scanWindow Amount of time for the duration of the LE scan. Scan_Window shall be less than
 * 					 or equal to Scan_Interval. The scan window should be a number in the range
 * 					 0x0004 to 0x4000. This corresponds to a time range 2.5 msec to 10240 msec.
 * 					 For a number N, Time = N x 0.625 msec.
 * @param own_address_type Type of our address used during advertising (@ref PUBLIC_ADDR, @ref RANDOM_ADDR).
 * @param filterDuplicates Duplicate filtering enabled or not.
 * 						   @arg 0x00: Do not filter the duplicates
 * 						   @arg 0x01: Filter duplicates
 *
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_start_limited_discovery_proc(uint16_t scanInterval, uint16_t scanWindow,
						uint8_t own_address_type, uint8_t filterDuplicates);

/**
 * @brief Start the general discovery procedure.
 * @note  The controller is commanded to start active scanning. The procedure is terminated when
 * 		  either the upper layers issue a command to terminate the procedure by issuing the command
 * 		  aci_gap_terminate_gap_procedure() with the procedure code set to GAP_GENERAL_DISCOVERY_PROC
 * 		  or a timeout happens. When the procedure is terminated due to any of the above reasons,
 * 		  @ref EVT_BLUE_GAP_PROCEDURE_COMPLETE event is returned with the procedure code set to
 * 		  @ref GAP_GENERAL_DISCOVERY_PROC. The device found when the procedure is ongoing is returned to
 * 		  the upper layers through the event @ref EVT_BLUE_GAP_DEVICE_FOUND.
 * @param scanInterval Time interval from when the Controller started its last LE scan until it begins
 * 					   the subsequent LE scan. The scan interval should be a number in the range
 * 					   0x0004 to 0x4000. This corresponds to a time range 2.5 msec to 10240 msec.
 * 					   For a number N, Time = N x 0.625 msec.
 * @param scanWindow Amount of time for the duration of the LE scan. Scan_Window shall be less than
 * 					 or equal to Scan_Interval. The scan window should be a number in the range
 * 					 0x0004 to 0x4000. This corresponds to a time range 2.5 msec to 10240 msec.
 * 					 For a number N, Time = N x 0.625 msec.
 * @param own_address_type Type of our address used during advertising (@ref PUBLIC_ADDR, @ref RANDOM_ADDR).
 * @param filterDuplicates Duplicate filtering enabled or not.
 * 						   @arg 0x00: Do not filter the duplicates
 * 						   @arg 0x01: Filter duplicates
 *
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_start_general_discovery_proc(uint16_t scanInterval, uint16_t scanWindow,
						uint8_t own_address_type, uint8_t filterDuplicates);

/**
 * @brief Start the auto connection establishment procedure.
 * @note The devices specified are added to the white list of the controller and a LE_Create_Connection
 * 		 call will be made to the controller by GAP with the initiator filter policy set to
 * 		 use whitelist to determine which advertiser to connect to. When a command is issued to
 * 		 terminate the procedure by upper layer, a LE_Create_Connection_Cancel call will be made to the
 * 		 controller by GAP.
 * 		 The procedure is terminated when either a connection is successfully established with one of
 * 		 the specified devices in the white list or the procedure is explicitly terminated by issuing
 * 		 the command aci_gap_terminate_gap_procedure() with the procedure code set to
 * 		 @ref GAP_AUTO_CONNECTION_ESTABLISHMENT_PROC. A @ref EVT_BLUE_GAP_PROCEDURE_COMPLETE event is returned with
 * 		 the procedure code set to @ref GAP_AUTO_CONNECTION_ESTABLISHMENT_PROC.
 * @param scanInterval Time interval from when the Controller started its last LE scan until it begins
 * 					   the subsequent LE scan. The scan interval should be a number in the range
 * 					   0x0004 to 0x4000. This corresponds to a time range 2.5 msec to 10240 msec.
 * 					   For a number N, Time = N x 0.625 msec.
 * @param scanWindow Amount of time for the duration of the LE scan. Scan_Window shall be less than
 * 					 or equal to Scan_Interval. The scan window should be a number in the range
 * 					 0x0004 to 0x4000. This corresponds to a time range 2.5 msec to 10240 msec.
 * 					 For a number N, Time = N x 0.625 msec.
 * @param own_bdaddr_type Type of our address used during advertising (PUBLIC_ADDR,RANDOM_ADDR).
 * @param conn_min_interval Minimum value for the connection event interval. This shall be less than or
 * 							equal to Conn_Interval_Max.\n
 * 							Range: 0x0006 to 0x0C80\n
 * 							Time = N x 1.25 msec\n
 * 							Time Range: 7.5 msec to 4 seconds
 * @param conn_max_interval Maximum value for the connection event interval. This shall be greater than or
 * 							equal to Conn_Interval_Min.\n
 * 							Range: 0x0006 to 0x0C80\n
 * 							Time = N x 1.25 msec\n
 * 							Time Range: 7.5 msec to 4 seconds
 * @param conn_latency Slave latency for the connection in number of connection events.\n
 * 						 Range: 0x0000 to 0x01F4
 * @param supervision_timeout Supervision timeout for the LE Link.\n
 * 							  Range: 0x000A to 0x0C80\n
 * 							  Time = N x 10 msec\n
 * 							  Time Range: 100 msec to 32 seconds
 * @param min_conn_length Minimum length of connection needed for the LE connection.\n
 * 						  Range: 0x0000 - 0xFFFF\n
 * 						  Time = N x 0.625 msec.
 * @param max_conn_length Maximum length of connection needed for the LE connection.\n
 * 						  Range: 0x0000 - 0xFFFF\n
 * 						  Time = N x 0.625 msec.
 * @param num_whitelist_entries Number of devices that have to be added to the whitelist.
 * @param addr_array addr_array will contain the addresses that have to be added into the whitelist. The
 * 					 format of the addr_array should be: address type followed by address.
 * 					 Example:
 * 					 @code
 * 					 uint8_t addr_array[] = {PUBLIC_ADDR,0x01,0x00,0x00,0xe1,0x80,0x02,
 * 					 	PUBLIC_ADDR,0x02,0x00,0x00,0xe1,0x80,0x02}
 * 					 @endcode
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_start_auto_conn_establishment(uint16_t scanInterval, uint16_t scanWindow,
						 uint8_t own_bdaddr_type, uint16_t conn_min_interval,	
						 uint16_t conn_max_interval, uint16_t conn_latency,	
						 uint16_t supervision_timeout, uint16_t min_conn_length, 
						 uint16_t max_conn_length, uint8_t num_whitelist_entries,
						 uint8_t *addr_array);

/**
 * @brief Start the direct connection establishment procedure.
 * @note  A LE_Create_Connection call will be made to the controller by GAP with the initiator filter
 * 		  policy set to ignore whitelist and process connectable advertising packets only for the
 * 		  specified device. The procedure can be terminated explicitly by the upper layer by issuing
 * 		  the command aci_gap_terminate_gap_procedure(). When a command is issued to terminate the
 * 		  procedure by upper layer, a LE_Create_Connection_Cancel call will be made to the controller
 * 		  by GAP.
 * 		  On termination of the procedure, a @ref EVT_LE_CONN_COMPLETE event is returned. The procedure can
 * 		  be explicitly terminated by the upper layer by issuing the command
 * 		  aci_gap_terminate_gap_procedure() with the procedure_code set to @ref GAP_DIRECT_CONNECTION_ESTABLISHMENT_PROC.
 * @param scanInterval Time interval from when the Controller started its last LE scan until it begins
 * 					   the subsequent LE scan. The scan interval should be a number in the range
 * 					   0x0004 to 0x4000. This corresponds to a time range 2.5 msec to 10240 msec.
 * 					   For a number N, Time = N x 0.625 msec.
 * @param scanWindow Amount of time for the duration of the LE scan. Scan_Window shall be less than
 * 					 or equal to Scan_Interval. The scan window should be a number in the range
 * 					 0x0004 to 0x4000. This corresponds to a time range 2.5 msec to 10240 msec.
 * 					 For a number N, Time = N x 0.625 msec.
 * @param peer_bdaddr_type Type of the peer address (@ref PUBLIC_ADDR, @ref RANDOM_ADDR).
 * @param peer_bdaddr Address of the peer device with which a connection has to be established.
 * @param own_bdaddr_type Type of our address used during advertising (PUBLIC_ADDR,RANDOM_ADDR).
 * @param conn_min_interval Minimum value for the connection event interval. This shall be less than or
 * 							equal to Conn_Interval_Max.\n
 * 							Range: 0x0006 to 0x0C80\n
 * 							Time = N x 1.25 msec\n
 * 							Time Range: 7.5 msec to 4 seconds
 * @param conn_max_interval Maximum value for the connection event interval. This shall be greater than or
 * 							equal to Conn_Interval_Min.\n
 * 							Range: 0x0006 to 0x0C80\n
 * 							Time = N x 1.25 msec\n
 * 							Time Range: 7.5 msec to 4 seconds
 * @param conn_latency Slave latency for the connection in number of connection events.\n
 * 						 Range: 0x0000 to 0x01F4
 * @param supervision_timeout Supervision timeout for the LE Link.\n
 * 							  Range: 0x000A to 0x0C80\n
 * 							  Time = N x 10 msec\n
 * 							  Time Range: 100 msec to 32 seconds
 * @param min_conn_length Minimum length of connection needed for the LE connection.\n
 * 						  Range: 0x0000 - 0xFFFF\n
 * 						  Time = N x 0.625 msec.
 * @param max_conn_length Maximum length of connection needed for the LE connection.\n
 * 						  Range: 0x0000 - 0xFFFF\n
 * 						  Time = N x 0.625 msec.
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_create_connection(uint16_t scanInterval, uint16_t scanWindow,
				     uint8_t peer_bdaddr_type, tBDAddr peer_bdaddr,	
				     uint8_t own_bdaddr_type, uint16_t conn_min_interval,	
				     uint16_t conn_max_interval, uint16_t conn_latency,	
				     uint16_t supervision_timeout, uint16_t min_conn_length, 
				     uint16_t max_conn_length);

/**
 * @brief Terminate the specified GATT procedure. @ref EVT_BLUE_GAP_PROCEDURE_COMPLETE event is
 *  	  returned with the procedure code set to the corresponding procedure.
 * @param procedure_code One of the procedure codes (gap_procedure_codes).
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_terminate_gap_procedure(uint8_t procedure_code);

/**
 * @brief Send the SM pairing request to start a pairing process.
 *        The authentication requirements and IO capabilities should be set
 *        before issuing this command using the Bluehci_Set_IO_Capabilities
 *        and Bluehci_Set_Authentication_Requirements commands.
 *
 * @param conn_handle Handle of the connection for which the pairing request has
 *                    to be sent.
 *
 * @param force_rebond 0x00: Pairing request is sent only if the device has not
 *                           previously bonded
 *                     0x01: Pairing request will be sent even if the device was
 *                           previously bonded
 *
 * @return Value indicating success or error code.
 */
tBleStatus aci_gap_send_pairing_request(uint16_t conn_handle, uint8_t force_rebond);

/**
 * @}
 */

/**
 * @defgroup GAP_Events GAP events
 * @{
 */

/**
 * This event is generated by the controller when the limited discoverable
 *  mode ends due to timeout (180 seconds)
 */
#define EVT_BLUE_GAP_LIMITED_DISCOVERABLE     (0x0400)


/**
 * This event is generated when the pairing process has completed successfully
 * or a pairing procedure timeout has occurred or the pairing has failed.
 * This is to notify the application that we have paired with a remote device
 * so that it can take further actions or to notify that a timeout has occurred
 *  so that the upper layer can decide to disconnect the link. See @ref evt_gap_pairing_cmplt.
 */
#define EVT_BLUE_GAP_PAIRING_CMPLT                (0x0401)
typedef __packed struct _evt_gap_pairing_cmplt{
  uint16_t conn_handle; /**< Connection handle on which the pairing procedure completed */
  /**
   * 0x00: Pairing Success. Pairing with a remote device was successful\n
   * 0x01: Pairing Timeout. The SMP timeout has elapsed and no further SMP commands will be processed until reconnection\n
   * 0x02: Pairing Failed. The pairing failed with the remote device.
   */
  uint8_t  status;
} PACKED evt_gap_pairing_cmplt;


/**
 * This event is generated by the Security manager to the application when a pass key is required for pairing.
 * When this event is received, the application has to respond with the aci_gap_pass_key_response() command.
 */
#define EVT_BLUE_GAP_PASS_KEY_REQUEST             (0x0402)
typedef __packed struct _evt_gap_pass_key_req{
  uint16_t conn_handle; /**< Connection handle for which the passkey has been requested. */
} PACKED evt_gap_pass_key_req;


/**
 * This event is generated by the Security manager to the application when the application
 * has set that authorization is required for reading/writing of attributes. This event will
 * be generated as soon as the pairing is complete. When this event is received,
 * aci_gap_authorization_response() command should be used by the application.
 */
#define EVT_BLUE_GAP_AUTHORIZATION_REQUEST        (0x0403)
typedef __packed struct _evt_gap_author_req{
  uint16_t conn_handle; /**< Connection handle for which authorization has been requested. */
} PACKED evt_gap_author_req;

/**
 * This event is generated when the slave security request is successfully sent to the master.
 */
#define EVT_BLUE_GAP_SLAVE_SECURITY_INITIATED     (0X0404)

/**
 * This event is generated when a pairing request is issued in response to a slave security
 * request from a master which has previously bonded with the slave. When this event is received,
 * the upper layer has to issue the command aci_gap_allow_rebond() in order to allow the slave
 * to continue the pairing process with the master.
 */
#define EVT_BLUE_GAP_BOND_LOST                    (0X0405)

/**
 * The event is given by the GAP layer to the upper layers when a device is discovered during scanning
 * as a consequence of one of the GAP procedures started by the upper layers. See @ref evt_gap_device_found.
 */
#define EVT_BLUE_GAP_DEVICE_FOUND                 (0x0406)
#ifndef BLUENRG_FW_6_4
typedef __packed struct _evt_gap_device_found{
  uint8_t peer_address_type; /**< Type of the peer address (@ref PUBLIC_ADDR, @ref RANDOM_ADDR). */
  uint8_t peer_address[6];   /**< Address of the peer device found during scanning. */
} PACKED evt_gap_device_found;
#else
typedef __packed struct _evt_gap_device_found{
#ifndef DOXYGEN_SHOULD_SKIP_THIS
  	uint8_t		evt_type;
	uint8_t		bdaddr_type;  /**< Type of the peer address (@ref PUBLIC_ADDR, @ref RANDOM_ADDR). */
	tBDAddr	    bdaddr;       /**< Address of the peer device found during scanning. */
	uint8_t		length;       /**< Lenght of advertising or scan response data. */
	uint8_t		data_RSSI[VARIABLE_SIZE]; /**< Advertising or scan response data + RSSI. RSSI is last octect (signed integer). */
} PACKED evt_gap_device_found;
#endif /* DOXYGEN_SHOULD_SKIP_THIS */
#endif

/**
 * This event is sent by the GAP to the upper layers when a procedure previously started has been terminated
 * by the upper layer or has completed for any other reason. See @ref evt_gap_procedure_complete.
 */
#define EVT_BLUE_GAP_PROCEDURE_COMPLETE           (0x0407)
typedef __packed struct _evt_gap_procedure_complete{
  uint8_t procedure_code; /**< Terminated procedure. See @ref gap_procedure_codes. */
  /**
   * @ref BLE_STATUS_SUCCESS, @ref BLE_STATUS_FAILED or @ref ERR_AUTH_FAILURE (procedure failed
   * due to authentication requirements).
   */
  uint8_t status;
  /**
   * Procedure specific data.\n
   * @li For Name Discovery Procedure:\n
   * the name of the peer device if the procedure completed successfully.
   * @li For General Connection Establishment Procedure:\n
   * The reconnection address written to the peripheral device if the peripheral is privacy enabled
   */
  uint8_t data[VARIABLE_SIZE];
} PACKED evt_gap_procedure_complete;

/**
 * This event is raised when the reconnection address is generated during the general connection
 * establishment procedure. The same address is set to the peer device also as a part of the general
 * connection establishment procedure. In order to make use of the reconnection address the next time
 * while connecting to the bonded peripheral, the application needs to use this reconnection address
 * as its own address as well as the peer address to which it wants to connect.
 */
#define EVT_BLUE_GAP_RECONNECTION_ADDRESS           (0x0408)
typedef __packed struct _evt_gap_reconnection_addr{
  uint8_t reconnection_address[6];
} PACKED evt_gap_reconnection_addr;

/**
 * @}
 */


#endif /* __BLUENRG_GAP_ACI_H__ */

