//
// TFT_4DGL is a class to drive 4D Systems TFT touch screens
//
// Copyright (C) <2010> Stephane ROCHON <stephane.rochon at free.fr>
//
// TFT_4DGL is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// TFT_4DGL is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with TFT_4DGL.  If not, see <http://www.gnu.org/licenses/>.

#include "mbed.h"
#include "LSM9DS0.h"
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string>
#include "TFT_4DGL.h"
#include <iostream>

// overwrite 4DGL library screen size settings in TFT_4DGL.h
#define SIZE_X  479
#define SIZE_Y  639
//

// SDO_XM and SDO_G are pulled up, so our addresses are:
#define LSM9DS0_XM_ADDR  0x1D // Would be 0x1E if SDO_XM is LOW
#define LSM9DS0_G_ADDR   0x6B // Would be 0x6A if SDO_G is LOW

#define REFRESH_TIME_MS 200

TFT_4DGL lcd(p13,p14,p11); // serial tx, serial rx, reset pin;
LSM9DS0 imu(p9, p10, LSM9DS0_G_ADDR, LSM9DS0_XM_ADDR);

class Brick { 
public:
    int x; 
    int y;
    int color;
    int width;
    int height;
    bool exist;
    
    Brick(){
        width = 10 * 2;
        height = 5 * 2;
        exist = true;
    }
    
    void set(int x, int y){
        this->x = x;
        this->y = y;    
    }
};

int main() {

    //Initial setup
    uint16_t status = imu.begin();
    lcd.baudrate(128000);
    // added - Set Display to 640 by 480 mode
    lcd.display_control(0x0c, 0x01);
    lcd.background_color(0);
    lcd.cls();

    int score = 0;
    Brick* wall[32];
    srand(time(0));
    for(int i = 0; i < 8; i ++){
        for( int j = 0; j < 4; j++){
            Brick * brick = new Brick();
            brick->set(5 + i * brick->width, 20 + j * brick->height);
            int c = 0;
            int num = rand() % 8 + 1;
            switch (num) {
                case 1:
                    c = 0xff0000;
                    break;
                case 2:
                    c = 0xffff00;
                    break;
                case 3:
                    c = 0x00FF00;
                    break;
                case 4:
                    c = 0x0000FF;
                    break;
                case 5:
                    c = 0xFFC0CB;
                    break;
                case 6:
                    c = 0xA52A2A;
                    break;
                case 7:
                    c = 0xd3d3d3;
                    break;
                case 8:
                    c = 0x800080;
                    break; 
                default:
                    c = 0x445566;
            }
            brick->color = c;
            wall[i*4 + j] = brick;
        }
    }
    
    float player_x = 60.0 * 2;
    float player_y = 120.0 * 2;
    float p_width = 30.0;
    
    float ball_x = 23.0 * 2;
    float ball_y = 49.0 * 2;
    float ball_vx = 15.0;
    float ball_vy = 15.0;
    float ball_r = 2.0;
    float width = 110.0 * 2;
    float height = 120.0 * 2;
    bool end = false;
    bool win = false;
    
    while (!end && !win)
    {
        lcd.cls();
        imu.readMag();
        imu.readAccel();
        
        for(int i = 0; i < 32; i++ ){
            if(wall[i]->exist){
                lcd.rectangle(wall[i]->x, wall[i]->y, wall[i]->x + wall[i]->width, wall[i]->y + wall[i]->height, wall[i]->color);        
            }
        }
        
        lcd.circle(ball_x, ball_y, ball_r, RED);
        lcd.line(player_x, player_y, player_x + p_width, player_y, BLUE);
        
        ball_x += ball_vx * 0.7;
        ball_y += ball_vy * 0.7;
        player_x = player_x - imu.ax_raw * 0.003;
        
        if(player_x + p_width > width){
            player_x = width - p_width;    
        }
        
        if(player_x < 0){
            player_x = 0;     
        }
        
        if(ball_x + ball_r > width){
            ball_x = width - ball_r;
            ball_vx *= -1;
        }
        
        if(ball_x - ball_r < 0){
            ball_x = ball_r;
            ball_vx *= -1;   
        }
        
        if(ball_y + ball_r > height){
            if(ball_x < player_x || ball_x > player_x + p_width){
                end = true;
            } else {
                ball_y = height - ball_r;
                ball_vy *= -1;
            }
        }
        
        if(ball_y - ball_r < 0){
            ball_y = ball_r;
            ball_vy *= -1;
        }
        
         for(int i = 0; i < 32; i++ ){
            if(wall[i]->exist){
                if (!(ball_x > wall[i]->x + wall[i]->width || 
                ball_x + ball_r < wall[i]->x ||
                ball_y > wall[i]->y + wall[i]->height ||
                ball_y + ball_r < wall[i]->y)){
                        wall[i]->exist = false;
                        score++;
                }    
            }
        }
        if(score == 32){
            win = true;    
        }
        wait_ms(REFRESH_TIME_MS);
    }
    lcd.cls();
    
    if(win){
        lcd.text_string("You Win!!!", 20, 60, FONT_8X8, WHITE);
    } else {
        lcd.text_string("Game Over...", 20, 60, FONT_8X8, WHITE);
    }

}