/*  BME280 BLE sensor -- bme280_wrapper.cpp
*
*   Licensed under the Apache License, Version 2.0 (the "License");
*   you may not use this file except in compliance with the License.
*   You may obtain a copy of the License at
*
*       http://www.apache.org/licenses/LICENSE-2.0
*
*   Unless required by applicable law or agreed to in writing, software
*   distributed under the License is distributed on an "AS IS" BASIS,
*   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*   See the License for the specific language governing permissions and
*   limitations under the License.
*
*   Copyright 2019 Fedor Chervyakov
*/

#include "mbed.h"
#include "bme280.h"
#include "bme280_wrapper.h"


I2C *sensor;


void user_delay_ms(uint32_t period)
{
    wait_ms(period);
}


int8_t user_i2c_read(uint8_t dev_id, uint8_t reg_addr, uint8_t *reg_data, uint16_t len)
{
    int8_t rslt = BME280_OK; /* Return 0 for Success, non-zero for failure */
    int addr8bit = dev_id << 1;
    
    if (0 == sensor->write(addr8bit, (char *) &reg_addr, 1)) /* Write register address */
    {
        if (0 != sensor->read( addr8bit, (char *) reg_data, (int) len)) /* Read data from registers */
        {
            rslt = BME280_E_COMM_FAIL;
        }
    }
    else
    {
        rslt = BME280_E_COMM_FAIL;
    }

    return rslt;
}


int8_t user_i2c_write(uint8_t dev_id, uint8_t reg_addr, uint8_t *reg_data, uint16_t len)
{
    int8_t rslt = BME280_OK; /* Return 0 for Success, non-zero for failure */

    int addr8bit = dev_id << 1;
    char *cmd;

    /* Allocate memory for the command that is to be written on the bus*/
    cmd = (char *) malloc(sizeof(char) * len * 2);
    
    int n;
    for (int i=0; i<len; i++) { /* Build write command */
        n = 2*i;
        cmd[n] = reg_addr+i; /* register address */
        cmd[n+1] = reg_data[i]; /* register data */
    }
    
    if (sensor->write(addr8bit, cmd, len*2)) /* Send command */
        rslt = BME280_E_COMM_FAIL;
    
    free(cmd); /* Free buffer */

    return rslt;
}


BME280::BME280(PinName sda, PinName scl) : _sda(sda), _scl(scl) {
    init();
}


int BME280::init() {
    int rslt = 0;
    uint8_t settings_sel;

    sensor = new I2C(_sda, _scl); /* Instantiate global I2C interface variable */

    _dev.dev_id = BME280_I2C_ADDR_PRIM;
    _dev.intf = BME280_I2C_INTF;
    _dev.write = user_i2c_write;
    _dev.read = user_i2c_read;
    _dev.delay_ms = user_delay_ms;
 
    bme280_init(&_dev);

    _dev.settings.osr_h = BME280_OVERSAMPLING_1X;
    _dev.settings.osr_p = BME280_OVERSAMPLING_16X;
    _dev.settings.osr_t = BME280_OVERSAMPLING_2X;
    _dev.settings.filter = BME280_FILTER_COEFF_2;
    
    settings_sel = BME280_OSR_PRESS_SEL | BME280_OSR_TEMP_SEL | BME280_OSR_HUM_SEL | BME280_FILTER_SEL;
    
    rslt = bme280_set_sensor_settings(settings_sel, &_dev);
    if (rslt != BME280_OK)
    {
        fprintf(stderr, "Failed to set sensor settings (code %+d).", rslt);
        return rslt;
    }
    return rslt;
}


int BME280::force_measurement() {
    int rslt = 0;

    rslt = bme280_set_sensor_mode(BME280_FORCED_MODE, &_dev);
    if (rslt != BME280_OK)
    {
        printf("Failed to set sensor mode (code %+d).\n", rslt);
        return rslt;
    }

    _dev.delay_ms(50);

    rslt = bme280_get_sensor_data(BME280_ALL, &comp_data, &_dev);
    if (rslt != BME280_OK)
    {
        printf("Failed to get sensor data (code %+d).\n", rslt);
        return rslt;
    }

    return rslt;
}
