#include "mbed.h"

/**
 * @file    main.cpp
 * @brief   main file.
 *
 * @addtogroup test
 * @author  fdufnews
 * @date    23.08.2014
 * @brief   Main module to test led, button and ticker interaction
 * @note    required LEDs on pin D9 and D11 in addition to that on the board
 * @{
 */

#define F_DELAY1 1.5
#define F_DELAY2 0.25

void offEnd(void);

Timeout redOn, redOff; // 2 timeout that call each other to make some sort of programmable oscillator
Ticker flipperPWM; // Ticker that ramp up and down an LED
InterruptIn button1(USER_BUTTON); // button that light an LED and modify timeouts period
DigitalOut led1(LED1); // The green LED onboard
DigitalOut redLED(D9); // a red LED on a breadboard
PwmOut orangeLED(D11); // an orange one on the breadboard


/** Function to toggle led1
 evry time the button is pushed this function is called
 and led1 state toggled*/
void toggleLed1(void){
    led1 = !led1;
}

/** Function called at end of onEnd timeout, sets new timeout defined by led1 state
 onEnd calls offEnd which calls onEnd making an oscillator*/
void onEnd(void){
    redLED=0; // led off
    float delay = led1?F_DELAY1:F_DELAY2; // sets new delay conditionned by led1 state
    redOff.attach(&offEnd,delay); // update delay
}

/** Function called at end of offEnd timeout, sets new timeout defined by led1 state
 offEnd calls onEnd which calls offEnd making an oscillator*/
void offEnd(void){
    redLED=1; // led on
    float delay = led1?F_DELAY1:F_DELAY2; // sets new delay conditionned by led1 state
    redOn.attach(&onEnd,delay); // update delay
}

/** Function to manage PWM for orange LED
 at each new tick rampPWM is called changing PWM value ramping up and down the LED*/
void rampPWM(void){
    static int rampDir=0;
    
    float curVal = orangeLED.read(); // read current programmed PWM value
    if (rampDir==0){ // if ramping up
        curVal+=0.05; // increment current PWM value
        orangeLED=curVal; // sets PWM
        if (curVal>=1.0){ // if limit reached
            rampDir=1; // toggle direction
        }
    }else if (rampDir==1){ // if ramping down
        curVal-=0.1; // decrement current PWM value
        orangeLED=curVal; // sets PWM
        if (curVal<=0.0){ // if limit reached
            rampDir=0; // toggle direction
        }
    }
}

/** main function
 sets system up by
   initializing LEDs state
   attaching interrupt
   launching ticker and timeout
   then enters an infinite while(1) loop
*/
int main(void){
    // sets LED default state
    led1=0;
    redLED=1;
    orangeLED.write(0.0);
    
    // attach functions called by ISR
    redOn.attach(&onEnd,F_DELAY1); // toggle redLED state
    flipperPWM.attach(&rampPWM,0.1); // change PWM value every 100ms
    button1.fall(&toggleLed1);
   while(1){}; // do nothing
}

/** @} */