/**
* @file MAX31850.cpp
* @brief library of MAX31850  1-Wire Thermocouple-to-Digital Converters,Cold-Junction Compensated  (http://www.maximintegrated.com/datasheet/index.mvp/id/7953)
* @author Frederic BLANC
* @date 2014_01_15
*/
#include "mbed.h"
#include "onewire.h"
#include "MAX31850.h"
#include "crc8.h"
#include "Horner_ITS90.h"
int MAX31850_Read_Scratch(uint8_t id[],uint8_t sp[],uint8_t n)
{
    if (id[0] == MAX31850_ID) {
        if (ow_reset(n))                 //reset OW
            return OW_SHORT_CIRCUIT;
        sp[0]=MAX31850_READ;   // command
        ow_command(n,sp[0], &id[0]);
        for ( int i=0 ; i< MAX31850_SP_SIZE; i++ ) { //read 8xdata + CRC16
            sp[i]=ow_byte_rd(n);
        }
        if(sp[8]!=crc8(sp,8))
            return OW_ERROR_CRC;
        return OW_OK;
    }
    return OW_ERROR_BAD_ID;
}

int MAX31850_Temp_TRUE(uint8_t *sp,double *temp)
{


    float temp_cj,temp_tc;
    double vtc;
    int err; 
    
    err = MAX31850_Temp_TC(sp,&temp_tc);
    if(err) {
        return err;
    }
    MAX31850_Temp_CJ(sp,&temp_cj);

    vtc = MAX31850_K * temp_tc ;
    *temp=0;
    if(temp_tc < 0.0) {
        for(int i=0;i<10;++i)
        {
        
        *temp+=ITS90_TCK_subzero[i]*pow(vtc,i);
        }
    } else if(temp_tc < 500.0) {
        for(int i=0;i<10;++i)
        {
        
        *temp+=ITS90_TCK_sub500[i]*pow(vtc,i);
        }
    } else {
        for(int i=0;i<10;++i)
        {
        
        *temp+=ITS90_TCK_sub1372[i]*pow(vtc,i);
        }
    }
    return OW_OK;
}


int MAX31850_Temp_TC(uint8_t *sp,float *temp)
{

    uint8_t *ptr_meas;
    uint16_t meas;

    if(sp[0]& 1)
        return MAX31850_ERR_FAULT;
    if(sp[2]& (1<<1))
        return MAX31850_ERR_OPEN_CIRCUIT;
    if(sp[2]& (1<<2))
        return MAX31850_ERR_SHORT2GND;
    if(sp[2]& (1<<3))
        return MAX31850_ERR_SHORT2VDD;
    ptr_meas=(uint8_t*) &meas;
    *ptr_meas = sp[0]  ;  // LSB
    *++ptr_meas = sp[1]; // MSB
    if ( meas & 0x8000 )  {

        meas ^= 0xffff;  // convert to positive => (twos complement)++
        meas++;
        *temp= -1.0 * MAX31850_ADC_TEMP_TC_UNIT * (float)(meas>>2);
    } else {
        *temp=  MAX31850_ADC_TEMP_TC_UNIT * (float)(meas>>2);
    }

    return OW_OK;
}

int MAX31850_Temp_CJ(uint8_t *sp,float *temp)
{
    uint8_t *ptr_meas;
    uint16_t meas;

    ptr_meas=(uint8_t*) &meas;
    *ptr_meas = sp[2]  ;  // LSB
    *++ptr_meas = sp[3]; // MSB
    if ( meas & 0x8000 )  {

        meas ^= 0xffff;  // convert to positive => (twos complement)++
        meas++;
        *temp= -1.0 * MAX31850_ADC_TEMP_CJ_UNIT * (float)(meas>>4);
    } else {
        *temp=  MAX31850_ADC_TEMP_CJ_UNIT * (float)(meas>>4);
    }
    return OW_OK;
}

int MAX31850_Start_meas(uint8_t id[],uint8_t n=0)
{
    if (id[0] == MAX31850_ID) {
        if (ow_reset(n))                 //reset OW
            return OW_SHORT_CIRCUIT;
        ow_command(n,MAX31850_CONVERT_T, &id[0]);
        return OW_OK;
    }
    return OW_ERROR_BAD_ID;
}