/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 * Copyright (c) 2014 LAAS-CNRS
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef ANALOGIN_DIFF_H
#define ANALOGIN_DIFF_H
#include "mbed.h"
#include "mbed_assert.h"
#include "analogin_api.h"
#include "platform.h"
#include "cmsis.h"
#include "pinmap.h"
#include "PeripheralNames.h"
#include "fsl_adc_hal.h"
#include "fsl_clock_manager.h"
#include "PeripheralPins.h"
#include "fsl_device_registers.h"

#define VERSION_ADC_DIFF "2014_12_05"

#if FSL_FEATURE_ADC_HAS_DIFF_MODE

#define MAX_FADC 6000000
#define ADC_DIFF(adc,ch) (((adc)<<1) | (ch)) 
//adc ch -> ADC Diff pin+ ADC<adc>_DP<ch> and pin- ADC<adc>_DM<ch>


/** class of AnalogIn_Diff for K64F
 * Example:
 * @code
 * #include "mbed.h"
 * #include "AnalogIn_Diff.h"
 * Ticker flipperADC;
 * bool flag_TX=false;
 * void flipADC()
 * {
 *     flag_TX=true;
 * }
 * int main()
 * {
 *   flipperADC.attach(&flipADC, 1.0);
 *   AnalogIn_Diff adc_diff(ADC_DIFF(0,1)); // ADC Diff pin+ ADC0_DP1 and pin- ADC0_DM1
 *   while (true) {
 *       if(flag_TX) {
 *           pc.printf("analog= %f \r\n",adc_diff.read()); //-0.5 < analog < 0.5 ; (-0.5 # ADC0_DP1=0.0V ADC0_DM1=3.3V) (0.0 # ADC0_DP1=1.65V ADC0_DM1=1.65V) (0.5 # ADC0_DP1=3.3V ADC0_DM1=0.0V)
 *           flag_TX=false;
 *       }
 *  }
 * }
 * @endcode
 */

class AnalogIn_Diff
{

public:
    /** Create an AnalogIn_Diff
     *
     * @param adc_ch is ADC_DIFF(#adc, #ch)
     *
     */
    AnalogIn_Diff(int adc_ch);

void test (int x);

    /** Destroy an AnalogIn_Diff
     */
    ~AnalogIn_Diff();

    /** Read the input voltage, represented as a float range [-0.5 ; 0.5]
     *
     * @returns A floating-point value representing the current input voltage, measured as a percentage
     */
    float read();

    /** Read the input voltage, represented as an 16-bit Signed 2's complement
     *
     * @returns
     *   16-bit signed representing the current input voltage, normalised to a 16-bit signed
     */
    int16_t read_raws16();

#ifdef MBED_OPERATORS
    /** An operator shorthand for read()
     *
     * The float() operator can be used as a shorthand for read() to simplify common code sequences
     *
     * Example:
     * @code
     * float x = volume.read();
     * float x = volume;
     *
     * if(volume.read() > 0.25) { ... }
     * if(volume > 0.25) { ... }
     * @endcode
     */
    operator float() {
        return read();
    }

#endif //MBED_OPERATORS

private:
    uint8_t chnNum;
    uint32_t instance ;
    uint32_t adc_addrs[];
};

#endif //FSL_FEATURE_ADC_HAS_DIFF_MODE
#endif //ANALOGIN_DIFF_H


