/**
 * @brief ADC128D818 12-Bit, 8-Channel, ADC System Monitor w/ Temp Sensor, Internal/External Reference, & I2C Interfac
 * http://www.ti.com/product/adc128d818/
 * @date 02/09/2013
 * @author F.BLANC LAAS-CNRS
 * http://homepages.laas.fr/fblanc/
 */

#ifndef ADC128D818_H

#define ADC128D818_H

#include "mbed.h"


//Library for the ADC128D818 12 BIT ADC.
enum ADC_MODE {
    ADC_MODE_0 = 0x00,
    ADC_MODE_1 = 0x01,
    ADC_MODE_2 = 0x02,
    ADC_MODE_3 = 0x03
};
enum ADC_ADDRESS {
    ADC_ADDRESS_LOW_LOW = 0x1D,
    ADC_ADDRESS_LOW_MID = 0x1E,
    ADC_ADDRESS_LOW_HIGH = 0x1F,
    ADC_ADDRESS_MID_LOW = 0x2D,
    ADC_ADDRESS_MID_MID = 0x2E,
    ADC_ADDRESS_MID_HIGH = 0x2F,
    ADC_ADDRESS_HIGH_LOW = 0x35,
    ADC_ADDRESS_HIGH_MID = 0x36,
    ADC_ADDRESS_HIGH_HIGH = 0x37
};
enum ADC_VREF {
    ADC_VREF_INT = 0x00,
    ADC_VREF_EXT = 0x01
};
enum ADC_RATE {
    ADC_RATE_LOW_POWER = 0x00,
    ADC_RATE_CONTINUOUS = 0x01
};
enum ADC_LIMIT {
    ADC_LIMIT_HIGH = 0x00,
    ADC_LIMIT_LOW = 0x01
};
enum ADC_CHANNEL {
    ADC_CHANNEL_IN0 = 0x00,
    ADC_CHANNEL_IN1 = 0x01,
    ADC_CHANNEL_IN2 = 0x02,
    ADC_CHANNEL_IN3 = 0x03,
    ADC_CHANNEL_IN4 = 0x04,
    ADC_CHANNEL_IN5 = 0x05,
    ADC_CHANNEL_IN6 = 0x06,
    ADC_CHANNEL_IN7 = 0x07,
    ADC_CHANNEL_TEMP = 0x07
};
enum ADC_INT {

    ADC_INT_IN0 = (char)~(0x01 <<0),
    ADC_INT_IN1 = (char)~(0x01 <<1),
    ADC_INT_IN2 = (char)~(0x01 <<2),
    ADC_INT_IN3 = (char)~(0x01 <<3),
    ADC_INT_IN4 = (char)~(0x01 <<4),
    ADC_INT_IN5 = (char)~(0x01 <<5),
    ADC_INT_IN6 = (char)~(0x01 <<6),
    ADC_INT_IN7 = (char)~(0x01 <<7),
    ADC_INT_TEMP = (char)~(0x01 <<7),
    ADC_INT_ALL = 0x00
};
enum ADC_ENABLE {

    ADC_ENABLE_IN0 = (char)~(0x01 <<0),
    ADC_ENABLE_IN1 = (char)~(0x01 <<1),
    ADC_ENABLE_IN2 = (char)~(0x01 <<2),
    ADC_ENABLE_IN3 = (char)~(0x01 <<3),
    ADC_ENABLE_IN4 = (char)~(0x01 <<4),
    ADC_ENABLE_IN5 = (char)~(0x01 <<5),
    ADC_ENABLE_IN6 = (char)~(0x01 <<6),
    ADC_ENABLE_IN7 = (char)~(0x01 <<7),
    ADC_ENABLE_TEMP = ~(0x01 <<7),
    ADC_ENABLE_ALL = 0x00
};
enum ADC_REG {
    ADC_REG_Configuration_Register  = 0x00,
    ADC_REG_Interrupt_Status_Register = 0x01,
    ADC_REG_Interrupt_Mask_Register = 0x03,
    ADC_REG_Conversion_Rate_Register = 0x07,
    ADC_REG_Channel_Disable_Register = 0x08,
    ADC_REG_One_Shot_Register = 0x09,
    ADC_REG_Deep_Shutdown_Register = 0x0A,
    ADC_REG_Advanced_Configuration_Register = 0x0B,
    ADC_REG_Busy_Status_Register = 0x0C,
    ADC_REG_Channel_Readings_Registers = 0x20,
    ADC_REG_Limit_Registers = 0x2A,
    ADC_REG_Manufacturer_ID_Register = 0x3E,
    ADC_REG_Revision_ID_Register = 0x3F
};
/** ADC128D818 class.
 *  Used for controlling a ADC128D818 adc connected via I2C.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "ADC128D818.h"
 * Serial pc(USBTX, USBRX);
 * ADC128D818 adc(p9,p10,p26);
 * AnalogOut dac(p18);
 * int main()
 * {
 * dac=0.5;
 *     printf("helloword_ADC128D818\r\n");
 *     wait_ms(100);
 *     int err=adc.init(ADC_ADDRESS_LOW_LOW, ADC_MODE_0, ADC_VREF_INT, ADC_RATE_CONTINUOUS, ADC_ENABLE_TEMP & ADC_ENABLE_IN1, ADC_INT_TEMP);
 *    switch (err) {
 *        case -1:
 *            printf("No Device\r\n");
 *            break;
 *        case -2:
 *             printf("Busy\r\n");
 *             break;
 *     }
 *     adc.init_limit(ADC_CHANNEL_IN0, 0x80, ADC_LIMIT_HIGH);
 *     adc.init_limit(ADC_CHANNEL_IN1, 0xA6, ADC_LIMIT_HIGH);
 *     adc.init_limit(ADC_CHANNEL_IN2, 0x80, ADC_LIMIT_HIGH);
 *     adc.init_limit(ADC_CHANNEL_IN3, 0x80, ADC_LIMIT_HIGH);
 *     adc.init_limit(ADC_CHANNEL_IN4, 0x80, ADC_LIMIT_HIGH);
 *    adc.init_limit(ADC_CHANNEL_IN5, 0x80, ADC_LIMIT_HIGH);
 *    adc.init_limit(ADC_CHANNEL_IN6, 0x80, ADC_LIMIT_HIGH);
 *    adc.init_limit(ADC_CHANNEL_TEMP, 30, ADC_LIMIT_HIGH);
 *    adc.init_limit(ADC_CHANNEL_TEMP, 29, ADC_LIMIT_LOW);
 *    char reg;
 *     reg=ADC_REG_Advanced_Configuration_Register;
 *     printf("reg %02X=%02X\r\n",reg,adc.read_register(reg));
 *     printf("reg %02X=%02X\r\n",0x2C,adc.read_channel(0x0C));
 *    adc.start();
 *    for(int i=0; i<1000; ++i) {
 *      wait(1);
 *      printf("ch0:%d : %d\r\n",i,adc.read_channel(ADC_CHANNEL_IN0));
 *       for( char c=1; c<7 ;++c)
 *       printf("ch%d:%d : %04X\r\n",c,i,adc.read_channel(c));
 *
 *       printf("temp:%d : %f\r\n",i,(adc.read_channel(ADC_CHANNEL_TEMP)>>7)/2.0);
 *        reg=0x01;
 *    printf("reg %02X=%02X\r\n",reg,adc.read_register(reg));
 *     }
 * }
 * @endcode
 */
class ADC128D818
{
protected:


    enum Configuration_Register {
        Configuration_Register_Start = 1<<0,
        Configuration_Register_INT_Enable = 1<<1,
        Configuration_Register_INT_Clear = 1<<3,
        Configuration_Register_Initialization = 1<<7
    };

    enum Busy_Status_Register {
        Busy_Status_Register_Busy = 1<<0,
        Busy_Status_Register_Not_Ready = 1<<1
    };

    enum Advanced_Configuration_Register {
        Advanced_Configuration_Register_External_Reference_Enable = 1<<0,
        Advanced_Configuration_Register_Mode_Select_0 = 1<<1,
        Advanced_Configuration_Register_Mode_Select_1 = 1<<2
    };

    enum Conversion_Rate_Register {
        Conversion_Rate_Register_Rate_Register = 1<<0
    };


private:
    I2C _i2c;
    char _data[2];
    char _address;
    char _mode;

public:
    /**
     * @brief Constructor.
     *
     * @param sda I2C
     * @param scl I2C
     * @param adc_int
     */
    ADC128D818(PinName sda, PinName scl, PinName adc_int );
    /**
    * @brief Destructor.
    */
    ~ADC128D818();

    InterruptIn _Adc_Int;
    /**
    * @brief read_channel
    * @param channel
    * @return u32_data
    * @date 02/09/2013
    */
    int read_channel(char channel);
    /**
    * @brief read_register
    * @param Register
    * @return u8_data
    * @date 02/09/2013
    */
    char read_register(char Register);
    /**
    * @brief init
    *
    * @param address I2C (7bits)
        ADC_ADDRESS_LOW_LOW
        ADC_ADDRESS_LOW_MID
        ADC_ADDRESS_LOW_HIGH
        ADC_ADDRESS_MID_LOW
        ADC_ADDRESS_MID_MID
        ADC_ADDRESS_MID_HIGH
        ADC_ADDRESS_HIGH_LOW
        ADC_ADDRESS_HIGH_MID
        ADC_ADDRESS_HIGH_HIGH
    * @param mode :
        ADC_MODE_0
        ADC_MODE_1
        ADC_MODE_2
        ADC_MODE_3
    * @param vref
        ADC_VREF_INT
        ADC_VREF_EXT
    * @param rate
        ADC_RATE_LOW_POWER
        ADC_RATE_CONTINUOUS
    * @param mask_channel
    * @param mask_int
    * @return error 0 OK, -1 NO DEVICE, -2 ADC is BUSY
    * @date 02/09/2013
    */
    int init(char address, char mode, char vref, char rate, char mask_channel, char mask_int);
    /**
    * @brief init_limit
    *
    * @param limit
    * @param high_low
        ADC_LIMIT_HIGH
        ADC_LIMIT_LOW
    * @return error 0 OK
    * @date 02/09/2013
    */
    int init_limit(char channel, char limit, char high_low);
    /**
    * @brief start
    * @date 02/09/2013
    */
    void start();
    /**
    * @brief stop
    * @date 02/09/2013
    */
    void stop();
};

#endif

