/**
 * @brief ADC128D818 12-Bit, 8-Channel, ADC System Monitor w/ Temp Sensor, Internal/External Reference, & I2C Interfac
 * http://www.ti.com/product/adc128d818/
 * @date 02/09/2013
 * @author F.BLANC LAAS-CNRS
 * http://homepages.laas.fr/fblanc/
 */
 
#include "ADC128D818.h"

/**
 * @brief Constructor.
 *
 * @param sda I2C
 * @param scl I2C
 * @param adc_int
 */
ADC128D818::ADC128D818(PinName sda, PinName scl, PinName adc_int) : _i2c(sda, scl), _Adc_Int (adc_int)
{
}

/**
 * @brief Destructor.
 */
ADC128D818::~ADC128D818()
{
}

/**
 * @brief init
 *
 * @param address I2C (7bits)
        ADC_ADDRESS_LOW_LOW
        ADC_ADDRESS_LOW_MID
        ADC_ADDRESS_LOW_HIGH
        ADC_ADDRESS_MID_LOW
        ADC_ADDRESS_MID_MID
        ADC_ADDRESS_MID_HIGH
        ADC_ADDRESS_HIGH_LOW
        ADC_ADDRESS_HIGH_MID
        ADC_ADDRESS_HIGH_HIGH
 * @param mode :
        ADC_MODE_0 
        ADC_MODE_1 
        ADC_MODE_2 
        ADC_MODE_3
 * @param vref
        ADC_VREF_INT
        ADC_VREF_EXT
 * @param rate
        ADC_RATE_LOW_POWER
        ADC_RATE_CONTINUOUS
 * @param mask_channel
 * @param mask_int
 * @return error 0 OK, -1 NO DEVICE, -2 ADC is BUSY
 * @date 02/09/2013
 */
int ADC128D818::init(char address, char mode, char vref, char rate, char mask_channel, char mask_int)
{

char data;
char cmd_data[2];

    _address=address << 1; //bug ?

    //2 test Busy_Status_Register
    cmd_data[0]= ADC_REG_Busy_Status_Register;

    if(_i2c.write(_address, cmd_data, 1))
        return  -1; //NO DEVICE
    _i2c.read(_address,&data,1); //read a byte

    if ((data & Busy_Status_Register_Not_Ready) == 1)
        return  -2; //ADC is BUSY
        
    ADC128D818::stop();
    //3 Program the Advanced Configuration Register
    data=0;
    switch (vref)
    {
        case ADC_VREF_INT:
            data&=~Advanced_Configuration_Register_External_Reference_Enable; //0
        break;
        case ADC_VREF_EXT:
            data|=Advanced_Configuration_Register_External_Reference_Enable; //1
        break;
    }
    switch (mode)
    {
        case ADC_MODE_0:
            data&=~Advanced_Configuration_Register_Mode_Select_0; //0
            data&=~Advanced_Configuration_Register_Mode_Select_1; //0
        break;
        case ADC_MODE_1:
            data&=~Advanced_Configuration_Register_Mode_Select_0; //0
            data|=Advanced_Configuration_Register_Mode_Select_1; //1
        break;
        case ADC_MODE_2:
            data|=~Advanced_Configuration_Register_Mode_Select_0; //1
            data&=Advanced_Configuration_Register_Mode_Select_1; //0
        break;
        case ADC_MODE_3:
            data|=~Advanced_Configuration_Register_Mode_Select_0; //1
            data|=Advanced_Configuration_Register_Mode_Select_1; //1
        break;
    }
   
    cmd_data[0]=ADC_REG_Advanced_Configuration_Register;
    cmd_data[1]=data;

   _i2c.write(_address, cmd_data, 2); //send a byte & wait acknowledged

    //4 Program the Conversion Rate Register
    data=0;
    switch (rate)
    {
        case ADC_RATE_LOW_POWER:
            data&=~Advanced_Configuration_Register_External_Reference_Enable; //0
        break;
        case ADC_RATE_CONTINUOUS:
            data|=Advanced_Configuration_Register_External_Reference_Enable; //1
        break;
    }
    
    cmd_data[0]=ADC_REG_Conversion_Rate_Register;
    cmd_data[1]=data;

    _i2c.write(_address, cmd_data, 2); //send a byte & wait acknowledged

    //5 Choose to enable or disable the channels using the Channel Disable Register

    cmd_data[0]=ADC_REG_Channel_Disable_Register;
    cmd_data[1]=mask_channel;

    _i2c.write(_address, cmd_data, 2); //send a byte & wait acknowledged

    //6 Using the Interrupt Mask Register

    cmd_data[0]=ADC_REG_Interrupt_Mask_Register;
    cmd_data[1]=mask_int;

    _i2c.write(_address, cmd_data, 2); //send a byte & wait acknowledged

return  0;
}

/**
 * @brief init_limit
 *
 * @param limit
 * @param high_low
        ADC_LIMIT_HIGH
        ADC_LIMIT_LOW
 * @return error 0 OK
 * @date 02/09/2013
 */
int ADC128D818::init_limit(char channel, char limit, char high_low)
{
    char cmd_data[2];

    cmd_data[0]=ADC_REG_Limit_Registers + channel * 2 + high_low;

    cmd_data[1]=limit;


    _i2c.write(_address, cmd_data, 2); //send a byte & wait acknowledged

   return  0;
}
/**
 * @brief read_channel
 * @param channel
 * @return u32_data
 * @date 02/09/2013
 */
int ADC128D818::read_channel(char channel)
{
    int u32_data=0;
    char data[2];
    char *ptr;
    char cmd[1] ;
    cmd[0] = ADC_REG_Channel_Readings_Registers + channel;
    _i2c.write(_address, cmd, 1); //send a byte & wait acknowledged
    
    _i2c.read(_address,data,2); //read a byte
    ptr=(char *) & u32_data;
    ptr[0]=data[1];
    ptr[1]=data[0];
   return  u32_data;
}

/**
 * @brief read_register
 * @param Register 
 * @return u8_data
 * @date 02/09/2013
 */

char ADC128D818::read_register(char Register)
{
    char cmd ;
    cmd = Register;
    
    _i2c.write(_address, &cmd, 1); //send a byte 
    _i2c.read(_address,&cmd,1); //read a byte
    
   return  cmd;
}
/**
 * @brief start
 * @date 02/09/2013
 */
void ADC128D818::start()
{
    char cmd_data[2];
    cmd_data[0]= ADC_REG_Configuration_Register;
    cmd_data[1]= Configuration_Register_Start | Configuration_Register_INT_Enable  ;

   _i2c.write(_address, cmd_data, 2); //send a 2 byte 

}
/**
 * @brief stop
 * @date 02/09/2013
 */
void ADC128D818::stop()
{
    char cmd_data[2];
    cmd_data[0]= ADC_REG_Configuration_Register;
    cmd_data[1]= 0 ;

   _i2c.write(_address, cmd_data, 2); //send a byte 

}