/**
 * DebouncedIn class version 1.0
 * Created by Andres Moya Bedoya, updated by Ben Faucher
 */

#include "mbed.h"

#ifndef _DEBOUNCED_IN_H_
#define _DEBOUNCED_IN_H_

/**
 * DebouncedIn object, uses software tickers to debounce mechanical inputs.
 */

    class DebouncedIn {
        public:      
             /** Create a DebouncedIn connected to the specified pin
              *
              *  @param in DigitalIn pin to connect to
              */
             DebouncedIn(PinName in);
             
             /** Same as before, with option to specify pin mode
              *
              *  @param in DigitalIn pin to connect to
              *  @param mode (optional) Set pull mode - PullUp, PullDown, PullNone, OpenDrain
              */
             DebouncedIn(PinName in, PinMode mode);
 
             /** Read the input state, represented as 0 or 1 (int)
              *
              *  @returns
              *    An integer representing the state of the input pin,
              *    0 for logical 0, 1 for logical 1. State changes when input 
              *    has been steady for at least 40ms (8 ticker cycles of 5ms).
              */
             int read (void);
             
             /** An operator shorthand for read()
              */
             operator int();
              
             /** Rising edge count (int)
              *
              *  @returns
              *    An integer representing the number of times the switch has
              *    changed from low to high. Count resets to zero when this
              *    function is called.
              */
             int rising(void);
             
              /** Falling edge count (int)
              *
              *  @returns
              *    An integer representing the number of times the switch has
              *    changed from high to low. Count resets to zero when this
              *    function is called.
              */
             int falling(void);
             
              /** Steady state tick count (int)
              *
              *  @returns
              *    An integer representing how many ticker cycles the input has been
              *    steady for. Ticker cycles every 5ms.
              */
             int steady(void);
              
        private :    
               // objects
               DigitalIn _in;    
               Ticker _ticker;
 
               // function to take a sample, and update flags
               void _sample(void);
 
               // counters and flags
               int _samples;
               int _output;
               int _output_last;
               int _rising_flag;
               int _falling_flag;
               int _state_counter;
 
    };
    
#endif