/*****************************************************/
/* MBED MULTI EFFECTOR                               */
/*                                                   */
/*                                                   */
/*****************************************************/

#include "mbed.h"
#include "EffectParam.h"
#include "TextLCD.h"
#include "R_Sw_Check.h"
#include "Distotion_Unit.h"
#include "Delay_Unit.h"

//Serial debug(USBTX, USBRX); // tx, rx

LocalFileSystem local("local");
Ticker      sampling;

AnalogIn    Ain(p17);                                   // Audio Input
AnalogOut   Aout(p18);                                  // Audio Output
TextLCD     lcd(p24, p25, p26, p27, p28, p29, p30);     // rs, rw, e, d0, d1, d2, d3)
DigitalIn   Rsw0A(p5);                                  // Rotary SW 0
DigitalIn   Rsw0B(p6);
DigitalIn   Rsw1A(p7);                                  // Rotary SW 1
DigitalIn   Rsw1B(p8);
DigitalIn   Rsw2A(p9);                                  // Rotary SW 2
DigitalIn   Rsw2B(p10);
DigitalIn   Rsw3A(p11);                                 // Rotary SW 3
DigitalIn   Rsw3B(p12);
DigitalOut  busyFlag(p13);                              // Effect Proc Busy Flag

/*******************************/
/* For Test Signal             */
/*******************************/
#define     TEST_SIGNAL_ENABLE      (0)                 // 1 : Internal SinWave for Debug
#define     TEST_SIGNAL_FREQ        (1000.0)            // Frequency [Hz]    
#define     TEST_SIGNAL_AMP         (30000.0)           // Amplitude
#define     PAI                     (3.14159)

/*******************************/
/* For ADC & DAC Setting       */
/*******************************/
#define     SAMPLING_TIME           (25.0)              // ADC Sampling Rate [us]
volatile unsigned int   *g_usiAd0cr, *g_usiAd0dr2;      // ADC Reg
unsigned int            *g_usiDacr;                     // DAC Reg

/*******************************/
/* Gloval Valinat              */
/*******************************/
unsigned int    g_usiFinalOut;                                              // DAC Out Final Data
int             g_ssBuff[10];                                               // Audio Data Buff
float           g_fTestWaveT;                                               // Test Sin Wave Phase
int             g_iBankNo = 0;                                              // Parameter Patch Bank No
int             g_iProgNo = 0;                                              // Parameter Patch Program No
int             g_iFlag = 0;                                                // Pacth Write Flag
int             g_iProgPatch[BANK_MAX_NUM][PROGNO_MAX_NUM][PP_MAX_NUM];     // Parameter Data


/*******************************/
/* Function                    */
/*******************************/
int  main(void);
void effectProcess(void);                       // Effect Process Function
void paramCheck(int*, int*);                    // Paramter Range Check 
void param1Change(int*, int*, int*, int*);      // Process for Changed Rotary SW1 
void param2Change(int*, int*, int*, int*);      // Process for Changed Rotary SW2
void param3Change(int*, int*, int*, int*);      // Process for Changed Rotary SW3
void param4Change(int*, int*, int*, int*);      // Process for Changed Rotary SW4

void progWrite(int*);                           // 
void distParamChange(int*, int*);               // Change Parameter for Ditotion
void delayParamChange(int*, int*);              // Change PArameter for Delay

void progPatchWite(int);                        // Paramter Write for Flash ROM 
void ProgramPatchChange(int, int);              // 

/*******************************/
/* Effect Process              */
/*******************************/
void effectProcess() {
    busyFlag = 1;
    // Line Out
    *g_usiDacr = g_usiFinalOut;
    // ADC Start
    *g_usiAd0cr = 0x01200204;

#if (TEST_SIGNAL_ENABLE == 1)                           // Test Signal Sin Wave
    g_ssBuff[0] = TEST_SIGNAL_AMP * sin(g_fTestWaveT);
    g_fTestWaveT = g_fTestWaveT + 2.0 * PAI * SAMPLING_TIME * TEST_SIGNAL_FREQ  / 1e6;
    if (g_fTestWaveT >= (2.0 * PAI))g_fTestWaveT = 0;
#endif

    //
    // Effect Func
    //

    g_ssBuff[1] = distotion(g_ssBuff[0]);               // Call Distotion Function
    g_ssBuff[2] = delay(g_ssBuff[1]);                   // CAll Delay Function

    //
    // Effect Func
    //

#if (TEST_SIGNAL_ENABLE == 0)
    while (1) {
        if ((*g_usiAd0dr2 & 0x80000000) != 0)break; // ADC Done ?
    }
    g_ssBuff[0] = (int)(*g_usiAd0dr2 & 0x0000FFF0) - 32768;
#endif

    g_usiFinalOut = 0x00010000 | (g_ssBuff[2] + 32768);

    busyFlag = 0;
}




/*******************************/
/* MAIN                        */
/*******************************/
int main() {

    int     i ,j ,k;
    int     iTemp1, iTemp2;
    int     iParamSwPol[4];
    int     iRsw1Num, iRsw2Num, iRsw3Num, iRsw4Num;
    char    str1[256], str2[256];

    iRsw1Num = iRsw2Num = iRsw3Num = iRsw4Num = 0;

    // LCD INIT
    lcd.cls();
    lcd.locate(0,0);
    lcd.printf("Multi Effector");
    lcd.locate(1,1);
    lcd.printf("Mode:");
    param1Change(&iRsw1Num, &iRsw2Num, &iRsw3Num, &iRsw4Num);

    // Program Patch File Open
    FILE *fp = fopen("/local/prm.txt", "r");
    if(!fp) {
        progPatchWite(1);       // No File
    }else{
        // Read Patch from File
        fgets(str1, 256, fp);
        fgets(str2, 256, fp);
        for(i=0;i<BANK_MAX_NUM;i++){
            for(j=0;j<PROGNO_MAX_NUM;j++){
                fscanf(fp, "%5d%5d", &iTemp1, &iTemp2);
                for(k=0;k<PP_MAX_NUM;k++){
                    fscanf(fp, "%5d", &g_iProgPatch[i][j][k]);
                }
            }
        }
        fclose(fp);
    }

    // ADC & DAC SETTING
    g_usiAd0cr     = (unsigned int*)0x40034000;
    g_usiAd0dr2    = (unsigned int*)0x40034018;
    g_usiDacr      = (unsigned int*)0x4008C000;

    // SAMPLING TIMER START
    sampling.attach_us(&effectProcess, SAMPLING_TIME);

    // Rotary SW State Check
    while (1) {
        switch (ucRotarySwPol(Rsw0A, Rsw0B, Rsw1A, Rsw1B, Rsw2A, Rsw2B, Rsw3A, Rsw3B, iParamSwPol)) {
            case 0:
                iRsw1Num += iParamSwPol[0];
                param1Change(&iRsw1Num, &iRsw2Num, &iRsw3Num, &iRsw4Num);
                break;
            case 1:
                iRsw2Num += iParamSwPol[1];
                param2Change(&iRsw1Num, &iRsw2Num, &iRsw3Num, &iRsw4Num);
                break;
            case 2:
                iRsw3Num += iParamSwPol[2];
                param3Change(&iRsw1Num, &iRsw2Num, &iRsw3Num, &iRsw4Num);
                break;
            case 3:
                iRsw4Num += iParamSwPol[3];
                param4Change(&iRsw1Num, &iRsw2Num, &iRsw3Num, &iRsw4Num);
                break;
            default:
                break;
        }
    }
}


/*******************************/
/* Param Check                 */
/*******************************/
void paramCheck(int* iParam, int iMaxValue) {

    if(*iParam >= iMaxValue)*iParam = 0;
    if(*iParam < 0)*iParam = iMaxValue - 1;

}


/*******************************/
/* Param1 Change               */
/*******************************/
void param1Change(int* iParam1, int* iParam2, int* iParam3, int* iParam4) {

    paramCheck(iParam1, MODE_MAX_NUM);

    lcd.locate(6,1);
    lcd.printf("              ");
    lcd.locate(0,2);
    lcd.printf("                    ");
    lcd.locate(0,3);
    lcd.printf("                    ");
    
    switch (*iParam1) {
        case MODE_PLAY:
            lcd.locate(6,1);
            lcd.printf("PLAY");
            lcd.locate(2,2);
            lcd.printf(" Bank PrgNo");
            *iParam2 = g_iBankNo;
            *iParam3 = g_iProgNo;
            break;
        case MODE_EDIT:
            lcd.locate(6,1);
            lcd.printf("EDIT");
            lcd.locate(2,2);
            lcd.printf("Effct Param Value");
            *iParam2 = *iParam3 = 0;
            break;
        case MODE_PROGWRITE:
            lcd.locate(6,1);
            lcd.printf("Prog Write");
            lcd.locate(2,2);
            lcd.printf(" Bank PrgNo Write");
            *iParam2 = g_iBankNo;
            *iParam3 = g_iProgNo;
            break;
        default:
            break;
    }
    
    param2Change(iParam1, iParam2, iParam3, iParam4);
    
}


/*******************************/
/* Param2 Change               */
/*******************************/
void param2Change(int* iParam1, int* iParam2, int* iParam3, int* iParam4) {

    lcd.locate(0,3);
    lcd.printf("        ");

    switch (*iParam1) {
        case MODE_PLAY:
            paramCheck(iParam2, BANK_MAX_NUM);
            g_iBankNo = *iParam2;
            *iParam3 = g_iProgNo;
            break;
        case MODE_EDIT:
            paramCheck(iParam2, EFFECT_MAX_NUM);
            *iParam3 = 0;
            break;
        case MODE_PROGWRITE:
            paramCheck(iParam2, BANK_MAX_NUM);
            g_iBankNo = *iParam2;
            *iParam3 = g_iProgNo;
            *iParam4 = 0;
            g_iFlag = 0;
        default:
            break;
    }

    if ((*iParam1 == MODE_PLAY) || (*iParam1 == MODE_PROGWRITE)) {
        lcd.locate(6,3);
        lcd.printf("%d",*iParam2);
    }

    if (*iParam1 == MODE_EDIT) {
        switch (*iParam2) {
            case EFFECT_DIST:
                lcd.locate(2,3);
                lcd.printf(" DIST");
                break;
            case EFFECT_DELAY:
                lcd.locate(2,3);
                lcd.printf("DELAY");
                break;
            default:
                break;
        }
    }
    
    param3Change(iParam1, iParam2, iParam3, iParam4);
    
}


/*******************************/
/* Param3 Change               */
/*******************************/
void param3Change(int* iParam1, int* iParam2, int* iParam3, int* iParam4) {

    lcd.locate(8,3);
    lcd.printf("     ");

    switch (*iParam1) {
        case MODE_PLAY:
            paramCheck(iParam3, PROGNO_MAX_NUM);
            g_iProgNo = *iParam3;
            break;
        case MODE_EDIT:
            switch (*iParam2) {
                case EFFECT_DIST:
                    paramCheck(iParam3, DIST_PARAM_MAX_NUM);
                    switch (*iParam3) {
                        case DIST_PARAM_BYPASS:     *iParam4 = g_iProgPatch[g_iBankNo][g_iProgNo][PP_DIST_PARAM_BYPASS];
                                                    break;
                        case DIST_PARAM_MODE:       *iParam4 = g_iProgPatch[g_iBankNo][g_iProgNo][PP_DIST_PARAM_MODE];
                                                    break;
                        case DIST_PARAM_INPUTGAIN:  *iParam4 = g_iProgPatch[g_iBankNo][g_iProgNo][PP_DIST_PARAM_INPUTGAIN];
                                                    break;
                        case DIST_PARAM_CLIPLEVEL:  *iParam4 = g_iProgPatch[g_iBankNo][g_iProgNo][PP_DIST_PARAM_CLIPLEVEL];
                                                    break;
                        case DIST_PARAM_OUTPUTGAIN: *iParam4 = g_iProgPatch[g_iBankNo][g_iProgNo][PP_DIST_PARAM_OUTPUTGAIN];
                                                    break;
                        default:                    break;   
                    }
                    distParamChange(iParam3, iParam4);
                    break;
                case EFFECT_DELAY:
                    paramCheck(iParam3, DLY_PARAM_MAX_NUM);
                    switch (*iParam3) {
                        case DLY_PARAM_BYPASS:      *iParam4 = g_iProgPatch[g_iBankNo][g_iProgNo][PP_DLY_PARAM_BYPASS];
                                                    break;
                        case DLY_PARAM_DELAYTIME:   *iParam4 = g_iProgPatch[g_iBankNo][g_iProgNo][PP_DLY_PARAM_DELAYTIME];
                                                    break;
                        case DLY_PARAM_FBGAIN:      *iParam4 = g_iProgPatch[g_iBankNo][g_iProgNo][PP_DLY_PARAM_FBGAIN];
                                                    break;
                        default:                    break;                    
                    }
                    delayParamChange(iParam3, iParam4);
                    break;
                default:
                    break;
            }
            break;
        case MODE_PROGWRITE:
            paramCheck(iParam3, PROGNO_MAX_NUM);
            g_iProgNo = *iParam3;
            *iParam4 = 0;
            g_iFlag = 0;
            progWrite(iParam4);
            break;
        default:
            break;
    }

    if ((*iParam1 == MODE_PLAY) || (*iParam1 == MODE_PROGWRITE)) {
        lcd.locate(12,3);
        lcd.printf("%d",*iParam3);
    }

}


/*******************************/
/* Param4 Change               */
/*******************************/
void param4Change(int* iParam1, int* iParam2, int* iParam3, int* iParam4) {

    lcd.locate(14,3);
    lcd.printf("      ");

    switch (*iParam1) {
        case MODE_EDIT:
            switch (*iParam2) {
                case EFFECT_DIST:
                    distParamChange(iParam3, iParam4);
                    break;
                case EFFECT_DELAY:
                    delayParamChange(iParam3, iParam4);
                    break;
                default:
                    break;
            }
            break;
        case MODE_PROGWRITE:
            progWrite(iParam4);
            break;
        default:
            break;
    }

}


/*******************************/
/* Distotion Param Change      */
/*******************************/
void distParamChange(int* iParam3, int* iParam4) {

    lcd.locate(8,3);
    lcd.printf("           ");
    lcd.locate(8,3);

    switch (*iParam3) {
        case DIST_PARAM_BYPASS:     
            paramCheck(iParam4, DIST_VALUE_BYPASS_MAX);
            g_iDistBypass = *iParam4;
            //g_iProgPatch[g_iBankNo][g_iProgNo][PP_DIST_PARAM_BYPASS] = *iParam4;
            if(g_iDistBypass == 0){
                lcd.printf("BYPSS   OFF");
            }else{
                lcd.printf("BYPSS    ON");
            }
            break;
        case DIST_PARAM_MODE:       
            paramCheck(iParam4, DIST_VALUE_MODE_MAX);
            g_iDistMode = *iParam4;
            //g_iProgPatch[g_iBankNo][g_iProgNo][PP_DIST_PARAM_MODE] = *iParam4;
            lcd.printf(" MODE    %d", *iParam4);
            break;
        case DIST_PARAM_INPUTGAIN: 
            paramCheck(iParam4, DIST_VALUE_INPUTGAIN_MAX);
            g_iDistInputGain = *iParam4;
            //g_iProgPatch[g_iBankNo][g_iProgNo][PP_DIST_PARAM_INPUTGAIN] = *iParam4;
            lcd.printf("INP.G   %d", *iParam4);
            break;
        case DIST_PARAM_CLIPLEVEL:  
            paramCheck(iParam4, DIST_VALUE_CLIPLEVEL_MAX);
            g_iDistClipLevel = *iParam4;
            //g_iProgPatch[g_iBankNo][g_iProgNo][PP_DIST_PARAM_CLIPLEVEL] = *iParam4;
            lcd.printf("CLP.L   %d", *iParam4);
            break;
        case DIST_PARAM_OUTPUTGAIN: 
            paramCheck(iParam4, DIST_VALUE_OUTPUTGAIN_MAX);
            g_iDistOutputGain = *iParam4;
            //g_iProgPatch[g_iBankNo][g_iProgNo][PP_DIST_PARAM_OUTPUTGAIN] = *iParam4;
            lcd.printf("OUT.G   %d", *iParam4);
            break;
        default:                    
            break;

    }
}

/*******************************/
/* Delay Param Change          */
/*******************************/
void delayParamChange(int* iParam3, int* iParam4) {

    lcd.locate(8,3);
    lcd.printf("           ");
    lcd.locate(8,3);

    switch (*iParam3) {
        case DLY_PARAM_BYPASS:      
            paramCheck(iParam4, DLY_VALUE_BYPASS_MAX);
            g_iDelayBypass = *iParam4;
            //g_iProgPatch[g_iBankNo][g_iProgNo][PP_DLY_PARAM_BYPASS] = *iParam4;
            if(g_iDelayBypass == 0){
                lcd.printf("BYPSS   OFF");
            }else{
                lcd.printf("BYPSS    ON");
            }
            break;
        case DLY_PARAM_DELAYTIME:   
            paramCheck(iParam4, DLY_VALUE_DELAYTIME_MAX);
            g_iDelayTime = *iParam4;
            //g_iProgPatch[g_iBankNo][g_iProgNo][PP_DLY_PARAM_DELAYTIME] = *iParam4;
            lcd.printf("DLY.T   %d", *iParam4);
            break;
        case DLY_PARAM_FBGAIN:      
            paramCheck(iParam4, DLY_VALUE_FBGAIN_MAX);
            g_iDelayFeedBackLevel = *iParam4;
            //g_iProgPatch[g_iBankNo][g_iProgNo][PP_DLY_PARAM_FBGAIN] = *iParam4;
            lcd.printf(" FB.G   %d", *iParam4);
            break;
        default:                    
            break;

    }
}

/*******************************/
/* Program Patch Write         */
/*******************************/
void progWrite(int* param){

    if(g_iFlag == 0){
        if((*param > 24) || (*param < -24)){
            lcd.locate(14,3);
            lcd.printf(" Done");
            ProgramPatchChange(g_iBankNo, g_iProgNo);
            progPatchWite(0);        
            g_iFlag = 1;
        }else{
            lcd.locate(14,3);
            lcd.printf("Unexe");
        }
    }else{
       lcd.locate(14,3);
       lcd.printf(" Done"); 
    }
}


/*******************************/
/* Program Patch Wite          */
/*******************************/
void progPatchWite(int iFlag){
   
    int i, j, k;
    
    FILE *fp = fopen("/local/prm.txt", "w");  
    if(!fp) {
        while(1){
            lcd.cls();
            wait(0.5);
            lcd.locate(1,1);
            lcd.printf("SYSTEM ERROR #2000");
            wait(0.5);
        }
    }
   
   fprintf(fp, " BANK PROG ---- DISTOTION ---  ----- DELAY ------\n");
   fprintf(fp, "            BYP MODE  ING C.Lv OUTG  BYP TIME   FB\n");
   for(i=0;i<BANK_MAX_NUM;i++){
       for(j=0;j<PROGNO_MAX_NUM;j++){
           if(iFlag == 1)ProgramPatchChange(i, j);           
           fprintf(fp, "%5d%5d", i, j);
           for(k=0;k<PP_MAX_NUM;k++){
               fprintf(fp, "%5d\0", g_iProgPatch[i][j][k]);
           }
           fprintf(fp, "\n");
       }
   }
    
   fclose(fp);
}


/*******************************/
/* Program Patch Chanege       */
/*******************************/
void ProgramPatchChange(int iBank, int iProg){

    // Distotion Param
    g_iProgPatch[iBank][iProg][PP_DIST_PARAM_BYPASS]        = g_iDistBypass;
    g_iProgPatch[iBank][iProg][PP_DIST_PARAM_MODE]          = g_iDistMode;
    g_iProgPatch[iBank][iProg][PP_DIST_PARAM_INPUTGAIN]     = g_iDistInputGain;
    g_iProgPatch[iBank][iProg][PP_DIST_PARAM_CLIPLEVEL]     = g_iDistClipLevel;
    g_iProgPatch[iBank][iProg][PP_DIST_PARAM_OUTPUTGAIN]    = g_iDistOutputGain;
    
    // Delay Param
    g_iProgPatch[iBank][iProg][PP_DLY_PARAM_BYPASS]         = g_iDelayBypass;
    g_iProgPatch[iBank][iProg][PP_DLY_PARAM_DELAYTIME]      = g_iDelayTime;
    g_iProgPatch[iBank][iProg][PP_DLY_PARAM_FBGAIN]         = g_iDelayFeedBackLevel;
    
}