/* mbed Microcontroller Library
 * Copyright (c) 2017 u-blox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "gnss_operations.h"

#define CHECK_TALKER(s) ((buffer[3] == s[0]) && (buffer[4] == s[1]) && (buffer[5] == s[2]))

// LEDs
DigitalOut ledRed(LED1, 1);
DigitalOut ledGreen(LED2, 1);
DigitalOut ledBlue(LED3, 1);

/* This example program for the u-blox C030 and C027 boards instantiates
 * the gnss interface and waits for time/position to be received from a satellite.
 * Progress may be monitored with a serial terminal running at 9600 baud.
 * The LED on the C030 board will turn green when this program is
 * operating correctly, pulse blue when a time reading has been received,
 * pulse white when GNSS position has been received or turn red if there is
 * a failure.
 * On the C027 and C030 boards the green/red (respectively) LED near the
 * GNSS module will flash as the module achieves a fix.
 */

GnssOperations gnss;
Serial host_serial(USBTX, USBRX);

bool print_hex_buffer(char *buffer, uint32_t length) {

	char log[256] = "";

	for (int i =0; i < length; i++) {
		sprintf(log, "%s%02X ", log, buffer[i]);
	}
	printf(log);

	return true;
}

void passThroughThreadHandler() {
	while (1) {

		gnss.send_to_gnss(host_serial.getc());

	}
}


int main()
{
    int gnssReturnCode;
    int length;
    char buffer[256];
    bool enable_pass_through = true;
    Thread passThroughThread;
    host_serial.baud(115200);
    
    printf ("Starting up...\n");
    if (gnss.init()) {
        printf ("Waiting for GNSS to receive something...\n");
        passThroughThread.start(callback(passThroughThreadHandler));
        gnss.enable_ubx_odo();
        gnss.enable_ubx_nav_odo();
        gnss.enable_ubx_nav_pvt();
        while (1) {
    		double lat = 0.0, lon = 0.0;

    		gnssReturnCode = gnss.getMessage(buffer, sizeof(buffer));

    		if (gnssReturnCode > 0) {
    			length = LENGTH(gnssReturnCode);

    			if (enable_pass_through){
    				for (int i=0; i<length; i++)
    					printf("%c", buffer[i]);
    			}

    			if ((PROTOCOL(gnssReturnCode) == GnssParser::NMEA) && (length > 6)) {

    				// Talker is $GA=Galileo $GB=Beidou $GL=Glonass $GN=Combined $GP=GNSS
    				if ((buffer[0] == '$') || buffer[1] == 'G') {
    				  if (CHECK_TALKER("GLL")) {
    						char ch;

    						if (gnss.getNmeaAngle(1, buffer, length, lat) &&
    							gnss.getNmeaAngle(3, buffer, length, lon) &&
    							gnss.getNmeaItem(6, buffer, length, ch) && (ch == 'A')) {
    						}
    					}
    				}
    				buffer[length] = '\0';

    				printf(buffer);
    			}
    			else if ((PROTOCOL(gnssReturnCode) == GnssParser::UBX) && (length > 6)) {
    				eUBX_MESSAGE ubx_message_type = gnss.get_ubx_message(buffer);

    				uint16_t payload_length = buffer[4] | (buffer[5] << 8);

    				//print_hex_buffer(buffer, payload_length + UBX_FRAME_SIZE);
    			}
    		}
         }
    } else {
        printf("Unable to initialise GNSS.\n");
    }

    ledRed = 0;
    ledGreen = 1;
    ledBlue = 1;
    printf("Should never get here.\n");
    MBED_ASSERT(false);
}

// End Of File
