/**
 * Copyright (c) 2017, Arm Limited and affiliates.
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include <stdio.h>

#include "lorawan/LoRaWANInterface.h"
#include "lorawan/system/lorawan_data_structures.h"
#include "events/EventQueue.h"

// Application helpers
#include "DummySensor.h"
#include "trace_helper.h"
#include "lora_radio_helper.h"

using namespace events;

// Max payload size can be LORAMAC_PHY_MAXPAYLOAD.
// This example only communicates with much shorter messages (<30 bytes).
// If longer messages are used, these buffers must be changed accordingly.
uint8_t tx_buffer[30];
uint8_t rx_buffer[30];

/*
 * Sets up an application dependent transmission timer in ms. Used only when Duty Cycling is off for testing
 */
#define TX_TIMER                        10000

/**
 * Maximum number of events for the event queue.
 * 10 is the safe number for the stack events, however, if application
 * also uses the queue for whatever purposes, this number should be increased.
 */
#define MAX_NUMBER_OF_EVENTS            10


// Maximum number of retries for CONFIRMED messages before giving up
#define LORAWAN_CONFIRMED_MSG_RETRY_COUNTER  3 
//DR_5=SF_7; DR_4=SF_8; DR_3=SF_9; DR_2=SF_10; DR_1=SF_11; DR_0=SF_12
#define LORAWAN_DATA_RATE                    DR_5 
// app port
#define LORAWAN_APP_PORT                     15
// tx message type 
#define LORAWAN_TX_MSG_TYPE                  MSG_UNCONFIRMED_FLAG
// number of channel 
#define LORAWAN_CHANNEL_NBR                  3
// timeout retry when channel is block in msec 
#define LORAWAN_CHANNEL_RETRY                3000



/**
 * Dummy pin for dummy sensor
 */
#define PC_9                            0

/**
 * Dummy sensor class object
 */
DS1820  ds1820(PC_9);

/**
* This event queue is the global event queue for both the
* application and stack. To conserve memory, the stack is designed to run
* in the same thread as the application and the application is responsible for
* providing an event queue to the stack that will be used for ISR deferment as
* well as application information event queuing.
*/
static EventQueue ev_queue(MAX_NUMBER_OF_EVENTS *EVENTS_EVENT_SIZE);

/**
 * Event handler.
 *
 * This will be passed to the LoRaWAN stack to queue events for the
 * application which in turn drive the application.
 */
static void lora_event_handler(lorawan_event_t event);

/**
 * Constructing Mbed LoRaWANInterface and passing it the radio object from lora_radio_helper.
 */
static LoRaWANInterface lorawan(radio);

/**
 * Application specific callbacks
 */
static lorawan_app_callbacks_t callbacks;


static void LoRa_PrintChannels() {
    /* print list of all channel frequencies */
    lorawan_channelplan_t channelPlan {};
    static loramac_channel_t channelbuf[10];
    
    channelPlan.channels = channelbuf;
    if (lorawan.get_channel_plan(channelPlan) == LORAWAN_STATUS_OK) {
        for (uint8_t i = 0; i < channelPlan.nb_channels; i++) {
            loramac_channel_t chan = channelPlan.channels[i];
            printf(" CHAN %d ID %d FREQ %lu RX1FREQ %lu Band %d DR min %d max %d\n",
                   (int) i, (int) chan.id, chan.ch_param.frequency,
                   chan.ch_param.rx1_frequency, (int) chan.ch_param.band,
                   (int) chan.ch_param.dr_range.fields.min,
                   (int) chan.ch_param.dr_range.fields.max);
        }
    } else {
        printf(" COULD NOT GET CHANNEL PLAN\n");
    }
}

/**
 * Entry point for application
 */
int main(void)
{
    static loramac_channel_t ttnChannels[] = {
        {0, {868100000, 0, {(DR_5 << 4) | DR_0}, 1}},
        {1, {868300000, 0, {(DR_5 << 4) | DR_0}, 1}},
        {2, {868500000, 0, {(DR_5 << 4) | DR_0}, 1}},
        {3, {867100000, 0, {(DR_5 << 4) | DR_0}, 0}},
        {4, {867300000, 0, {(DR_5 << 4) | DR_0}, 0}},
        {5, {867500000, 0, {(DR_5 << 4) | DR_0}, 0}},
        {6, {867700000, 0, {(DR_5 << 4) | DR_0}, 0}},
        {7, {867900000, 0, {(DR_5 << 4) | DR_0}, 0}}
};
    lorawan_channelplan_t channelPlan {};
    // setup tracing
    setup_trace();

    // stores the status of a call to LoRaWAN protocol
    lorawan_status_t retcode;
    printf("---------------------------- \n");
// LORAWAN: Initialize LoRaWAN stack
    if (lorawan.initialize(&ev_queue) != LORAWAN_STATUS_OK) {
        printf(" LoRa initialization failed! \n");
        return -1;
    }
    printf(" Mbed LoRaWANStack initialized \n");

// LORAWAN: prepare application callbacks
    callbacks.events = mbed::callback(lora_event_handler);
    lorawan.add_app_callbacks(&callbacks);

// LORAWAN: Set number of retries in case of CONFIRMED messages
    if (lorawan.set_confirmed_msg_retries(LORAWAN_CONFIRMED_MSG_RETRY_COUNTER)
            != LORAWAN_STATUS_OK) {
        printf(" Set_confirmed_msg_retries failed! \n");
        return -1;
    }
    printf(" CONFIRMED message retries : %d \n",
           LORAWAN_CONFIRMED_MSG_RETRY_COUNTER);
 
 // LORAWAN: settaggio canali   
    channelPlan.channels = (loramac_channel_t*) ttnChannels;
    channelPlan.nb_channels = LORAWAN_CHANNEL_NBR;
    if (lorawan.set_channel_plan(channelPlan) == LORAWAN_STATUS_OK) {
        printf(" [+] Setting TTN channels\n");
    } else {
        printf(" [-] Failed to set TTN channels! Debug return code.\n");
    }
    LoRa_PrintChannels();
      
// LORAWAN:  data rate
//   if (lorawan.enable_adaptive_datarate() != LORAWAN_STATUS_OK) {
//      printf("\r\n enable_adaptive_datarate failed! \r\n");
//     return -1;
// }
    
    // Enable adaptive data rate
    if (lorawan.disable_adaptive_datarate() != LORAWAN_STATUS_OK) {
        printf(" disable_adaptive_datarate failed! \r\n");
        return -1;
    }
    printf(" Adaptive data  rate (ADR) - disabled \r\n");
    lorawan.set_datarate(LORAWAN_DATA_RATE);

    retcode = lorawan.connect();

    if (retcode == LORAWAN_STATUS_OK ||
            retcode == LORAWAN_STATUS_CONNECT_IN_PROGRESS) {
    } else {
        printf(" Connection error, code = %d \n", retcode);
        return -1;
    }

    printf(" Connection - In Progress ...\n");

    // make your event queue dispatching events forever
    ev_queue.dispatch_forever();

    return 0;
}

/**
 * Sends a message to the Network Server
 */
static void send_message()
{
    uint16_t packet_len;
    int16_t retcode;
    float sensor_value;
 

    if (ds1820.begin()) {
        ds1820.startConversion();
        sensor_value = ds1820.read();
        printf("\n -------------------------\n");
        printf(" Dummy Sensor Value = %3.1f \n", sensor_value);
        ds1820.startConversion();
    } else {
        printf(" No sensor found \n");
        return;
    }
 
    packet_len = sprintf((char *) tx_buffer, " Dummy Sensor Value is %3.1f",
                         sensor_value);

    retcode = lorawan.send(LORAWAN_APP_PORT, tx_buffer, packet_len,
                           LORAWAN_TX_MSG_TYPE);

    if (retcode < 0) {
        retcode == LORAWAN_STATUS_WOULD_BLOCK ? printf(" send - WOULD BLOCK\r\n")
        : printf(" send() - Error code %d \n", retcode);

        if (retcode == LORAWAN_STATUS_WOULD_BLOCK) {
            //retry in 3 seconds
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                ev_queue.call_in(LORAWAN_CHANNEL_RETRY, send_message);
            }
        }
        return;
    }
    
    printf(" %d bytes scheduled for transmission \n", retcode);
    memset(tx_buffer, 0, sizeof(tx_buffer));
}

/**
 * Receive a message from the Network Server
 */
static void receive_message()
{
    uint8_t port;
    int flags;
    int16_t retcode = lorawan.receive(rx_buffer, sizeof(rx_buffer), port, flags);

    if (retcode < 0) {
        printf(" receive() - Error code %d \r\n", retcode);
        return;
    }

    printf(" RX Data on port %u (%d bytes): ", port, retcode);
    for (uint8_t i = 0; i < retcode; i++) {
        printf("%02x ", rx_buffer[i]);
    }
    printf("\n");
    
    memset(rx_buffer, 0, sizeof(rx_buffer));
}

/**
 * Event handler
 */
static void lora_event_handler(lorawan_event_t event)
{
    int16_t retcode;
    lorawan_tx_metadata additional_data; 
    int backoff_data;
    
    switch (event) {
        case CONNECTED:
            printf(" Connection - Successful \n");
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            } else {
                ev_queue.call_every(TX_TIMER, send_message);
            }

            break;
        case DISCONNECTED:
            ev_queue.break_dispatch();
            printf(" Disconnected Successfully \n");
            break;
        case TX_DONE:
            printf(" Message Sent to Network Server \n");
            
            retcode = lorawan.get_tx_metadata(additional_data);
            switch (retcode)
            {
                case LORAWAN_STATUS_NOT_INITIALIZED:
                  printf(" Lorawan stack not initialized\n");
                  break;
              
                case LORAWAN_STATUS_METADATA_NOT_AVAILABLE:
                  printf(" Metadata not available\n");
                  break;
              
                case LORAWAN_STATUS_OK :
                   printf(" TX Channel: %d \n",additional_data.channel);
                   printf(" TOA (msec): %d \n",additional_data.tx_toa);
                   printf(" Data rate: %u \n",additional_data.data_rate);
                   break;
            }
            
            retcode = lorawan.get_backoff_metadata(backoff_data);
            switch (retcode)
            {
                case LORAWAN_STATUS_NOT_INITIALIZED:
                  printf(" Lorawan stack not initialized\n");
                  break;
              
                case LORAWAN_STATUS_METADATA_NOT_AVAILABLE:
                  printf(" Backoff not available\n");
                  break;
              
                case LORAWAN_STATUS_OK :
                   printf(" Backoff: %d \n",backoff_data);
                   break;
            }           
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            }
            break;
        case TX_TIMEOUT:
        case TX_ERROR:
        case TX_CRYPTO_ERROR:
        case TX_SCHEDULING_ERROR:
            printf(" Transmission Error - EventCode = %d \r\n", event);
            // try again
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            }
            break;
        case RX_DONE:
            printf(" Received message from Network Server \r\n");
            receive_message();
            break;
        case RX_TIMEOUT:
        case RX_ERROR:
            printf(" Error in reception - Code = %d \r\n", event);
            break;
        case JOIN_FAILURE:
            printf(" OTAA Failed - Check Keys \r\n");
            break;
        case UPLINK_REQUIRED:
            printf(" Uplink required by NS \r\n");
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            }
            break;
        default: 
            MBED_ASSERT(" Unknown Event");
    }
}


// EOF
