/* NextionLCD Library v1.0
 * Copyright (c) 2018 Grant Phillips
 * grant.phillips@mandela.ac.za
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#ifndef NEXTIONLCD_H
#define NEXTIONLCD_H
 
#include "mbed.h"

#define BLACK   0
#define BLUE    31
#define BROWN   48192
#define GREEN   2016
#define YELLOW  65504
#define RED     63488
#define GRAY    33840
#define WHITE   65535


/** Class library for a Nextion graphics LCD to provide basic graphics and touch functions.
 * This library DOES NOT provide the HMI functionality that is included with Nextion LCDs.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "NextionLCD.h"
 *
 * NextionLCD lcd(PB_6, PB_7);     //Tx, Rx
 * 
 * int main() 
 * {
 *     char str[20];
 *     
 *     lcd.ClrScr(BLACK);  //clear the display and fill it with BLACK pixels
 *     
 *     //draw some shapes
 *     lcd.FillCircle(100,100,50,RED);
 *     lcd.DrawPixel(100,100,WHITE);
 *     lcd.DrawRectangle(50,50,100,100,BLUE);
 *     lcd.DrawLine(50,50,100,150,YELLOW);
 * 
 *     while(1) 
 *     {
 *         if (lcd.Touch())        //test if the lcd was touched
 *         {
 *             sprintf(str, "Touched: %d,%d", lcd.TouchX(), lcd.TouchY()); //print to string buffer
 *             lcd.DrawString(0,0,200,30,0,GREEN,GRAY,0,0,str);            //display string on LCD:
 *                                                                         // x,y          : 0,0
 *                                                                         // width, height: 200, 300
 *                                                                         // font         : 0
 *                                                                         // fontcolor    : GREEN
 *                                                                         // backcolor    : GRAY
 *                                                                         // x alignment  : 0 (left)
 *                                                                         // y alignment  : 0 (top)
 *         }
 *         else
 *         {
 *             lcd.DrawString(0,0,200,30,0,BLACK,BLACK,0,0,"");            //blank display area when not touched
 *         }
 *     }
 * }
 * @endcode
 */
 
class NextionLCD {
  public:
    /** Create a NextionLCD object for a graphics LCD connected to the specified pins. 
    * @param Tx USART TX pin used to connect to Nextion LCD's RX pin
    * @param Rx USART RX pin used to connect to Nextion LCD's TX pin
    */
    NextionLCD(PinName Tx, PinName Rx);
       
    /** Clears the lcd by filling it with the specified color pixels. 
    * @param Color Integer value (0 to 65535) to represent the color to fill the screen with - represented in 16-bit 565 color format
    */
    void ClrScr(uint16_t color);
        
    /** Draws a string on the lcd at the specified xy position. 
    * @param x x position.
    * @param y y position.
    * @param w Width of string area.
    * @param h Height of string area.
    * @param font Font ID to use (0:smallest - 7:largest).
    * @param fontcolor Color Integer value (0 to 65535) to represent the font color of the string - represented in 16-bit 565 color format.
    * @param backcolor Color Integer value (0 to 65535) to represent the background color of the string - represented in 16-bit 565 color format.
    * @param xcenter Horizontal alignment (0: left-aligned, 1: entered, 2: right-aligned).
    * @param ycenter Vertical alignment (0: upper-aligned, 1: entered, 2: lower-aligned).
    * @param str String content.
    */
    void DrawString(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint8_t font, uint16_t fontcolor, uint16_t backcolor, uint8_t xcenter, uint8_t ycenter, char *str);
    
    /** Draws a pixel on the lcd at the specified xy position. 
    * @param x x position
    * @param y y position
    * @param color Color of the pixel (0 to 65535) - represented in 16-bit 565 color format.
    */
    void DrawPixel(uint16_t x, uint16_t y, uint16_t color);
    
    /** Draws a line on the lcd from x1,y1 to x2,y2. 
    * @param x1 x coordinate starting position 
    * @param y1 y coordinate starting position
    * @param x2 x coordinate ending position
    * @param y2 y coordinate ending position
    * @param color Color of the line (0 to 65535) - represented in 16-bit 565 color format.
    */
    void DrawLine(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t color);
    
    /** Draws a rectangle on the lcd from x,y with width w and height h. 
    * @param x x coordinate of top-left corner position 
    * @param y y coordinate of top-left corner position 
    * @param w Width of the rectangle
    * @param h Height of the rectangle
    * @param color Color of the rectangle (0 to 65535) - represented in 16-bit 565 color format.
    */
    void DrawRectangle(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t color);
    
    /** Draws a filled rectangle on the lcd from x,y with width w and height h. 
    * @param x x coordinate of top-left corner position 
    * @param y y coordinate of top-left corner position 
    * @param w Width of the rectangle
    * @param h Height of the rectangle
    * @param color Color of the rectangle (0 to 65535) - represented in 16-bit 565 color format.
    */
    void FillRectangle(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t color);
    
    /** Draws a circle on the lcd at x,y with a radius of r.
    * @param x x coordinate position
    * @param y y coordinate position
    * @param r Radius of the circle
    * @param color Color of the circle (0 to 65535) - represented in 16-bit 565 color format.
    */
    void DrawCircle(uint16_t x, uint16_t y, uint16_t r, uint16_t color);
    
    /** Draws a filled circle on the lcd at x,y with a radius of r.
    * @param x x coordinate position
    * @param y y coordinate position
    * @param r Radius of the circle
    * @param color Color of the circle (0 to 65535) - represented in 16-bit 565 color format.
    */
    void FillCircle(uint16_t x, uint16_t y, uint16_t r, uint16_t color);

    /** Determines if the touchscreen is being touched or not. 
    *
    * @retval true LCD is being touched.
    * @retval false LCD is not being touched.
    */
    bool Touch(void);
    
    /** Get the X coordinate of the last touch on the touchscreen.
    *
    * @retval Integer indicating the X coordinate of the last touch.
    */
    int TouchX(void);
    
    /** Get the Y coordinate of the last touch on the touchscreen.
    *
    * @retval Integer indicating the X coordinate of the last touch.
    */
    int TouchY(void);
    
  private:
    RawSerial lcd;             //Serial object for connecting to Nextion LCD
    //Serial pc;
    bool mTouch;
    int mTouchX, mTouchY;
    char mRxMsg[40];
    int mRxIdx;
    void RxInterrupt(void); //Rx Interrupt
};
 
#endif