/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
 /*
 Print.cpp - Base class that provides print() and println()
 Copyright (c) 2008 David A. Mellis.  All right reserved.
 
 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.
 
 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.
 
 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 
 Modified 23 November 2006 by David A. Mellis
 */
 
#ifndef arduino_SERIAL_H
#define arduino_SERIAL_H

#include "mbed.h"
#include "platform.h"

#if DEVICE_SERIAL
 
#include "SerialBase.h"
#include "serial_api.h"

#define DEC 10
#define HEX 16
#define OCT 8
#define BIN 2
 
//namespace mbed {
 
/** A serial port (UART) for communication with other serial devices
 * This is a variation of the Serial class that doesn't use streams,
 * thus making it safe to use in interrupt handlers with the RTOS.
 *
 * Can be used for Full Duplex communication, or Simplex by specifying
 * one pin as NC (Not Connected)
 *
 * Example:
 * @code
 * // Send a char to the PC
 *
 * #include "mbed.h"
 *
 * RawSerial pc(USBTX, USBRX);
 *
 * int main() {
 *     pc.putc('A');
 * }
 * @endcode
 */
class ardSerial: public mbed::SerialBase {
 
public:
    /** Create a RawSerial port, connected to the specified transmit and receive pins
     *
     *  @param tx Transmit pin
     *  @param rx Receive pin
     *
     *  @note
     *    Either tx or rx may be specified as NC if unused
     */
    ardSerial(PinName tx, PinName rx);
 
    /** Write a char to the serial port
     *
     * @param c The char to write
     *
     * @returns The written char or -1 if an error occured
     */
    int putc(int c);
 
    /** Read a char from the serial port
     *
     * @returns The char read from the serial port
     */
    int getc();
 
    /** Write a string to the serial port
     *
     * @param str The string to write
     *
     * @returns 0 if the write succeeds, EOF for error
     */
    int puts(const char *str);

    //virtual int write(uint8_t) = 0;
    int write(uint8_t);
    int write(const char *str) {
      if (str == NULL) return 0;
      return write((const uint8_t *)str, strlen(str));
    }
    virtual int write(const uint8_t *buffer, size_t size);
    
    int print(const char[]);
    int print(char);
    int print(unsigned char, int = DEC);
    int print(int, int = DEC);
    int print(unsigned int, int = DEC);
    int print(long, int = DEC);
    int print(unsigned long, int = DEC);
    int print(double, int = 2);

    int printNumber(unsigned long, uint8_t);
    int printFloat(double, uint8_t);
};
 
//} // namespace mbed
 
#endif
 
#endif

