#ifndef LOG_H
#define LOG_H

#include "mbed.h"

#include "stdarg.h"
#include "stdio.h"

/**
  \file log.h
  \brief Logging utilities
  
  This file contains logging utilities
*/

#define LOG_BUFSIZE 4096

/// Enumeration used to enable/disable various serial lines in the logger
typedef enum {
  LOG_USB = 0,
  LOG_SER_9_10,
  LOG_SER_13_14,
  LOG_SER_28_27,
  LOG_MAX
} SerialLines;

/// Combination serial/file logger
class Log {
private:
  // Accomodate all 3 serial ports and USB
  Serial *m_serial[4];
  bool    m_enable[4];
  
  // File log
  LocalFileSystem local;

public:
  /// Constructor
  inline Log() : local("local")
  {
    // Write to file
    FILE *logfile = fopen("/local/NetTool.log", "w");
    fputs("NetTool - Welcome!", logfile);
    fclose(logfile);
  
    // Set up the serial classes
    m_serial[LOG_USB]       = new Serial(USBTX,USBRX);
    m_serial[LOG_SER_9_10]  = new Serial(p9,p10);
    m_serial[LOG_SER_13_14] = new Serial(p13,p14);
    m_serial[LOG_SER_28_27] = new Serial(p28,p27);
    
    // Disable logging to all of them by default
    for (int idx = 0; idx < LOG_MAX; ++idx)
    {
      m_enable[idx] = false;
    }
  }

  /// Enable logging to the given serial line
  inline void enable(SerialLines idx)
  {
    m_enable[idx] = true;
  }
  
  /// Disable logging to the given serial line
  inline void disable(SerialLines idx)
  {
    m_enable[idx] = false;
  }
  
  /// This can log messages up to 4095 characters and has printf semantics
  /// All log messages include an implicit \r\n at the end
  inline bool printf(char *format, ...)
  {
    static char buffer[LOG_BUFSIZE];
    static int count;
    static va_list va;
    
    va_start(va, format);
    count = vsnprintf(buffer, LOG_BUFSIZE, format, va);
    
    // Ensure that the log message fit
    if (count > LOG_BUFSIZE-1)
    {
      // Log an error message if it didn't
      Log::printf("Log message too long: %4d", count);
    }
    else
    {
      // Write all characters from the message
      puts(buffer);
      // Send the EOM (\r\n)
      puts("\r\n");
    }
    va_end(va);
    return true;
  } // printf
  
  /// Put the given string (no implicit \r\n) on all enabled serial lines and to the logfile
  bool puts(const char *str)
  {
    // Write to file
    FILE *logfile = fopen("/local/NetTool.log", "a");
    fputs(str, logfile);
    fclose(logfile);
    
    // Write all characters from the message
    while (*str)
    {
      // Write to all serial devices
      for (int s = 0; s < LOG_MAX; ++s)
      {
        // Only write if enabled
        if (m_enable[s])
          s[*m_serial].putc(*str++);
      }
    }
    return true;
  }

};

#endif