#ifndef SCANNER_H
#define SCANNER_H

#include "main.h"
#include "net/net.h"
#include "util/log.h"
#include "sniffer.h"

#include <cstring>
#include <set>

/**
  \file scanner.h
  \brief Basic TCP Port Scanner
  
  The class in this file facilitates a very fast (3 second scan, 7 second wait) TCP port scan
  of all ports from 1-65535, keeping and printing a list of all ports on which a TCP SYN/ACK
  is received (which should indicate that the port is open and waiting for connections).
*/

#define SCANNER_PADSIZE   0
#define SCANNER_FRAMESIZE (sizeof(Ethernet_FrameHeader) + sizeof(IP_PacketHeader) + sizeof(TCP_SegmentHeader) + SCANNER_PADSIZE)

/// Demo - TCP Port Scanner
class Scanner
{
private:
  Sniffer *sniffer;
  Ticker scan_timer;
  Timeout scan_timeout;
  
  u8 raw_frame[SCANNER_FRAMESIZE];
  
  IP_Address destip;
  u16 port_idx;
  u16 pseudo_checksum;
  
  set<u16> open_ports;
  
  LocalFileSystem local;

public:
  /// Constructor
  inline Scanner(Sniffer *_sniffer)
  : sniffer(_sniffer), local("local")
  {
    sniffer->attach_tcp(this, &Scanner::handle_tcp);
  }
  
  /// Handle incoming TCP packets
  inline void handle_tcp(TCP_SegmentHeader *packet, u32 data_bytes)
  {
    if (packet->syn && packet->ack)
    {
      open_ports.insert(packet->source_port);
    }
  }
  
  /// Handle the conclusion of the port scan (should not be called externally)
  inline void finish()
  {
    FILE *fp = fopen("/local/PortScan.txt", "w");
    fprintf(fp, "Open Ports on %d.%d.%d.%d:\n", destip.octet[0], destip.octet[1], destip.octet[2], destip.octet[3]);
    for (set<u16>::iterator iter = open_ports.begin(); iter != open_ports.end(); ++iter)
    {
      fprintf(fp, "  TCP:%-5d OPEN\n", *iter);
    }
    fclose(fp);  
    
    main_log.printf("Open ports:");
    for (set<u16>::iterator iter = open_ports.begin(); iter != open_ports.end(); ++iter)
    {
      main_log.printf("  TCP:%-5d OPEN", *iter);
    }
    main_log.printf("Port scan complete.");
  }
  
  /// Start a TCP port scan
  inline void start(Ethernet_MAC src, Ethernet_MAC dst, IP_Address srcip, IP_Address dstip)
  {
    // Create the ethernet frame, IP packet, and TCP segment memory mapping
    static Ethernet_FrameHeader *frame = (Ethernet_FrameHeader*)raw_frame;
    static IP_PacketHeader *packet = (IP_PacketHeader*)frame->payload;
    static TCP_SegmentHeader *segment = (TCP_SegmentHeader*)packet->data;
    
    destip = dstip;
    main_log.printf("Starting TCP port scan of %d.%d.%d.%d...", dstip.octet[0], dstip.octet[1], dstip.octet[2], dstip.octet[3]);
    
    // Zero the frame
    memset(raw_frame, '\0', SCANNER_FRAMESIZE);
    
    //sniffer->inject(dst, frame->ethertype, packet, 0); //sizeof(raw_frame)-sizeof(Ethernet_FrameHeader));

    // Fill in the frame (constant for all TCP connections)
    frame->destination = dst;
    frame->source = src;
    frame->ethertype = ETHERTYPE_IPV4;
    
    // Fill in the IP packet
    packet->source = srcip; // Can't change with destination back-to-back? lol
    packet->version = 0x04;
    packet->header_bytes_div4 = 5; // *4 = 20
    packet->packet_bytes = SCANNER_FRAMESIZE-sizeof(Ethernet_FrameHeader);
    packet->ttl = 64;
    packet->protocol = IPPROTO_TCP;
    packet->destination = dstip;
    
    // Fill in the TCP segment
    segment->sequence_number = 0xBADBEEF0;
    segment->data_offset_bytes_div4 = sizeof(TCP_SegmentHeader)/4;
    segment->syn = 1;
    segment->window_size = 8192;
    pseudo_checksum = pseudo_header_checksum(srcip, dstip, sizeof(TCP_SegmentHeader));
    
    // Initialize the scanning
    port_idx = 0;
    
    open_ports.clear();
    scan_timer.attach_us(this, &Scanner::scan, 50);
    //scan();
  }
  
  /// Incremental scan updates (should not be called externally)
  inline void scan()
  {
    // Common ports... currently unused
    static u16 ports[] = {1, 2, 3, 5, 7, 9, 11, 13, 17, 18, 19, 20, 21, 22, 23, 24, 25, 35, 37, 39,
      41, 42, 43, 47, 49, 50, 51, 52, 53, 54, 56, 58, 70, 79, 80, 83, 88, 90, 101, 102,
      104, 105, 107, 108, 109, 110, 111, 113, 113, 115, 117, 118, 119, 135, 137, 138, 139, 143, 152, 153,
      156, 162, 170, 177, 179, 194, 199, 201, 209, 210, 213, 218, 220, 259, 264, 308, 311, 318, 350, 351,
      366, 369, 371, 383, 384, 387, 389, 401, 427, 443, 444, 445, 464, 475, 497, 504, 512, 513, 514, 515,
      520, 524, 530, 532, 540, 542, 543, 544, 546, 547, 548, 554, 556, 563, 587, 591, 593, 604, 631, 635,
      636, 639, 641, 646, 647, 648, 653, 654, 657, 660, 674, 691, 692, 694, 695, 699, 700, 701, 702, 706,
      711, 712, 749, 750, 751, 752, 753, 754, 760, 860, 873, 902, 989, 990, 991, 992, 993, 995, 1058,
      1080, 1085, 1098, 1099, 1140, 1169, 1176, 1182, 1194, 1198, 1200, 1214, 1220, 1223, 1241, 1270, 1293, 1337, 1352, 1387,
      1414, 1417, 1418, 1419, 1420, 1431, 1433, 1470, 1494, 1512, 1513, 1521, 1524, 1533, 1547, 1677, 1720, 1723, 1755, 1761,
      1762, 1763, 1764, 1765, 1766, 1767, 1768, 1801, 1812, 1813, 1863, 1935, 1947, 1970, 1971, 1972, 1984, 1994, 1998, 2000,
      2031, 2053, 2073, 2074, 2082, 2083, 2086, 2102, 2103, 2104, 2105, 2144, 2145, 2161, 2181, 2210, 2211, 2212, 2219, 2220,
      2261, 2262, 2369, 2370, 2404, 2447, 2483, 2484, 2500, 2612, 2713, 2714, 2735, 2809, 2868, 2947, 2948, 2949, 3050, 3051,
      3074, 3225, 3233, 3235, 3260, 3268, 3269, 3283, 3305, 3306, 3386, 3389, 3396, 3412, 3455, 3423, 3424, 3478, 3483, 3516,
      3532, 3533, 3606, 3632, 3689, 3690, 3702, 3880, 3868, 3900, 3945, 3999, 4018, 4089, 4093, 4096, 4111, 4116, 4321, 4662,
      4728, 4840, 4843, 4847, 4993, 4894, 4899, 4950, 5000, 5001, 5003, 5004, 5005, 5051, 5060, 5061, 5070, 5084, 5085, 5099,
      5151, 5154, 5190, 5222, 5269, 5298, 5351, 5355, 5402, 5405, 5421, 5432, 5556, 5631, 5814, 5900, 5984, 5999, 6000, 6005,
      6086, 6110, 6111, 6112, 6129, 6346, 6347, 6350, 6432, 6444, 6445, 6619, 6665, 6666, 6667, 6668, 6669, 6888, 6969, 7005,
      7006, 7400, 7401, 7402, 7547, 7787, 7788, 8000, 8008, 8078, 8080, 8118, 8123, 8243, 8280, 8400, 8442, 8880, 8888, 9009,
      9080, 9100, 9105, 9119, 9306, 9312, 9418, 9535, 9536, 9800, 9898, 9996, 10008, 10010, 10050, 10051, 10113, 10114, 10115,
      10116, 13076, 13720, 13721, 13724, 13782, 13783, 13785, 13786, 15000, 15345, 17500, 18104, 19283, 19315, 22347, 22350,
      24465, 24554, 26000, 31457, 33434, 40000, 43047, 43048, 47808};
    
    // Create the ethernet frame, IP packet, and TCP segment memory mapping
    static Ethernet_FrameHeader *frame = (Ethernet_FrameHeader*)raw_frame;
    static IP_PacketHeader *packet = (IP_PacketHeader*)frame->payload;
    static TCP_SegmentHeader *segment = (TCP_SegmentHeader*)packet->data;
    
    segment->source_port = port_idx; //ports[port_idx];
    segment->destination_port = port_idx; //ports[port_idx];
    
    fix_endian_tcp(segment);
    segment->checksum = checksum(segment, sizeof(TCP_SegmentHeader), &segment->checksum, sizeof(segment->checksum), pseudo_checksum);
    
    fix_endian_ip(packet);
    packet->header_checksum = checksum(packet, sizeof(IP_PacketHeader), &packet->header_checksum, sizeof(packet->header_checksum));
    
    fix_endian_ethernet(frame);
    sniffer->inject(frame, SCANNER_FRAMESIZE);
    
    fix_endian_ethernet(frame);
    fix_endian_ip(packet);
    fix_endian_tcp(segment);
    
    // Update sequence number
    segment->sequence_number++;
    
    //if (port_idx >= sizeof(ports)/sizeof(u16)) scan_timer.detach();
    if (port_idx >= 65535)
    {
      scan_timer.detach();
      scan_timeout.attach(this, &Scanner::finish, 7);
    }
    port_idx++;
  }
};

#endif