/**
 @file net.h
 @brief This file encompasses all of the networking headers and includes them automatically
 
 This file has some utility functions and definitions used by all of the networking headers,
 and includes them all.  This is the only file necessary to include to use all of the networking
 facilities in nettool
*/

#ifndef NETWORK_H
#define NETWORK_H

#include "mbed.h"
#include "main.h"

#include "ctype.h"

/// General networking checksum - Used for IP, TCP, UDP, ICMP, etc.
/// Computes the one's complement of the one's complement sum of all of the given bytes except for the memory
/// at skip_byte for skip_count bytes (e.g. the checksum).  Optionally resumes computation from the (given) last checksum.
inline u16 checksum(void *_mem, unsigned int bytes, void *skip_byte = NULL, unsigned int skip_count = 0, u16 last = 0)
{
  u32 sum = 0;
  u16 *mem = (u16*)_mem;
  unsigned int skip_start = (u16*)skip_byte - mem;
  
  if (last)
    sum = last ^ 0xFFFF;
  
  //main_log.printf("CK:            0x%8X", sum);
  for (register unsigned int i = 0; i < bytes/sizeof(u16); ++i)
  {
    // Skip bytes we don't use (e.g. the checksum itself)
    if (i == skip_start)
      i += skip_count/sizeof(u16);
    
    // Sum them up
    sum += mem[i];
    
    //main_log.printf("CK: + 0x%04X = 0x%8X", mem[i], sum);
  }
    
  // One's complement of the one's complement sum
  sum += sum >> 16;
  sum &= 0xFFFF;
  sum ^= 0xFFFF;
  
  //main_log.printf("CK:          ~ 0x%8X", sum);
  
  return (u16)(sum);
}

/// Generic u16 endian swapping
inline void fix_endian_u16(u16 *p)
{
  char *bytes = (char*)p;
  bytes[0] ^= bytes[1];
  bytes[1] ^= bytes[0];
  bytes[0] ^= bytes[1];
}

/// Generic u32 endian swapping
inline void fix_endian_u32(u32 *p)
{
  char *bytes = (char*)p;
  // Swap outer bytes
  bytes[0] ^= bytes[3];
  bytes[3] ^= bytes[0];
  bytes[0] ^= bytes[3];
  // Swap inner bytes
  bytes[1] ^= bytes[2];
  bytes[2] ^= bytes[1];
  bytes[1] ^= bytes[2];
}

/* Printing character */
#define PCHAR(x) (isprint(x)?(x):'.')

/// Hex dump a word-aligned number of bytes (will print extra bytes if length is not a multiple of 32 bits)
inline void hex_dump(void *base, unsigned int length)
{
  char line[/*indent*/ 2 + /*0x*/ 2 + /*addr*/ 8 + /* : */ 3 + /*4xXXXX */ 4*5 + /*: */ 2 + /*8x.*/ 8 + /*\0*/ 1];
  for (char *p = (char*)base; p - (char*)base < length; p += 8)
  {
    sprintf(line, "  0x%08X : %02X%02X %02X%02X %02X%02X %02X%02X : %c%c%c%c%c%c%c%c", p,
                  p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7],
                  PCHAR(p[0]), PCHAR(p[1]), PCHAR(p[2]), PCHAR(p[3]), PCHAR(p[4]), PCHAR(p[5]), PCHAR(p[6]), PCHAR(p[7]));
    main_log.printf(line);
  }
}

// Ethernet
#include "ethernet.h"

// ARP and IP
#include "arp.h"
#include "ip.h"

// TCP, UDP, and ICMP
#include "tcp.h"
#include "udp.h"
#include "icmp.h"

#endif // NETWORK_H