/**
 @file net.h
 @brief This file encompasses all of the networking headers and includes them automatically
 
 This file has some utility functions and definitions used by all of the networking headers,
 and includes them all.  This is the only file necessary to include to use all of the networking
 facilities in nettool
*/

#ifndef NETWORK_H
#define NETWORK_H

#include "mbed.h"

#include "ctype.h"

/// General networking checksum - Used for IP, TCP, UDP, ICMP, etc.
/// Computes the one's complement of the one's complement sum of all of the given bytes except for the memory
/// at skip_byte for skip_count bytes (e.g. the checksum).  Optionally resumes computation from the (given) last checksum.
inline u16 checksum(void *_mem, unsigned int bytes, void *skip_byte = NULL, unsigned int skip_count = 0, u16 last = 0)
{
  u32 sum = 0;
  u16 *mem = (u16*)_mem;
  unsigned int skip_start = (u16*)skip_byte - mem;
  
  if (last)
    sum = last ^ 0xFFFF;
  
  //main_log.printf("CK:            0x%8X", sum);
  for (register unsigned int i = 0; i < bytes/sizeof(u16); ++i)
  {
    // Skip bytes we don't use (e.g. the checksum itself)
    if (i == skip_start)
      i += skip_count/sizeof(u16);
    
    // Sum them up
    sum += mem[i];
    
    //main_log.printf("CK: + 0x%04X = 0x%8X", mem[i], sum);
  }
    
  // One's complement of the one's complement sum
  sum += sum >> 16;
  sum &= 0xFFFF;
  sum ^= 0xFFFF;
  
  //main_log.printf("CK:          ~ 0x%8X", sum);
  
  return (u16)(sum);
}

/// Generic u16 endian swapping
inline void fix_endian_u16(u16 *p)
{
  char *bytes = (char*)p;
  bytes[0] ^= bytes[1];
  bytes[1] ^= bytes[0];
  bytes[0] ^= bytes[1];
}

/// Generic u32 endian swapping
inline void fix_endian_u32(u32 *p)
{
  char *bytes = (char*)p;
  // Swap outer bytes
  bytes[0] ^= bytes[3];
  bytes[3] ^= bytes[0];
  bytes[0] ^= bytes[3];
  // Swap inner bytes
  bytes[1] ^= bytes[2];
  bytes[2] ^= bytes[1];
  bytes[1] ^= bytes[2];
}

/* Printing character */
#define PCHAR(x) (isprint(x)?(x):'.')


// Ethernet
#include "ethernet.h"

// ARP and IP
#include "ip.h"

// TCP, UDP, and ICMP
#include "udp.h"

#endif // NETWORK_H