#include "mbed.h"
#include "iHvZ.hpp"
#include "XBee.hpp"

#include <vector>
#include <string>

Tag::Tag(iHvZ *game, PinName tagbtn)
: m_game(game), m_tag(tagbtn), m_tag_int(tagbtn)
{
    m_tag_int.rise(this, &Tag::rise);
    m_game->xbee_device().attach(this, &Tag::rx);
}

void Tag::rise()
{
    Serial usb(USBTX,USBRX);
    
    // debounce
    wait(.1);
    if (!m_tag) return;
    
    event(TAG_EVENT_BUTTON);
}

void Tag::rx()
{
    event(TAG_EVENT_RX);
}

void Tag::timeout()
{
    event(TAG_EVENT_TIMEOUT);
}

void Tag::event(TagEvent ev)
{
    Serial usb(USBTX,USBRX);
    static TagState state = TAG_STATE_WAITING;
    static string peer = "";
    static string peer_tagid = "";
    static string type = ""; // = (m_game.status()==STATUS_HUMAN)?"STUN":"TAG";
    static string ack = "";
    
    XBeePacket pkt;
    
    if (ev == TAG_EVENT_TIMEOUT)
    {
        // Display the timeout character (TODO)
        usb.printf("!! TIMEOUT !!\r\n");
        m_game->alphanumdisplay_device().display('*');
        //wait(1);
        state = TAG_STATE_WAITING;
    }
    
    if (ev == TAG_EVENT_RX)
    {
        // Get the packet that triggered this event
        pkt = m_game->xbee_device().read();
    }

    do switch (state)
    {
        case TAG_STATE_WAITING:
            // Zero everything
            peer = "";
            peer_tagid = "";
            type = "";
            ack = "";
            // Display H/Z
            // m_game->alphanumdisplay_device().display((m_game->status()==STATUS_HUMAN)?'H':'Z');
            // Check transitions
            if (ev == TAG_EVENT_BUTTON)     state = TAG_STATE_TX_SEND;
            else if (ev == TAG_EVENT_RX)    state = TAG_STATE_RX_RECV;
            else                            return;
            continue;
        // Sending states
        case TAG_STATE_TX_SEND:
            // First, check if this zombie is stunned (if so, they can't do anything)
            if (m_game->stun_tracker().stunned()) return;
            // Get whether we are tagging or stunning
            type = (m_game->status()==STATUS_HUMAN)?"STUN":"TAG";
            // Generate a random acknowledge character for display
            do { ack = string(1, 'A'+rand()%26); } while (ack == "H" || ack == "Z");
            // Send the SYN message
            m_game->xbee_device().broadcast(type + "\001SYN\001" + ack);
            // Display the acknowledge character (TODO)
            usb.printf("CHECK: %s (send)\r\n", ack.c_str());
             m_game->alphanumdisplay_device().display(ack[0]);
            // Attach a timeout that will revert to waiting
            m_timeout.attach(this, &Tag::timeout, TAG_TIMEOUT_TIME);
            state = TAG_STATE_TX_WAITACK;
            continue;
        case TAG_STATE_TX_WAITACK:
            if (ev == TAG_EVENT_RX)
            {
                // Make sure it is an ack packet with the right ack character
                vector<string> pieces = pkt.split_data();
                if (pieces.size() != 5) return;
                if (pieces[0] != type) return;
                if (pieces[1] != "ACK") return;
                if (pieces[2] != ack) return;
                ack = pieces[3];
                // Store the peer ID and tag ID
                peer = pkt.source();
                peer_tagid = pieces[4];
                // Display the new character for verification (TODO)
                usb.printf("CHECK: %s (send ack)\r\n", ack.c_str());
                m_game->alphanumdisplay_device().display(ack[0]);
                // Attach a timeout
                m_timeout.attach(this, &Tag::timeout, TAG_TIMEOUT_TIME);
                state = TAG_STATE_TX_WAITBTN;
            }
            return;
        case TAG_STATE_TX_WAITBTN:
            if (ev == TAG_EVENT_BUTTON)
            {
                state = TAG_STATE_TX_SENDCOMP;
                continue;
            }
            return;
        case TAG_STATE_TX_SENDCOMP:
            // Send the FIN message indicating that the transaction is complete
            m_game->xbee_device().send(peer, type+"\001FIN\001"+ack);
            // Display the completion character (TODO)
            usb.printf("COMPLETE: %s\r\n", type.c_str());
            m_game->alphanumdisplay_device().display('^');
            state = TAG_STATE_WAITING;
            m_timeout.detach();
            
            // Register the successful tag or stun
            if (type == "TAG" && m_game->status() == STATUS_ZOMBIE)
                m_game->register_tag(peer_tagid);
            if (type == "STUN" && m_game->status() == STATUS_HUMAN)
                m_game->register_stun(peer_tagid);
            return;
        // Receiving states
        case TAG_STATE_RX_RECV:
            if (ev == TAG_EVENT_RX)
            {
                vector<string> pieces = pkt.split_data();
                if (pieces.size() != 3) return;
                type = pieces[0];
                if (pieces[1] != "SYN") return;
                ack = pieces[2];
                // Register the source
                peer = pkt.source();
                // Display the verification character (TODO)
                usb.printf("CHECK: %s (receive)\r\n", ack.c_str());
                m_game->alphanumdisplay_device().display(ack[0]);
                // Attach a timeout
                m_timeout.attach(this, &Tag::timeout, TAG_TIMEOUT_TIME);
                state = TAG_STATE_RX_WAITBTN;
            }
            return;
        case TAG_STATE_RX_WAITBTN:
            if (ev == TAG_EVENT_BUTTON)
            {
                state = TAG_STATE_RX_SENDACK;
                continue;
            }
            return;
        case TAG_STATE_RX_SENDACK:
            {
                // Generate a new acknowledgment character
                string oldack = ack;
                ack = string(1, '0'+rand()%9);
                // Send the ack packet
                m_game->xbee_device().send(peer, type+"\001ACK\001"+oldack+"\001"+ack+"\001"+m_game->life());
                // Display the verification character (TODO)
                usb.printf("CHECK: %s (receive ack)\r\n", ack.c_str());
                m_game->alphanumdisplay_device().display(ack[0]);
                // Attach a timeout
                m_timeout.attach(this, &Tag::timeout, TAG_TIMEOUT_TIME);
                state = TAG_STATE_RX_WAITCOMP;
            }
            return;
        case TAG_STATE_RX_WAITCOMP:
            if (ev == TAG_EVENT_RX)
            {
                vector<string> pieces = pkt.split_data();
                // Make sure the completion packet is well-formed
                if (pieces.size() != 3) return;
                if (pieces[0] != type) return;
                if (pieces[1] != "FIN") return;
                if (pieces[2] != ack) return;
                // Make sure the peer is the same
                if (peer != pkt.source()) return;
                // Display the completion character (TODO)
                usb.printf("COMPLETE: %s\r\n", type.c_str());
                m_game->alphanumdisplay_device().display('^');
                m_timeout.detach();
                state = TAG_STATE_WAITING;
                
                // Stun or tag the player
                if (type == "TAG" && m_game->status() == STATUS_HUMAN)
                    m_game->tagged();
                if (type == "STUN" && m_game->status() == STATUS_ZOMBIE)
                    m_game->stunned();
            }
            return;
    } while (1);
}